/*
 * DocumentHeader.java
 *
 * Copyright 2001 by BRiSK Software,
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package makedocjr;

import java.io.*;

/**
 * <code>DocumentHeader</code> class is used to read/write PilotDoc header records
 * NOTE: Information about the various fields was taken from
 * <a href="http://www.concentric.net/~n9mtb/cq/doc/format.html">CQ Codeworks</a>
 *
 * @author Jeffrey A. Krzysztow
 * @author Pat Beirne
 * @version 1.0
 */
public class DocumentHeader {
	/**
	 * constant to indicate PilotDoc is uncompressed
	 */
	public final static short UNCOMPRESSED = 1;

	/**
	 * constant to indicate PilotDoc is compressed
	 */
	public final static short COMPRESSED = 2;

	/**
	 * size of text record
	 */
	public final static short textRecordSize = 4096;

	/**
	 * version of PilotDoc
	 */
	public short version = 0;					// 2 Word	1=plain text, 2=compressed

	/**
	 * spare (unknown)
	 */
	public short spare = 0;						// 2 Word

	/**
	 * length of uncompressed PilotDoc
	 */
	public int storyLen = 0;					// 4 DWord	in bytes, when decompressed

	/**
	 * number of PilotDoc records
	 */
	public short numRecords = 0;				// 2 Word	text records

	/**
	 * uncompressed text record size
	 */
	public short recordSize = textRecordSize;	// 2 Word	usually 0x1000

	/**
	 * currently viewed position in the document
	 * not used by all readers
	 */
	public int position = 0;					// 4 DWord
												// 16 bytes for Document Header

	/**
	 * The size of the DocumentHeader in bytes
	 * @return the number of bytes in the DocumentHeader
	 */
	public static int getSize() {
		return(16);
	}

	/**
	 * Reads the DocumentHeader from the DataInput
	 * @param the DataInput to read from
	 */
	public void read(DataInput di) throws IOException {
		version = di.readShort();
		spare = di.readShort();
		storyLen = di.readInt();
		numRecords = di.readShort();
		recordSize = di.readShort();
		position = di.readInt();
	}

	/**
	 * Writes the DocumentHeader to the DataInput
	 * @param the DataInput to write to
	 */
	public void write(DataOutput out) throws IOException {
		out.writeShort(version);
		out.writeShort(spare);
		out.writeInt(storyLen);
		out.writeShort(numRecords);
		out.writeShort(recordSize);
		out.writeInt(position);
	}

	/**
	 * indicates whether the document is compressed or not
	 */
	public boolean isCompressed() {
		return version == COMPRESSED;
	}

	/**
	 * override Object.toString()
	 */
	public String toString() {
		return ">> DocumentHeader <<"
			+ "\nversion = " + version
			+ "\nspare = " + spare
			+ "\nstoryLen = " + storyLen
			+ "\nnumRecords = " + numRecords
			+ "\nRecSize = " + recordSize
			+ "\nposition = " + position;
	}

}

