/*
 * Bookmark.java
 *
 * Copyright 2001 Steve Held
 * Copyright 2001 by BRiSK Software,
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package makedocjr;

import java.io.*;
import javax.swing.text.Position;

/**
 * <code>Bookmark</code> class is used to read/write DOC bookmark records
 *
 * @author Jeffrey A. Krzysztow
 * @author Steve Held
 * @author Reinhold F. Kainhofer
 * @author Pat Beirne
 * @version 1.0
 */
public class Bookmark implements Comparable {

	private String name = "";   // 16 Bytes	Bookmark name
	private int fileOffset = -1; // 4 DWord	file offset
								 // 20 bytes bookmark
	private boolean compareNamesNotLocation = false;
	private Position position;

	/**
	 * Default constructor, don't do anything
	 */
	public Bookmark() {}

	/**
	 * Constructor to duplicate an instance of this class
	 */
	public Bookmark(Bookmark bmk) {
		name = bmk.name;
		position = bmk.position;
		fileOffset = bmk.fileOffset;
	}

	/**
	 * Gets the name of the bookmark
	 * @return the name of the bookmark
	 */

	public String getName() {
		return name;
	}

	/**
	 * Sets the name of the bookmark
	 * @param name  the name of the bookmark
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * Gets the offset into the DOC for this Bookmark
	 * @return the offset into the DOC for this Bookmark
	 */

	public int getFileOffset() {
		if (position != null)
			fileOffset = position.getOffset();
		return fileOffset;
	}

	/**
	 * Sets the offset into the DOC for this Bookmark.
	 * If the offset might change before encoding, use
	 * <code>setPosition</code> instead.
	 * @param fileOffset    the offset into the DOC where this Bookmark occurs
	 */

	public void setFileOffset(int fileOffset) {
		this.fileOffset = fileOffset;
	}

	/**
	 * Gets the size of this Bookmark in bytes
	 * @return the number of bytes in the Bookmark
	 */
	public static int getSize() {
		return(20);
	}

	/**
	 * Reads this Bookmark from the DataInput
	 * @param di    the DataInput to read from
	 */
	public void read(DataInput di) throws IOException {
		byte[] b = new byte[16];
		di.readFully(b);
		name = new String(b);
		int i = name.indexOf('\0');
		if(i >= 0) {
			name = name.substring(0, i);
		}
		fileOffset = di.readInt();
	}

	/**
	 * Writes this Bookmark to the DataOutput
	 * @param out  the DataOutput to write to
	 */
	public void write(DataOutput out) throws IOException {
		byte[] b = new byte[16];
		byte[] bName = name.getBytes();
		for(int x=0; x < b.length; x++) {
			if(x < bName.length) {
				b[x] = bName[x];
			}
			else {
				b[x] = 0;
			}
		}
		out.write(b);
		if (position != null) fileOffset = position.getOffset();
		out.writeInt(fileOffset);
	}

	/**
	 * override Object.toString()
	 */
	public String toString() {
		return ">> Bookmark <<\nname = `" + name
			+ "`\nfileOffset = " + fileOffset;
	}

	/*
	 * If set to true, comparison of this Bookmark with other Bookmarks will be
	 * based on the Bookmarks' names. If set to false, comparison will be based
	 * on the location of the Bookmarks. Used by MakeDocJr.
	 */
	void setCompareNamesNotLocation(boolean compareNamesNotLocation) {
		this.compareNamesNotLocation = compareNamesNotLocation;
	}

	/**
	 * Implementation of the <code>Comparable</code> interface.<br><br>
	 *
	 * Users should normally not sort Bookmarks directly using any of the
	 * <code>Collections</code> sorting methods. Instead, the methods
	 * <code>MakeDocJr.sortBookmarksByLocation</code> or
	 * <code>MakeDocJr.sortBookmarksByName</code> should be called.
	 *
	 * @param o    another Bookmark to compare to
	 * @return a negative integer, zero, or a positive integer as this Bookmark
	 * is less than, equal to, or greater than the specified Bookmark.
	 */

	public int compareTo(Object o) {
		Bookmark bm = (Bookmark) o;
		if (!compareNamesNotLocation) {
			return (getFileOffset() < bm.getFileOffset()) ? -1 :
					((getFileOffset() == bm.getFileOffset()) ? 0 : 1);
		} else {
			return name.toLowerCase().compareTo(bm.name.toLowerCase());
		}
	}

	/**
	 * Sets the <code>java.swing.text.Position</code> of this Bookmark.
	 * If the bookmark location might change, use this method instead of
	 * setOffset().
	 *
	 * @param position  the Position of this bookmark
	 */
	public void setPosition(Position position) {
		this.position = position;
	}

}

