/*
 * Copyright 2002 Steve Held
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package net.onza.op.upswing.utilityfields;

import javax.swing.JTextField;
import javax.swing.text.*;

/**
 * This subclass of <code>javax.swing.JTextField</code> permits only a single
 * line of text to appear in a textfield.
 *
 * @author Steve Held
 * @version 1.0
 */

final public class SingleLineTextField extends JTextField {

	/**
	 * Constructs a new TextField. A default model is created, the initial
	 * string is null, and the number of columns is set to 0.
	 */

	public SingleLineTextField() {
	}

	/**
	 * Constructs a new empty TextField with the specified number of columns.
	 * A default model is created and the initial string is set to null.
	 *
	 * @param cols   the number of columns to use to calculate the preferred
	 * width. If columns is set to zero, the preferred width will be whatever
	 * naturally results from the component implementation.
	 */

	public SingleLineTextField(int cols) {
		super(cols);
	}

	/**
	 * Constructs a new TextField initialized with the specified text. A
	 * default model is created and the number of columns is 0.
	 *
	 * @param text      the text to be displayed, or null
	 */

	public SingleLineTextField(String text) {
		super(text);
	}

	/**
	 * Constructs a new TextField initialized with the specified text and
	 * columns. A default model is created.
	 *
	 * @param text      the text to be displayed, or null
	 * @param columns   the number of columns to use to calculate the
	 * preferred width. If columns is set to zero, the preferred width will be
	 * whatever naturally results from the component implementation.
	 */

	public SingleLineTextField(String text, int columns) {
		super(text, columns);
	}

	protected Document createDefaultModel() {
		return new SingleLineDocument();
	}

	static class SingleLineDocument extends PlainDocument {
		public void insertString(int offs, String str, AttributeSet a)
				throws BadLocationException {
			if (str == null) {
				return;
			}
			char[] textToInsert = str.toCharArray();
			char[] temp;
			int insertLength = textToInsert.length;
			temp = new char[insertLength];
			for (int i = 0; i < insertLength; i++) {
				temp[i] = textToInsert[i];
				if (temp[i] == '\n') {
					temp[i] = ' ';
				}
			}
			char[] textWillInsert = new char[insertLength];
			System.arraycopy(temp, 0, textWillInsert, 0, insertLength);
			super.insertString(offs, new String(textWillInsert), a);
		}
	}
}