/*
 * Copyright 2002 Steve Held
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package net.onza.op.upswing.utilityfields;

import javax.swing.JTextField;
import javax.swing.text.*;

/**
 * This subclass of <code>javax.swing.JTextField</code> permits only a single
 * line of text to appear in a textfield. Furthermore, by calling
 * <code>setMaxLength(int maxLength)</code>, no more than <code>maxLength</code>
 * characters will be allowed in the textfield.
 *
 * @author Steve Held
 * @version 1.0
 */

final public class FiniteTextField extends JTextField {

	/**
	 * Constructs a new TextField. A default model is created, the initial
	 * string is null, and the number of columns is set to 0.
	 */

	public FiniteTextField() {
	}

	/**
	 * Constructs a new empty TextField with the specified number of columns.
	 * A default model is created and the initial string is set to null.
	 *
	 * @param cols   the number of columns to use to calculate the preferred
	 * width. If columns is set to zero, the preferred width will be whatever
	 * naturally results from the component implementation.
	 */

	public FiniteTextField(int cols) {
		super(cols);
	}

	/**
	 * Constructs a new TextField initialized with the specified text.
	 * A default model is created and the number of columns is 0.
	 *
	 * @param text      the text to be displayed, or null
	 */

	public FiniteTextField(String text) {
		super(text);
	}

	/**
	 * Constructs a new TextField initialized with the specified text and
	 * columns. A default model is created.
	 *
	 * @param text      the text to be displayed, or null
	 * @param columns   the number of columns to use to calculate the preferred
	 * width. If columns is set to zero, the preferred width will be whatever
	 * naturally results from the component implementation.
	 */

	public FiniteTextField(String text, int columns) {
		super(text, columns);
	}
	/**
	 * Sets the maximum number of characters allowed in the textfield. If set to
	 * zero, there is no limit (for all practical purposes).
	 *
	 * @param maxDigits the maximum number of characters allowed in the textfield
	 */

	public void setMaxLength(int maxLength) {
		FiniteTextDocument.maxLength = maxLength;
	}

	/**
	 * Gets the maximum number of characters allowed in the textfield.
	 *
	 * @return  An <code>int</code> representing the maximum number of characters
	 *          allowed in the textfield. If zero, no limit is imposed.
	 */

	public int getMaxLength() {
		return FiniteTextDocument.maxLength;
	}

	protected Document createDefaultModel() {
		return new FiniteTextDocument();
	}

	static class FiniteTextDocument extends PlainDocument {

		static int maxLength;

		public void insertString(int offs, String str, AttributeSet a)
				throws BadLocationException {

			int presentLength = this.getLength();
			if (str == null) return;
			if (maxLength > 0 && ( offs >= maxLength ||
								   presentLength >= maxLength) ) return;

			char[] textToInsert = str.toCharArray();
			char[] temp;
			int proposedLength = presentLength + textToInsert.length;
			int insertLength;
			if (proposedLength > maxLength) {
				insertLength = maxLength - presentLength;
				temp = new char[insertLength];
				for (int i = 0; i < insertLength; i++) {
					temp[i] = textToInsert[i];
					if (temp[i] == '\n') {
						temp[i] = ' ';
					}
				}
			} else {
				insertLength = textToInsert.length;
				temp = new char[insertLength];
				for (int i = 0; i < insertLength; i++) {
					temp[i] = textToInsert[i];
					if (temp[i] == '\n') {
						temp[i] = ' ';
					}
				}
			}
			char[] textWillInsert = new char[insertLength];
			System.arraycopy(temp, 0, textWillInsert, 0, insertLength);
			super.insertString(offs, new String(textWillInsert), a);
		}
	}
}