/*
 * Copyright 2002 Steve Held
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package net.onza.op.upswing.utilityfields;

import javax.swing.JTextField;
import javax.swing.text.*;

/**
 * This subclass of <code>javax.swing.JTextField</code> permits only digits to
 * be entered into the textfield. Furthermore, by calling
 * <code>setMaxDigits(int maxDigits)</code>, no more than <code>maxDigits</code>
 * number of digits will be allowed in the textfield.
 *
 * @author Steve Held
 * @version 1.0
 */

final public class DigitsOnlyTextField extends JTextField {

	/**
	 * Constructs a new TextField. A default model is created, the initial
	 * string is null, and the number of columns is set to 0.
	 */

	public DigitsOnlyTextField() {
	}

	/**
	 * Constructs a new empty TextField with the specified number of columns.
	 * A default model is created and the initial string is set to null.
	 *
	 * @param cols   the number of columns to use to calculate the preferred
	 * width. If columns is set to zero, the preferred width will be whatever
	 * naturally results from the component implementation.
	 */

	public DigitsOnlyTextField(int cols) {
		super(cols);
	}

	/**
	 * Constructs a new TextField initialized with the specified text. A default
	 * model is created and the number of columns is 0.
	 *
	 * @param text      the text to be displayed, or null
	 */

	public DigitsOnlyTextField(String text) {
		super(text);
	}

	/**
	 * Constructs a new TextField initialized with the specified text and
	 * columns. A default model is created.
	 *
	 * @param text      the text to be displayed, or null
	 * @param columns   the number of columns to use to calculate the preferred
	 * width. If columns is set to zero, the preferred width will be whatever
	 * naturally results from the component implementation.
	 */

	public DigitsOnlyTextField(String text, int columns) {
		super(text, columns);
	}

	/**
	 * Sets the maximum number of digits allowed in the textfield. If set to
	 * zero, there is no limit (for all practical purposes). Negative numbers
	 * are ignored.<br>
	 * <br>
	 * <u>Note:</u> <code>maxDigits</code> sets the maximum <i>number of digits</i>,
	 * not the maximum <i>value</i>.
	 *
	 * @param maxDigits     the maximum number of digits allowed in the textfield
	 */

	public void setMaxDigits(int maxDigits) {
		if (maxDigits < 0) return;
		DigitsOnlyDocument.maxDigits = maxDigits;
	}

	/**
	 * Gets the maximum number of digits allowed in the textfield.
	 *
	 * @return  An <code>int</code> representing the maximum number of digits
	 *          allowed in the textfield. If zero, no limit is imposed.
	 */

	public int getMaxDigits() {
		return DigitsOnlyDocument.maxDigits;
	}

protected Document createDefaultModel() {
	return new DigitsOnlyDocument();
}


	static class DigitsOnlyDocument extends PlainDocument {

		static int maxDigits;

		public void insertString(int offs, String str, AttributeSet a)
				throws BadLocationException {
			if (str == null) return;
			if (maxDigits > 0 && offs >= maxDigits) return;

			char[] numbers = str.toCharArray();
			char[] numbersOnly = new char[numbers.length];
			int i, j;
			for (i = j = 0; i < numbers.length; i++, j++) {
				numbersOnly[j] = numbers[i];
				if (!Character.isDigit(numbers[i])) {
					j--;
				}
			}
			int totalDigits = offs + j;
			if (maxDigits != 0) j = ( totalDigits > maxDigits ? maxDigits - offs : j );
			char[] temp = new char[j];
			System.arraycopy(numbersOnly, 0, temp, 0, j);
			super.insertString(offs, new String(temp), a);
		}
	}
}