/*
 * EncodeOptions.java
 *
 * Copyright 2003, 2001 Steve Held
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package makedocjr;

import java.io.*;

/**
 * <code>EncodeOptions</code> is used to set the options for encoding text into
 * a DOC file using <code>MakeDocJr</code>.
 *
 * @author Steve Held
 * @author Jeffrey A. Krzysztow
 * @version 1.0.1
 */
public class EncodeOptions implements Cloneable, Serializable {

	/**
	 * represents the option to not sort the bookmarks
	 */
	public static final int NO_SORT = 0;

	/**
	 * represents the option to sort the bookmarks by location
	 */
	public static final int LOCATION_SORT = 1;

	/**
	 * represents the option to sort the bookmarks by name
	 */
	public static final int ALPHA_SORT = 2;

	boolean compress = true;
	int docID = DatabaseHeader.ReaderID;
	boolean compressEOL = false;
	boolean unwrap = false;
	int lineLength = 0;
	boolean makePrivate = false;
	byte category = 0;
	int sortBookmarks = EncodeOptions.NO_SORT;
	transient String autoBookmark = null;
	String directoryPath = System.getProperty("file.separator");

	/**
	 * Default constructor
	 */
	public EncodeOptions() {
	}

	/**
	 * Checks if DOC file will be compressed when encoded.
	 *
	 * @return true if DOC file will be compressed when encoded
	 */
	public boolean getCompress() {
		return compress;
	}

	/**
	 * Determines if the DOC file will be compressed when encoded. Defaults
	 * to <code>true</code>.
	 *
	 * @param compress  set to false to not compress the DOC file
	 */
	public void setCompress(boolean compress) {
		this.compress = compress;
	}

	/**
	 * Gets the ID representing the type of DOC file.
	 *
	 * @return the ID representing the type of DOC file. DatabaseHeader
	 * supports two types: Generic (DatabaseHeader.ReaderID) and TealDoc
	 * (DatabaseHeader.TealDocID)
	 */
	public int getDocID() {
		return docID;
	}

	/**
	 * Sets the ID representing the type of DOC file. Default value is the
	 * generic <code>DatabaseHeader.ReaderID</code>.
	 *
	 * @param docID     the ID representing the type of DOC file.
	 * DatabaseHeader supports two types: Generic
	 * (DatabaseHeader.ReaderID) and TealDoc (DatabaseHeader.TealDocID)
	 */
	public void setDocID(int docID) {
		this.docID = docID;
	}

	/**
	 * Checks if multiple linebreaks will be removed during encoding.
	 *
	 * @return true if multiple linebreaks will be removed during encoding
	 */
	public boolean getCompressEOL() {
		return compressEOL;
	}

	/**
	 * Determines if multiple linebreaks will be reduced to two. Two linebreaks
	 * typically are used to represent a new paragraph. Default is set to false.
	 *
	 * @param compressEOL
	 */
	public void setCompressEOL(boolean compressEOL) {
		this.compressEOL = compressEOL;
	}

	/**
	 * Checks if single linebreaks will be removed.
	 *
	 * @return true if single linebreaks will be removed
	 */
	public boolean getUnwrap() {
		return unwrap;
	}

	/**
	 * Determines if single linebreaks will be removed. This can have the effect of
	 * ruining lists, but makes paragraphs flow without "broken" lines.
	 * Default is set to false.
	 *
	 * @param unwrap        true if single linebreaks will be removed
	 */
	public void setUnwrap(boolean unwrap) {
		this.unwrap = unwrap;
	}

	/**
	 * Checks the minimum length a line must be before being wrapped.
	 *
	 * @return the minimum length a line must be before being wrapped.
	 */
	public int getLineLength() {
		return lineLength;
	}

	/**
	 * Sets the minimum length a line must be before being wrapped. This can be
	 * used to preserve lists in the text. <code>setUnwrap(true)</code>
	 * must also be set. Default value is 0.
	 *
	 * @param lineLength    the minimum length a line must be before being wrapped
	 */
	public void setLineLength(int lineLength) {
		this.lineLength = lineLength;
	}

	/**
	 * Checks if the DOC file will be encoded as private.
	 *
	 * @return true if the DOC file will be encoded as private
	 */
	public boolean getMakePrivate() {
		return makePrivate;
	}

	/**
	 * Determines if the DOC file will be encoded as private. Default value is
	 * false, which means the DOC file will not be private.
	 *
	 * @param makePrivate   boolean indicating the DOC file will be encoded as private
	 */
	public void setMakePrivate(boolean makePrivate) {
		this.makePrivate = makePrivate;
	}

	/**
	 * Gets the DOC file category number.
	 *
	 * @return the number representing the DOC file category.
	 */
	public byte getCategory() {
		return category;
	}

	/**
	 * Sets the DOC file category number. Common categories on a PDA are
	 * business and personal, but they can be anything. Therefore, there are no
	 * standard numbers that correlate to categories on a PDA.
	 *
	 * @param category  the number corresponding to a DOC file category.
	 */
	public void setCategory(byte category) {
		this.category = category;
	}

	/**
	 * Gets the integer corresponding to the bookmark sorting option.
	 *
	 * return the integer  corresponding to the bookmark sorting option
	 */
	public int getSortBookmarks() {
		return sortBookmarks;
	}

	/**
	 * Sets the bookmark sorting option. Default value is
	 * <code>EncodeOptions.NO_SORT</code>
	 *
	 * @param sortBookmarks     int representing the type of bookmark sort to
	 * perform. Use EncodeOptions.NO_SORT for no sorting,
	 * EncodeOptions.LOCATION_SORT for sorting by location, and
	 * EncodeOptions.ALPHA_SORT for sorting by name.
	 */
	public void setSortBookmarks(int sortBookmarks) {
		this.sortBookmarks = sortBookmarks;
	}

	/**
	 * Gets the text that MakeDocJr will search for at the beginning of every line
	 * to create bookmarks automatically. A null value means that there will be no
	 * creation of bookmarks automatically.
	 *
	 * @return the text to use in creating bookmarks automatically.
	 * A null value means that bookmarks will not be created automatically.
	 */
	public String getAutoBookmark() {
		return autoBookmark;
	}

	/**
	 * MakeDocJr can create bookmarks for the DOC file automatically when encoding.
	 * It will search for all lines beginning with the specified String and create
	 * bookmarks if there are any lines found. If a null value is passed in, bookmarks
	 * will not be created automatically.
	 *
	 * @param autoBookmark  a String to base the creation of bookmarks automatically.
	 * use null to cancel creation of bookmarks automatically.
	 */
	public void setAutoBookmark(String autoBookmark) {
		this.autoBookmark = autoBookmark;
	}

	/**
	 * Gets the default path for saving encoded files.
	 *
	 * @return the default path for saving encoded files as a String
	 */
	public String getDirectoryPath() {
		return directoryPath;
	}

	/**
	 * Sets the default directory path for saving encoded files.
	 *
	 * @param directoryPath     the path of the default directory to save
	 * encoded files
	 */
	public void setDirectoryPath(String directoryPath) {
		this.directoryPath = directoryPath;
	}

	/**
	 * Overrides Object.clone
	 */
	public Object clone() {
		EncodeOptions eo = null;
		try {
			eo = (EncodeOptions) super.clone();
		} catch (CloneNotSupportedException e) { } // should never happen
		return eo;
	}

}