/*
 * DatabaseHeader.java
 *
 * Copyright 2001 by BRiSK Software,
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package makedocjr;

import java.io.*;
import java.util.*;
import java.util.zip.*;

/**
 * <code>DatabaseHeader</code> class is used to read/write Palm database header
 * records.
 * NOTE: Information about the various fields was taken from
 * <a href="www.roadcoders.com/pdb.html">Road Coders</a>
 *
 * @author Jeffrey A. Krzysztow
 * @author Pat Beirne
 * @version 1.0
 */
public class DatabaseHeader {
	/**
	 * database ID for TealDoc PilotDoc reader
	 */
	public final static int TealDocID = 0x546c4463;	// 'TlDc'

	/**
	 * database ID for generic PilotDoc reader
	 */
	public final static int ReaderID = 0x52454164;	// 'REAd'

	/**
	 * indicates type of data stored in database
	 */
	public final static int TEXt = 0x54455874;		// 'TEXt'

	/**
	 * name of database
	 */
	public String name = "";					// 32

	/**
	 * attributes of database
	 *  0x0001 Resource database
	 *	0x0002 Read-only
	 *	0x0004 Dirty AppInfoArea
	 *	0x0008 Backup this database (i.e. no conduit exists)
	 *	0x0010 Okay to install newer over existing copy, if present on PalmPilot
	 *	0x0020 Force the PalmPilot to reset after this database is installed
	 *	0x0040 Don't allow copy of file to be beamed to other Pilot.
	 *  0x0080 File stream database
	 *  0x0100 Should be "hidden"
	 *  0x8000 Database was not closed properly
	 */
	public short attribute = 0;					// 2 Word

	/**
	 * version of database
	 * defined by application
	 */
	public short version = 0;					// 2 Word

	/**
	 * creation date of database
	 * Expressed as the number of seconds since January 1, 1904.
	 * <b>The database will not install if this value is zero.</b>
	 */
	public int creationDate = 0;				// 4 DWord

	/**
	 * last modification date of database
	 * Expressed as the number of seconds since January 1, 1904.
	 * <b>The database will not install if this value is zero.</b>
	 */
	public int modificationDate = 0;			// 4 DWord

	/**
	 * last date database was HotSynced
	 * Expressed as the number of seconds since January 1, 1904.
	 * The database will install if this value is zero.
	 */
	public int lastBackupDate = 0;				// 4 DWord

	/**
	 * modification number
	 * Set to zero
	 */
	public int modificationNumber = 0;			// 4 DWord

	/**
	 * appInfoID
	 * The byte number in the PDB file (counting from zero) at
	 * which the AppInfoArea is located. This must be the first
	 * entry in the Data portion of the PDB file. If this
	 * database does not have an AppInfoArea, set this value to
	 * zero.
	 */
	public int appInfoID = 0;					// 4 DWord

	/**
	 * sortInfoID
	 * The byte number in the PDB file (counting from zero) at
	 * which the SortInfoArea is located. This must be placed
	 * immediately after the AppInfoArea, if one exists, within
	 * the Data portion of the PDB file. If this database does
	 * not have a SortInfoArea, set this value to zero. Do not
	 * use this.
	 */
	public int sortInfoID = 0;					// 4 DWord

	/**
	 * ID of application that can update database
	 */
	public int typeID = 0;						// 4 DWord

	/**
	 * ID of application
	 */
	public int creatorID = 0;					// 4 DWord

	/**
	 * uniqueIDSeed
	 * This is used to generate the Unique ID number of
	 * subsequent records. This should be set to zero.
	 */
	public int uniqueIDSeed = 0;				// 4 DWord

	/**
	 * nextRecordListID
	 * Set this to zero. The element is used only in the
	 * in-memory representation of a PDB file, but exists
	 * in the external version for consistency.
	 */
	public int nextRecordListID = 0;			// 4 DWord

	/**
	 * number of records in database
	 */
	public short numRecords = 0;				// 2 Word
												// 78 bytes for Database Header

	/**
	 * Default constructor
	 */
	DatabaseHeader() {
		// find the current time, and convert into number of seconds since Jan 1, 1904
		Calendar cl = Calendar.getInstance();
		Date now = new Date();
		creationDate = (int)((now.getTime() + cl.get(Calendar.ZONE_OFFSET) + cl.get(Calendar.DST_OFFSET))
			/ 1000 + SecondsSince1904);
		modificationDate = creationDate;
	}

	/**
	 * The size of the DatabaseHeader in bytes
	 * @return the number of bytes in the DatabaseHeader
	 */
	public static int getSize() {
		return(78);
	}

	/**
	 * Reads the DatabaseHeader from the DataInput
	 * Added DataFormatException in version 1.3.2
	 * @param the DataInput to read from
	 */
	public void read(DataInput di)
			throws IOException, DataFormatException {
		byte[] b = new byte[32];
		di.readFully(b);
		name = new String(b);
		int i = name.indexOf(0);
		if (i == -1) throw new DataFormatException();
		name = name.substring(0, i);
		attribute = di.readShort();
		version = di.readShort();
		creationDate = di.readInt();
		modificationDate = di.readInt();
		lastBackupDate = di.readInt();
		modificationNumber = di.readInt();
		appInfoID = di.readInt();
		sortInfoID = di.readInt();
		typeID = di.readInt();
		creatorID = di.readInt();
		uniqueIDSeed = di.readInt();
		nextRecordListID = di.readInt();
		numRecords = di.readShort();
	}

	/**
	 * Writes the DatabaseHeader to the DataInput
	 * @param the DataInput to write to
	 */
	public void write(DataOutput out) throws IOException {
		byte[] b = new byte[32];
		b[b.length - 1] = 0;
		byte[] bName = name.getBytes();
		for(int x = 0; x < b.length - 1; x++) {
			if(x < bName.length) {
				b[x] = bName[x];
			}
			else {
				b[x] = 0;
			}
		}
		if(bName.length >= b.length) {
			b[b.length - 2] = (byte)'.';
			b[b.length - 3] = (byte)'.';
			b[b.length - 4] = (byte)'.';
		}
		out.write(b);
		out.writeShort(attribute);
		out.writeShort(version);
		out.writeInt(creationDate);
		out.writeInt(modificationDate);
		out.writeInt(lastBackupDate);
		out.writeInt(modificationNumber);
		out.writeInt(appInfoID);
		out.writeInt(sortInfoID);
		out.writeInt(typeID);
		out.writeInt(creatorID);
		out.writeInt(uniqueIDSeed);
		out.writeInt(nextRecordListID);
		out.writeShort(numRecords);
	}

	/**
	 * override Object.toString()
	 * @since 1.0
	 */
	public String toString()  {
		char[] creatorIDa = new char[4];
		creatorIDa[0] = (char)((creatorID >> 24) & 0xff);
		creatorIDa[1] = (char)((creatorID >> 16) & 0xff);
		creatorIDa[2] = (char)((creatorID >> 8) & 0xff);
		creatorIDa[3] = (char)(creatorID & 0xff);
		char[] typeIDa = new char[4];
		typeIDa[0] = (char)((typeID >> 24) & 0xff);
		typeIDa[1] = (char)((typeID >> 16) & 0xff);
		typeIDa[2] = (char)((typeID >> 8) & 0xff);
		typeIDa[3] = (char)(typeID & 0xff);
		return ">> DatabaseHeader <<"
			+ "\nname = `" + name + "`"
			+ "\nattribute = " + attribute
			+ "\nversion = " + version
			+ "\ncreationDate = " + creationDate
			+ "\nmodificationDate = " + modificationDate
			+ "\nlastBackupDate = " + lastBackupDate
			+ "\nmodificationNumber = " + modificationNumber
			+ "\nappInfoID = " + appInfoID
			+ "\nsortInfoID = " + sortInfoID
			+ "\ntypeID = " + typeID + " 0x" + Integer.toHexString(typeID) + " `" + new String(typeIDa) + "`"
			+ "\ncreatorID = " + creatorID + " 0x" + Integer.toHexString(creatorID) + " `" + new String(creatorIDa) + "`"
			+ "\nuniqueIDSeed = " + uniqueIDSeed + "\nnextRecordListID = " + nextRecordListID
			+ "\nnumRecords = " + numRecords;
	}

	/**
	 * Sets the modification date member to now
	 */
	public void setModificationDate() {
		// find the current time, and convert into number of seconds since Jan 1, 1904
		Calendar cl = Calendar.getInstance();
		Date now = new Date();
		modificationDate = (int)((now.getTime() + cl.get(Calendar.ZONE_OFFSET) + cl.get(Calendar.DST_OFFSET))
			/ 1000 + SecondsSince1904);
	}

	/**
	 * returns the modification date as a Date
	 * @return Date indicating modification date
	 */
	public Date getModificationDate() {
		Calendar cl = Calendar.getInstance();
		long temp = new Integer(modificationDate).longValue();
		if(temp < 0) {
			temp += ((long) 1) << 32;
		}
		return new Date((temp - SecondsSince1904) * 1000 - cl.get(Calendar.ZONE_OFFSET) - cl.get(Calendar.DST_OFFSET));
	}

	/**
	 * returns the creation date as a Date
	 * @return Date indicating creation date
	 */
	public Date getCreationDate() {
		Calendar cl = Calendar.getInstance();
		long temp = new Integer(creationDate).longValue();
		if(temp < 0) {
			temp += ((long) 1) << 32;
		}
		return new Date((temp - SecondsSince1904) * 1000 - cl.get(Calendar.ZONE_OFFSET) - cl.get(Calendar.DST_OFFSET));
	}

	/**
	 * Seconds since 1904
	 * @see SecondsSince1904
	 */
	private final static long SecondsSince1904 = 2082844800;

	/**
	 * converts the String representation of typeID to an int representation
	 * @param typeID String representation of typeID
	 * @return int representation of typeID
	 */
	public static int convertStringToTypeID(String typeID) {
		byte typeIDa[] = typeID.getBytes();
		return ((0xff & typeIDa[0]) << 24)
			+ ((0xff & typeIDa[1]) << 16)
			+ ((0xff & typeIDa[2]) << 8)
			+ (0xff & typeIDa[3]);
	}

	/**
	 * converts the int representation of typeID to a String representation
	 * @param typeID int representation of typeID
	 * @return String representation of typeID
	 */
	public static String convertIntToStringTypeID(int typeID)	 {
		char[] typeIDa = new char[4];
		typeIDa[0] = (char)((typeID >> 24) & 0xff);
		typeIDa[1] = (char)((typeID >> 16) & 0xff);
		typeIDa[2] = (char)((typeID >> 8) & 0xff);
		typeIDa[3] = (char)(typeID & 0xff);
		return new String(typeIDa);
	}

	/**
	 * converts the String representation of creatorID to an int representation
	 * @param creatorID String representation of creatorID
	 * @return int representation of creatorID
	 */
	public static int convertStringToCreatorID(String creatorID) {
		byte creatorIDa[] = creatorID.getBytes();
		return ((0xff & creatorIDa[0]) << 24)
			+ ((0xff & creatorIDa[1]) << 16)
			+ ((0xff & creatorIDa[2]) << 8)
			+ (0xff & creatorIDa[3]);
	}

	/**
	 * converts the int representation of creatorID to a String representation
	 * @param creatorID int representation of creatorID
	 * @return String representation of creatorID
	 */
	public static String convertIntToStringCreatorID(int creatorID)	 {
		char[] creatorIDa = new char[4];
		creatorIDa[0] = (char)((creatorID >> 24) & 0xff);
		creatorIDa[1] = (char)((creatorID >> 16) & 0xff);
		creatorIDa[2] = (char)((creatorID >> 8) & 0xff);
		creatorIDa[3] = (char)(creatorID & 0xff);
		return new String(creatorIDa);
	}

}

