#include "itemset.h"
#include "order.h"
#include "miner.h"
#include "seq_tree.h"
#include "dataset.h"
#include <getopt.h>
#include <math.h>
#include <stdio.h>
#include <string.h>



void
outofmemory()
{
	fprintf(stderr, "Oh noes! out of memory\n");
	exit(1);
}








int
main(int argc, char ** argv)
{
	static struct option longopts[] = {
		{"out",             required_argument,  NULL, 'o'},
		{"in",              required_argument,  NULL, 'i'},
		{"format",          required_argument,  NULL, 'p'},
		{"order",           required_argument,  NULL, 'r'},
		{"thresh",          required_argument,  NULL, 't'},
		{"free_alpha",      required_argument,  NULL, 'f'},
		{"free_thresh",     required_argument,  NULL, 'F'},
		{"ts_alpha",        required_argument,  NULL, 's'},
		{"ts_thresh",       required_argument,  NULL, 'S'},
		{"ndi_alpha",       required_argument,  NULL, 'n'},
		{"ndi_thresh",      required_argument,  NULL, 'N'},
		{"help",            no_argument,        NULL, 'h'},
		{ NULL,             0,                  NULL,  0 }
	};

	const char *inname = NULL;
	const char *outname = NULL;
	const char *format = "%i (%t)";
	const char *order = "";


	params par;

	std::set_new_handler(outofmemory);

	int ch;
	while ((ch = getopt_long(argc, argv, "o:i:hp:t:f:F:s:S:n:N:r:", longopts, NULL)) != -1) {
		switch (ch) {
			case 'h':
				printf("Usage: %s -i <input file> -o <output file> [options]\n", argv[0]);
				printf("  -h    print this help\n");
				printf("  -i    input file\n");
				printf("  -o    output file\n");
				printf("constraints:\n");
				printf("  -t    support threshold\n");
				printf("  -f    alpha for free itemsets\n");
				printf("  -F    threshold for free itemsets\n");
				printf("  -s    alpha for totally shattered itemsets\n");
				printf("  -S    threshold for totally shattered itemsets\n");
				printf("  -n    alpha for non-derivable itemsets\n");
				printf("  -N    threshold for non-derivable itemsets\n");
				printf("order:\n");
				printf("  -r  <order method>\n");
				printf("      support    order acc. to support\n");
				printf("      rob-free   order acc. to robustness for free itemsets\n");
				printf("      rob-ts     order acc. to robustness for totally shattered itemsets\n");
				printf("      rob-ndi    order acc. to robustness for non-derivable itemsets\n");
				printf("      mv-free    order acc. to margin-vectors for free itemsets\n");
				printf("      mv-ts      order acc. to margin-vectors for totally shattered itemsets\n");
				printf("      mv-ndi     order acc. to margin-vectors for non-derivable itemsets\n");
				printf("printing:\n");
				printf("  -p    format pattern used for printing itemset, default: %%i (%%t) \n");
				printf("        %%i  items\n");
				printf("        %%t  support\n");
				printf("        %%f  robustness for free itemsets\n");
				printf("        %%s  robustness for totally shattered itemsets\n");
				printf("        %%n  robustness for non-derivable itemsets\n");


				return 0;
				break;
			case 't':
				par.sup_threshold = atoi(optarg);
				break;
			case 'i':
				inname = optarg;
				break;
			case 'p':
				format = optarg;
				break;
			case 'r':
				order = optarg;
				break;
			case 'o':
				outname = optarg;
				break;
			case 'F':
				par.free.threshold = atof(optarg);
				break;
			case 'f':
				par.free.la = log(1 - atof(optarg));
				par.free.set = true;
				break;
			case 'S':
				par.ts.threshold = atof(optarg);
				break;
			case 's':
				par.ts.la = log(1 - atof(optarg));
				par.ts.set = true;
				break;
			case 'N':
				par.ndi.threshold = atof(optarg);
				break;
			case 'n':
				par.ndi.la = log(1 - atof(optarg));
				par.ndi.set = true;
				break;
		}
	}


	if (inname == NULL) {
		fprintf(stderr, "Missing input file\n");
		return 1;
	}
	
	if (outname == NULL) {
		fprintf(stderr, "Missing output file\n");
		return 1;
	}

	FILE *f = fopen(inname, "r");
	dataset d;
	d.read_file(f);
	fclose(f);

	miner m(d, par);
	m.mine();

	itemsetvector itemsets = m.itemsets();

	if (strcmp(order, "support") == 0)
		order_support(itemsets);
	else if (strcmp(order, "rob-free") == 0)
		order_robust_free(itemsets);
	else if (strcmp(order, "rob-ndi") == 0)
		order_robust_ndi(itemsets);
	else if (strcmp(order, "rob-ts") == 0)
		order_robust_ts(itemsets);
	else if (strcmp(order, "mv-free") == 0)
		order_mv_free(itemsets);
	else if (strcmp(order, "mv-ts") == 0)
		order_mv_ts(itemsets);
	else if (strcmp(order, "mv-ndi") == 0)
		order_mv_ndi(itemsets);

	f = fopen(outname, "w");
	for (uint32_t i = 0; i < itemsets.size(); i++)
		itemsets[i]->print(f, format);
	fclose(f);

	return 0;
}
