#include <vector>
#include <stdint.h>
#include <math.h>
#include <assert.h>
#include <fstream>
#include <sstream>
#include <limits>

typedef std::vector<uint32_t> intvector;
typedef std::vector<double> doublevector;

class kendall {
	public:
		kendall(const doublevector & x, const doublevector & y) :
			m_x(x), m_y(y), m_perm(x.size()), m_temp(x.size())
		{
			for (uint32_t i = 0; i < m_perm.size(); i++) m_perm[i] = i;
		}

		bool operator() (int i,int j) {return m_x[i] < m_x[j] || (m_x[i] == m_x[j] && m_y[i] < m_y[j]);}

		double stat();
	protected:

		uint64_t mergesort(uint32_t offset, uint32_t len);
		uint64_t ties(const doublevector & x) const;
		uint64_t jointties() const;

		double x(uint32_t i) const {return m_x[m_perm[i]];}
		double y(uint32_t i) const {return m_y[m_perm[i]];}

		doublevector m_x, m_y;
		intvector m_perm, m_temp;
};

double
kendall::stat()
{
	std::sort(m_perm.begin(), m_perm.end(), *this);

	for (uint32_t i = 1; i < m_x.size(); i++)
		assert(x(i - 1) < x(i) || (x(i - 1) == x(i) && y(i - 1) <= y(i)));

	uint64_t t = jointties();
	uint64_t u = ties(m_x);
	uint64_t e = mergesort(0, m_x.size());
	uint64_t v = ties(m_y);
	uint64_t tot = m_x.size()*(m_x.size() - 1) / 2;

	//printf("%llu %d\n", u, m_x.size());

	double d = sqrt(tot - u)*sqrt(tot - v);


	return ((tot - (v + u - t)) - 2.0 * e) / d;
}

uint64_t
kendall::ties(const doublevector & x) const
{
	uint64_t res = 0;
	uint32_t first = 0;

	for (uint32_t i = 0; i < x.size(); i++)  {
		if (x[m_perm[first]] != x[m_perm[i]]) {
			res += uint64_t(i - first) * (i - first - 1) / 2;
			first = i;
		}
	}
	res += uint64_t(x.size() - first) * (x.size() - first - 1) / 2;
	return res;
}

uint64_t
kendall::jointties() const
{
	uint64_t res = 0;
	uint32_t first = 0;

	for (uint32_t i = 0; i < m_x.size(); i++)  {
		if (x(first) != x(i) || y(first) != y(i)) {
			res += uint64_t(i - first) * (i - first - 1) / 2;
			first = i;
		}
	}
	res += uint64_t(m_x.size() - first) * (m_x.size() - first - 1) / 2;
	return res;
}




uint64_t
kendall::mergesort(uint32_t offset, uint32_t len)
{
	assert(offset + len <= m_x.size());

	if (len == 1) return 0;
	if (len == 2) {
		if (y(offset) <= y(offset + 1))
			return 0;
		std::swap(m_perm[offset], m_perm[offset + 1]);
		return 1;
	}

	uint32_t len1 = len / 2;
	uint32_t len2 = len - len1;
	uint32_t middle = offset + len1;

	uint32_t exchange = 0;
	exchange += mergesort(offset, len1);
	exchange += mergesort(middle, len2);

	if (y(middle - 1) <= y(middle))
		return exchange;

	int32_t i = 0, j = 0, k = 0;
	int32_t d = 0;

	while (j < len1 || k < len2) {
		if (k == len2 || (j < len1 && y(offset + j) <= y(middle + k))) {
			m_temp[i] = m_perm[offset + j];
			d = i - j;
			j++;
		}
		else {
			m_temp[i] = m_perm[middle + k];
			d = int32_t(offset) + i - (int32_t(middle) + k);
			k++;
		}
		if (d > 0) exchange += d;
		i++;
	}

	std::copy(m_temp.begin(), m_temp.begin() + len, m_perm.begin() + offset);
	return exchange;
}

std::pair<uint32_t, uint32_t>
findlimits(const char *file)
{
	std::ifstream f;
	f.open(file);

	std::string line;

	uint32_t a = std::numeric_limits<uint32_t>::max();
	uint32_t b = 0;

	uint32_t i = 1;
	while (std::getline(f, line)) {
		std::istringstream sf(line);
		uint32_t v;
		while (sf >> v) {
			a = std::min(a, v);
			b = std::max(b, v);
		}
		i++;
	}
	return std::make_pair(a, b);
}




void
read(const char *file, doublevector & out)
{
	std::pair<int, int> l = findlimits(file);
	out.resize(l.second - l.first + 1);

	std::ifstream f;
	f.open(file);

	std::string line;

	uint32_t i = 1;
	while (std::getline(f, line)) {
		std::istringstream sf(line);
		uint32_t v;
		while (sf >> v) {
			out[v - l.first] = i;
		}
		i++;
	}

	uint32_t j = 0;
	for (uint32_t i = 0; i < out.size(); i++) 
		if (out[i] > 0) out[j++] = out[i];
	out.resize(j);
}


int
main(int argc, char **argv)
{
	doublevector x, y;
	read(argv[1], x);
	read(argv[2], y);

	kendall k(x, y);

	printf("%f\n", k.stat());

	return 0;
}
