#include "graph.h"

#include <map>

#include <stdio.h>

typedef std::multimap<weight_t, uint32_t> weightnodemap;

typedef std::map<uint32_t, uint32_t> labelmap;


graph*
read(const char *name, bool weighted, bool duplicate)
{
	FILE *f = fopen(name, "r");

	uint32_t ecnt = 0;
	uint32_t vcnt = 0;
	labelmap lm;

	uint32_t a, b;
	double r;


	while (fscanf(f, "%d%d", &a, &b) == 2) {
		if (weighted) fscanf(f, "%lf", &r);
		if (duplicate && a > b) continue;
		if (lm.find(a) == lm.end()) lm[a] = vcnt++;
		if (lm.find(b) == lm.end()) lm[b] = vcnt++;
		ecnt++;
	}


	weightvector w(ecnt);
	std::fill(w.begin(), w.end(), 1);
	std::vector<std::pair<uint32_t, uint32_t> > edges(ecnt);


	rewind(f);
	for (uint32_t i = 0; i < ecnt; i++) {
		do {	
			fscanf(f, "%d%d", &a, &b);
			if (weighted) fscanf(f, "%lf", &w[i]);
		} while (duplicate && a > b);
		edges[i] = std::make_pair(lm[a], lm[b]);
	}

	graph *g = new graph(edges.begin(), edges.end(), w.begin(), vcnt);


	// Create labels

	intvector reverse(vcnt);
	for (labelmap::iterator it = lm.begin(); it != lm.end(); ++it)
		reverse[it->second] = it->first;

	namemap labels = boost::get(boost::vertex_name, *g);

	viterator vi = boost::vertices(*g).first;
	for (uint32_t i = 0; i < vcnt; i++) {
		labels[vi[i]] = reverse[i];
	}

	return g;
}


double
vertexweights(graph & g, weightvector & w)
{
	double tw = 0;

	w.resize(boost::num_vertices(g));
	std::fill(w.begin(), w.end(), 0);

	indexmap index = boost::get(boost::vertex_index, g);
	weightmap wm = boost::get(boost::edge_weight, g);

	eiterator ei, ei_end;
	for (boost::tie(ei, ei_end) = boost::edges(g); ei != ei_end; ++ei) {
		//printf("%d %d\n", index[boost::source(*ei, g)], index[boost::target(*ei, g)]);
		w[index[boost::source(*ei, g)]] += wm[*ei];
		w[index[boost::target(*ei, g)]] += wm[*ei];
		tw += wm[*ei];
	}

	return tw;
}


void
order(graph & g, weightvector & w, intvector & ord)
{
	ord.resize(boost::num_vertices(g));

	vertexweights(g, w);

	indexmap index = boost::get(boost::vertex_index, g);
	weightmap wm = boost::get(boost::edge_weight, g);

	weightnodemap queue;
	std::vector<weightnodemap::iterator> loc(w.size());

	for (uint32_t i = 0; i < w.size(); i++)
		loc[i] = queue.insert(std::make_pair(w[i], i));

	for (uint32_t i = 0; i < w.size(); i++) {
		uint32_t ind = queue.begin()->second;
		w[i] = loc[ind]->first;
		ord[i] = loc[ind]->second;
		queue.erase(loc[ind]);
		loc[ind] = queue.end();

		viterator vi = boost::vertices(g).first;
		vertex v = vi[ind];

		oiterator oi, oi_end;
		for (boost::tie(oi, oi_end) = boost::out_edges(v, g); oi != oi_end; ++oi) {
			uint32_t j =  index[boost::target(*oi, g)];

			if (loc[j] == queue.end()) continue;

			weight_t c = loc[j]->first - wm[*oi];
			queue.erase(loc[j]);
			loc[j] = queue.insert(std::make_pair(c, j));

		}
	}
}

void
print(FILE *f, graph & g, const weightvector & w, const intvector & ord)
{
	namemap labels = boost::get(boost::vertex_name, g);
	viterator vi = boost::vertices(g).first;

	for (uint32_t i = 0; i < ord.size(); i++) {
		fprintf(f, "%d %d %f\n", ord[i], labels[vi[ord[i]]], w[i]);
	}
}


void
print(FILE *f, graph & g, const intvector & ord, const intvector & groups)
{
	namemap labels = boost::get(boost::vertex_name, g);
	viterator vi = boost::vertices(g).first;
	uint32_t cur = groups[0];

	for (uint32_t i = 0; i < ord.size(); i++) {
		if (groups[i] != cur) {
			fprintf(f, "\n");
			cur = groups[i];
		}
		fprintf(f, "%d ", labels[vi[ord[i]]]);
	}
	fprintf(f, "\n");
}

