#include "graph.h"
#include "border.h"

#include <getopt.h>
#include <stdio.h>

#include <list>

struct profile {
	profile() : weight(0), size(0) {}
	double weight;
	uint32_t size;
};

typedef std::list<profile> profilelist;

void
core(const weightvector & w, intvector & out, profilelist & pl)
{
	out.resize(w.size());
	uint32_t cur = 0;
	weight_t cw = w[0];
	profile p;

	for (uint32_t i = 0; i < w.size(); i++) {
		if (w[i] > cw)  {
			cw = w[i];
			cur++;
			pl.push_front(p);
			p = profile(); 
		}

		out[i] = cur;
		p.weight += w[i];
		p.size++;
	}
	pl.push_front(p);
}


void
pava(const weightvector & w, intvector & out, profilelist & pl)
{
	blockvector bv(w.size());
	border(w, bv);

	out.resize(w.size());

	uint32_t j = w.size() - 1;
	uint32_t cur = 0;
	profile p;

	for (int32_t i = w.size() - 1; i >= 0; i--) {
		if (uint32_t(i) + bv[j].size <= j) {
			j = i;
			cur++;
			pl.push_back(p);
			p = profile(); 
		}
		out[i] = cur;
		p.weight += w[i];
		p.size++;
	}
	pl.push_back(p);
}

uint32_t 
monotonic(graph & g, compinfo & c, double lw, intvector & groups, profilelist & pl, uint32_t val)
{
	uint32_t n = c.vertices.size();

	double alpha = (c.totalw)/n + 1/(double(n)*double(n));

	printf("%d %f %f %f\n", n, alpha, c.totalw, lw);

	//printf("%f/%d  %f/%d -> %f/%d -> %f\n", tw, tnodes, lw, lnodes, tw - lw, tnodes - lnodes, alpha);

	boolvector out;

	double cw;
	uint32_t cnodes;
	boost::tie(cw, cnodes) = compact(g, c, alpha, out);
	assert(cnodes < n);
	if (cnodes == 0) {
		profile p;
		p.weight = c.totalw;
		p.size = n;
		pl.push_front(p);

		set(g, c, groups, val);
		return val;
	}

	compinfo upper, lower;
	split(c, out, upper, lower);

	val = monotonic(g, lower, upper.totalw, groups, pl, val);
	val = monotonic(g, upper, lw, groups, pl, val + 1);
	return val;
}

void
monotonic(graph & g, intvector & ord, intvector & groups, profilelist & pl)
{
	compinfo c;
	init(g, c);

	ord.resize(c.w.size());
	groups.resize(c.w.size());
	for (uint32_t i = 0; i < ord.size(); i++) ord[i] = i;

	monotonic(g, c, 0, groups, pl, 0);

	typedef std::multimap<uint32_t, uint32_t> intmap;
	intmap queue;

	for (uint32_t i = 0; i < groups.size(); i++)
		queue.insert(std::make_pair(groups[i],i));

	intmap::iterator it = queue.begin();
	for (uint32_t i = 0; i < groups.size(); i++, ++it) {
		groups[i] = it->first;
		ord[i] = it->second;
	}
}


/*
void
monotonic(graph & g, intvector & ord, intvector & groups)
{
	weightvector w;
	vertexweights(g, w);


	ord.resize(w.size());
	groups.resize(w.size());
	for (uint32_t i = 0; i < ord.size(); i++) ord[i] = i;

	double alpha = 0;
	double inc = 1 / (double(w.size()) * double(w.size()));
	uint32_t cur = 0;
	uint32_t csize = w.size() - 1;

	while (true) {
		alpha += inc;

		boolvector out;

		double cw;
		uint32_t cnodes;
		boost::tie(cw, cnodes) = compact(g, w, alpha, out);
		if (cnodes == 0) break;

		alpha = cw / cnodes;
		printf("%f\n\n", alpha);

		for (uint32_t i = 0; i < csize; ) {
			for (; i < csize && out[ord[i]]; i++);
			for (; i <= csize && !out[ord[csize]]; csize--)
				groups[csize] = cur;
			
			if (i < csize) std::swap(ord[i], ord[csize]);
		}

		cur++;
	}
	
}
*/

void
print(FILE *f, profilelist & pl)
{
	uint32_t ind = 1;
	for (profilelist::iterator it = pl.begin(); it != pl.end(); ++it) {
		fprintf(f, "%f %d %d\n", it->weight / it->size, it->size, ind);
		ind += it->size;
	}
}


int
main(int argc, char **argv)
{
	static struct option longopts[] = {
		{"method",          required_argument,  NULL, 'm'},
		{"out",             required_argument,  NULL, 'o'},
		{"in",              required_argument,  NULL, 'i'},
		{"prof",            required_argument,  NULL, 'k'},
		{"help",            no_argument,        NULL, 'h'},
		{"weighted",        no_argument,        NULL, 'w'},
		{ NULL,             0,                  NULL,  0 }
	};

	char *inname = NULL;
	char *outname = NULL;
	char *profname = NULL;
	char method = 'g';
	bool weighted = false;
	bool multiple = false;


	int ch;
	while ((ch = getopt_long(argc, argv, "who:i:p:m:d", longopts, NULL)) != -1) {
		switch (ch) {
			case 'h':
				printf("Usage: %s -i <input file> -o <output file> [-wh] [-p profile]\n", argv[0]);
				printf("  -h    print this help\n");
				printf("  -i    input file\n");
				printf("  -o    output file\n");
				printf("  -p    profile file\n");
				printf("  -d    clean dublicate edges\n");
				printf("  -w    input file has weights\n");
				printf("  -m    method e = exact, g = greedy (default), c = k-core\n");
				return 0;
				break;
			case 'i':
				inname = optarg;
				break;
			case 'o':
				outname = optarg;
				break;
			case 'p':
				profname = optarg;
				break;
			case 'm':
				method = optarg[0];
				break;
			case 'w':
				weighted = true;
				break;
			case 'd':
				multiple = true;
				break;
		}

	}
	if (inname == NULL) {
		printf("Missing input file\n");
		return 1;
	}

	graph *g = read(inname, weighted, multiple);

	weightvector w;
	intvector ord;
	intvector groups;
	profilelist pl;

	switch (method) {
		case 'e':
			monotonic(*g, ord, groups, pl);
			break;
		case 'c':
			order(*g, w, ord);
			core(w, groups, pl);
			break;
		case 'g':
			order(*g, w, ord);
			pava(w, groups, pl);
			break;
		default:
			printf("Unknown method\n");
			return 2;
	}


	if (profname == NULL)
		print(stdout, pl);
	else { 
		FILE *out = fopen(profname, "w");
		print(out, pl);
		fclose(out);
	}

	if (outname == NULL)
		print(stdout, *g, ord, groups);
	else { 
		FILE *out = fopen(outname, "w");
		print(out, *g, ord, groups);
		fclose(out);
	}

	delete g;

	return 0;
}
