#include "graph.h"

#include <boost/graph/boykov_kolmogorov_max_flow.hpp>
#include <boost/graph/push_relabel_max_flow.hpp>
#include <boost/graph/edmonds_karp_max_flow.hpp>
#include <boost/graph/graph_utility.hpp>


typedef boost::adjacency_list_traits <boost::vecS, boost::vecS, boost::directedS> ditraits;
typedef boost::adjacency_list <boost::vecS, boost::vecS, boost::directedS,
	boost::property < boost::vertex_index_t, uint32_t,
	boost::property < boost::vertex_color_t, boost::default_color_type,
	boost::property < boost::vertex_distance_t, uint32_t,
	boost::property < boost::vertex_predecessor_t, ditraits::edge_descriptor > > > >,

	boost::property < boost::edge_weight_t, uint32_t,
	boost::property < boost::edge_capacity_t, double,
	boost::property < boost::edge_residual_capacity_t, double,
	boost::property < boost::edge_reverse_t, ditraits::edge_descriptor > > > > > digraph;


typedef boost::graph_traits<digraph>::edge_iterator dieiterator;
typedef boost::graph_traits<digraph>::out_edge_iterator odieiterator;
typedef boost::graph_traits<digraph>::vertex_iterator diviterator;

typedef boost::graph_traits<digraph>::vertex_descriptor divertex;

typedef boost::property_map<digraph, boost::edge_residual_capacity_t>::type rescapmap;
typedef boost::property_map<digraph, boost::edge_capacity_t>::type capmap;
typedef boost::property_map<digraph, boost::edge_reverse_t>::type revmap;
typedef boost::property_map<digraph, boost::vertex_color_t>::type colormap;
typedef boost::property_map<digraph, boost::vertex_index_t>::type diindexmap;
typedef boost::property_map<digraph, boost::edge_weight_t>::type diweightmap;


void
init(graph & g, compinfo & c)
{
	uint32_t n = boost::num_vertices(g);
	uint32_t m = boost::num_edges(g);

	weightmap wm = boost::get(boost::edge_weight, g);
    indexmap index = boost::get(boost::vertex_index, g);

	c.totalw = vertexweights(g, c.w);

	c.vertices.resize(n);
	c.innerw.resize(n);
	c.edges.resize(m);

 
	viterator vi = boost::vertices(g).first;
	for (uint32_t i = 0; i < n; i++)
		c.vertices[i] = vi[i];

	eiterator ei, ei_end;
	uint32_t j = 0;
	for (boost::tie(ei, ei_end) = boost::edges(g); ei != ei_end; ++ei) {
		c.edges[j].i = index[boost::source(*ei, g)];
		c.edges[j].j = index[boost::target(*ei, g)];
		c.edges[j].w = wm[*ei];
		j++;
	}
}

void
set(graph & g, const compinfo & c, intvector & out, uint32_t v)
{
    indexmap index = boost::get(boost::vertex_index, g);

	for (uint32_t i = 0; i < c.vertices.size(); i++) {
		out[index[c.vertices[i]]] = v;
	}
}

void
split(const compinfo & c, const boolvector & cut, compinfo & pos, compinfo & neg)
{
	uint32_t n = 0;
	for (uint32_t i = 0; i < cut.size(); i++) n += cut[i];

	pos.vertices.resize(n);
	neg.vertices.resize(cut.size() - n);

	pos.w.resize(n);
	neg.w.resize(cut.size() - n);
	pos.innerw.resize(n);
	neg.innerw.resize(cut.size() - n);

	intvector m(cut.size());

	for (uint32_t i = 0, pi = 0, ni = 0; i < cut.size(); i++) {
		if (cut[i]) {
			pos.vertices[pi] = c.vertices[i];
			pos.w[pi] = c.w[i];
			pos.innerw[pi] = c.innerw[i];
			m[i] = pi++;
		}
		else {
			neg.vertices[ni] = c.vertices[i];
			neg.w[ni] = c.w[i];
			neg.innerw[ni] = c.innerw[i];
			m[i] = ni++;
		}
	}

	neg.totalw = c.totalw;


	for (uint32_t ind = 0; ind < c.edges.size(); ind++) {
		uint32_t i = c.edges[ind].i;
		uint32_t j = c.edges[ind].j;
		weight_t w = c.edges[ind].w;
		
		if (cut[i] && cut[j]) {
			compinfo::edge e = {m[i], m[j], w};
			pos.edges.push_back(e);
		}
		if (!cut[i] && !cut[j]) {
			compinfo::edge e = {m[i], m[j], w};
			neg.edges.push_back(e);
		}
		if (cut[i] && !cut[j]) {
			pos.w[m[i]] -= w;
			neg.w[m[j]] -= w;
			neg.innerw[m[j]] += w;
		}
		if (!cut[i] && cut[j]) {
			pos.w[m[j]] -= w;
			neg.w[m[i]] -= w;
			neg.innerw[m[i]] += w;
		}
	}

	pos.totalw = 0;

	for (uint32_t i = 0; i < pos.w.size(); i++) {
		pos.totalw += pos.w[i] / 2 + pos.innerw[i];
	}

	neg.totalw = 0;

	for (uint32_t i = 0; i < neg.w.size(); i++) {
		neg.totalw += neg.w[i] / 2 + neg.innerw[i];
	}
}


std::pair<double, uint32_t>
compact(graph & g, const compinfo & c,  double alpha, boolvector & out)
{
	uint32_t n = c.vertices.size();
	uint32_t m = c.edges.size();

	std::vector<std::pair<uint32_t, uint32_t> > edges(2*m + 4*n);
	std::vector<double> cap(2*m + 4*n);

    //indexmap index = boost::get(boost::vertex_index, g);
	//weightmap wm = boost::get(boost::edge_weight, g);

    //eiterator ei, ei_end;

	uint32_t ind = 0;
	for (compinfo::edgelist::const_iterator ei = c.edges.begin(); ei != c.edges.end(); ++ei) {
		edges[ind] = std::make_pair(ei->i, ei->j);
		edges[ind + 1] = std::make_pair(ei->j, ei->i);
		cap[ind + 1] = cap[ind] = ei->w;
		ind += 2;
	}


	for (uint32_t i = 0; i < n; i++) {
		edges[ind] = std::make_pair(n, i);
		edges[ind + 1] = std::make_pair(i, n);
		cap[ind] = c.w[i] + 2*c.innerw[i];
		ind += 2;
	}

	for (uint32_t i = 0; i < n; i++) {
		edges[ind] = std::make_pair(n + 1, i);
		edges[ind + 1] = std::make_pair(i, n + 1);
		cap[ind + 1] = 2*alpha;
		ind += 2;
	}

	// trick to recover edge order
	intvector edge_indices(edges.size());
	for (uint32_t i = 0; i < edge_indices.size(); i++) edge_indices[i] = i;

	digraph dg(edges.begin(), edges.end(), edge_indices.begin(), n + 2);

	capmap cm = boost::get(boost::edge_capacity, dg);
	revmap rev = boost::get(boost::edge_reverse, dg);

	diweightmap dw = boost::get(boost::edge_weight, dg);
	std::vector<dieiterator> eiters(edges.size());


    dieiterator di, di_end; //= boost::edges(dg).first;

	for (boost::tie(di, di_end) = boost::edges(dg); di != di_end; ++di) {
		cm[*di] = cap[dw[*di]];
		eiters[dw[*di]] = di;
	}
	
	// assumes that edges are in the same order as inserted
	for (uint32_t i = 0; i < eiters.size(); i+=2) {
		rev[*eiters[i]] = *eiters[i + 1];
		rev[*eiters[i + 1]] = *eiters[i];
	}

	diviterator vi = boost::vertices(dg).first;


	double flow = boykov_kolmogorov_max_flow(dg, vi[n], vi[n + 1]);
	//double flow = push_relabel_max_flow(dg, vi[n], vi[n + 1]);
	//double flow = edmonds_karp_max_flow(dg, vi[n], vi[n + 1]);



	out.resize(n);
	colormap com = boost::get(boost::vertex_color, dg);
	uint32_t cnodes = 0;
	for (uint32_t i = 0; i < n; i++) {
		out[i] = com[vi[i]] == boost::black_color;
		cnodes += out[i];
		//if (out[i]) printf("%d\n", i);
	}

	//printf("%f %d %f\n", flow, cnodes, totalw / 2 + alpha*cnodes - flow / 2);

	//printf("%f %f %f\n", flow, totalw, totalw / 2 + alpha*cnodes - flow / 2);

	/*
	boost::print_graph(dg);

	std::cout << "c flow values:" << std::endl;
	diviterator u_iter, u_end;
	odieiterator oi, o_end;
	for (boost::tie(u_iter, u_end) = boost::vertices(dg); u_iter != u_end; ++u_iter)
		for (boost::tie(oi, o_end) = boost::out_edges(*u_iter, dg); oi != o_end; ++oi)
			if (cm[*oi] > 0 && cm[*oi] - rescm[*oi] > 0)
				std::cout << "f " << *u_iter << " " << target(*oi, g) << " " << cm[*oi] << " "
					<< (cm[*oi] - rescm[*oi]) << std::endl;*/


	return std::make_pair(c.totalw + alpha*cnodes - flow / 2, cnodes);
}
