%Y = undiscretize(DY)
%
%Undiscretizes a discretization DY and returns the undiscretized matrix Y.

% Implemented by Markus Ojala in 2010

function Y = undiscretize(DY)

if(isfield(DY,'elem2row')) %sparse
    Y = sparse(double(DY.elem2row),double(DY.elem2col),DY.sortedValues,DY.rows,DY.cols);
else
    Y = DY.sortedValues(DY.pos2elem);
end

switch(DY.dataType)
    case 's' %similar   
        %nothing is needed anymore
                
    case 'n' %nominal or prediscretized data 
        if(isfield(DY,'elem2row')) %sparse
            Y = spfun(@(i) DY.origNominalValues(i), Y);
        else
            Y = DY.origNominalValues(Y);
        end
        
    case {'d','c'} %dissimilar
        if(isfield(DY,'elem2row')) %sparse
            colElems = cellfun(@numel,DY.sortedColValues)'; %number of elements in each column
            
            [i,j,v]=find(Y); %find nonzero elements
            v=v-1; % 2 -> 1, 1 -> 0 
            
            if(DY.dataType == 'c') %1 -> 0.5, 0 -> randomly to 0 or 1
                v = abs(v/2 - round(rand(size(v))));
            end    
            
            for k=1:numel(v)
                if ~isnan(v(k))
                    v(k) = round(v(k).*(colElems(j(k))-1))+1; %columnwise ranks
                    v(k) = DY.sortedColValues{j(k)}(v(k)); %replace columnwise ranks by values
                end
            end
            
            Y = sparse(i,j,v,size(Y,1),size(Y,2));

        else
            if(DY.dataType == 'c') %1 -> 0.5, 0 -> randomly to 0 or 1
                Y = abs(Y/2 - round(rand(size(Y))));
            end
            
            Y = round(bsxfun(@times,Y,cellfun(@numel,DY.sortedColValues)-1))+1; %columnwise ranks            
            for j=1:size(Y,2) %replace columnwise ranks by values
                Y(~isnan(Y(:,j)),j) = DY.sortedColValues{j}(Y(~isnan(Y(:,j)),j)); 
            end
        end    
        
    otherwise
        error('Unknown dataType');
end

end
