%[DY,accRate,realChanges,attempts] = swap(DX,attempts)
%
%Randomizes a matrix X while preserving its row and column value
%distributions approximately. 
%
%Needs swap.jar and trove-2.1.0.jar to work. 
%
%
%Input: 
%
% DX            A discretized matrix containing the desired tolerance ranges for
%               each value. To discretize a matrix X, call first DX = discretize(X).
%               See also the help of 'discretize' to modify the tolerance ranges.
%
% attempts      An optional parameter that gives the number of iterations swaps
%               are tried to perform. The attempts is scaled with the number of 
%               elements. If negative value is given, its absolute value is used. 
%
%
%Output:
%
% DY            The randomized discretization that fulfills the constraints.
% accRate       Average acceptance rate of the swaps.
% realChanges   The number of accepted swaps per the number of elements.
% attemps       The number of attempted swaps per the number of elements
%
%
%Example 1: Produce one randomized sample Y of X:
%
% DX = discretize(X)
% DY = swap(DX)
% Y = undiscretize(X)
%
%
%Example 2: General approach to assess the significance of structural measure S(X):
%
% %Change X and the function S with your data and structural measure
% X = rand(50,5); %some data without any structure
% S = @(A) max(max(triu(corrcoef(A'),1))); %some structural measure, this is the maximum correlation between two rows
%
% origResult = S(X)
%
% k = 99; %the number of randomized samples
% randResults = zeros(k,1);
%
% DX = discretize(X);
% DX0 = swap(DX); %a common starting state, guarantees exchangeability (Besag)
%
% for i=1:k,
%     DY = swap(DX0);
%     Y = undiscretize(DY);
%     randResults(i) = S(Y);
% end
%
% %If the p-value is less than, say 0.05, the result is regarded as significant
% pValue = (sum(randResults >= origResult) + 1) / (k + 1)
%
% %Some statistics of the randomized results:
% meanRandResults = mean(randResults)
% stdRandResults = std(randResults)

% Implemented by Markus Ojala in 2010

function [DY,accRate,realChanges,attempts] = swap(DX,attempts)

if(all(cellfun(@isempty,regexp(javaclasspath,'swap.jar'))))    
    javaaddpath('swap.jar');
end

if(all(cellfun(@isempty,regexp(javaclasspath,'trove-2.1.0.jar'))))
    javaaddpath('trove-2.1.0.jar'); 
end

persistent randGen;
if(isempty(randGen))
    randGen = MersenneTwisterFast;
end

if(~isstruct(DX))
    error('discretized X was not given - call DX = discretize(X) first');    
end

if(nargin<2)
    attempts = [];    
end

%calculate the average size of row and column tolerance ranges for deciding
%whether using tranpose would be beneficial
meanRowNbrs = mean(diff(DX.elem2rowIndexRange,1,2));
meanColNbrs = mean(diff(DX.elem2colIndexRange,1,2));

%perform SwapDiscretized either on the original matrix or transposed matrix
if(meanRowNbrs > meanColNbrs) %transpose would be beneficial    
    [TDY,accRate,realChanges,attempts] = performSwap(transposeDiscretization(DX),attempts,randGen);
    DY = transposeDiscretization(TDY); 
else    
    [DY,accRate,realChanges,attempts] = performSwap(DX,attempts,randGen);
end

end

function TDX = transposeDiscretization(DX)
TDX = DX;
if(isfield(DX,'elem2row')) %sparse
    TDX.elem2row = DX.elem2col;
    TDX.elem2col = DX.elem2row;
    TDX.rows = DX.cols;
    TDX.cols = DX.rows;
else
    TDX.pos2elem = DX.pos2elem';
end
TDX.elem2rowIndexRange = DX.elem2colIndexRange;
TDX.elem2colIndexRange = DX.elem2rowIndexRange;
end

function [DY,accRate,realChanges,attempts] = performSwap(DX,attempts,randGen)

    n = size(DX.elem2colIndexRange,1); %the number of elements

    %calculate default number of attempts if empty
    if(isempty(attempts))
        if(isfield(DX,'elem2row')) %sparse
            sparsity = 1 - n / DX.rows / DX.cols;
            attempts = log(n)/(sparsity^2 + (1-sparsity)^2*mean(diff(DX.elem2colIndexRange,1,2))/n);
        else
            attempts = log(n)*n/mean(diff(DX.elem2colIndexRange,1,2));
        end
    end

    %scale attempts by n, or use the absolute value if given negative value
    if(attempts < 0)
        attempts = -attempts;
    else
        attempts = round(n * attempts);
    end


    %randomize the matrix and parse result
    
    DY = DX; %this guarantees that all extra information is also passed
    
    if(isfield(DX,'elem2row')) %sparse
        S = SwapParse(DX.elem2row-1,DX.elem2col-1,DX.elem2rowIndexRange-1,DX.elem2colIndexRange-1,DX.rows,attempts,randGen);
        DY.elem2row = S.getElem2Row+1;
        DY.elem2col = S.getElem2Col+1;        
    else
        S = Swap(DX.pos2elem-1,DX.elem2rowIndexRange-1,DX.elem2colIndexRange-1,attempts,randGen);
        DY.pos2elem = S.getPos2Elem+1;
    end        
        
    DY.elem2rowIndexRange = S.getElem2RowIndexRange + 1;
    DY.elem2colIndexRange = S.getElem2ColIndexRange + 1;
        
    accRate = S.getAcceptanceRate;
    realChanges = double(S.getRealChanges) / n;
    attempts = attempts / n;
end
