/**
 * SwapConstrained with sparse matrices.
 * 
 * Randomizes a given discretization with allowed tolerance ranges while 
 * preserving the constraints. This implementation should be used with the 
 * provided Matlab interface. See more help in the Matlab methods.
 *
 * Note that the elements are the ordinal numbers, ranks, of the original 
 * values. For each element, we have two index ranges that contain the 
 * allowed fluctuation ranges for the element in the original row and 
 * column. The number of attempts is the number of times the swaps are 
 * tried to perform. 
 *
 * Compared to the SwapConstrained with full matrices, this sparse
 * implementation tracks only the nonzero elements. Additionally, swap are
 * done in two ways to guarantee that elements change also columns, i.e., 
 * by first selecting either a1 or b1. z
 *
 * Implemented by Markus Ojala in 2010
 */

import gnu.trove.TLongIntHashMap; //faster HashMap than Java's built-in

public class SwapParse{
    private long realChanges, attempts, rows;        
    private int[] elem2row, elem2col;
    private int[][] elem2rowIndexRange, elem2colIndexRange;
    private long elemLongPosition(long row, long col) { return row + rows * col; }       
            
    //public functions to get the output
    public long getRealChanges() { return realChanges; } 
    public long getAttempts() { return attempts; } 
    public double getAcceptanceRate() { return realChanges/(double)attempts; }    
    public int[] getElem2Row() { return elem2row; }    
    public int[] getElem2Col() { return elem2col; }  
    public int[][] getElem2RowIndexRange() { return elem2rowIndexRange; } 
    public int[][] getElem2ColIndexRange() { return elem2colIndexRange; }   
    
    /** Assumes that the elements are in increasing order (thus pos2elem gives the current ordering) */    
    public SwapParse(int[] elem2row, int[] elem2col, int[][] elem2rowIndexRange, int[][] elem2colIndexRange, long rows, long attempts, MersenneTwisterFast randGen){
        this.attempts = attempts; this.rows = rows; this.elem2row = elem2row; this.elem2col = elem2col; 
        this.elem2rowIndexRange = elem2rowIndexRange; this.elem2colIndexRange = elem2colIndexRange;
         
        //init data structures
        int elems = elem2row.length;
        TLongIntHashMap pos2elem = new TLongIntHashMap(2*elems); //position (row,col) to element index, position = elemLongPosition(row, col) 
        for(int k = 0; k < elems; k++)
            pos2elem.put(elemLongPosition(elem2row[k], elem2col[k]), k);
        
        /*
         * Tries to perform the following swaps the number of attempts times.
         *
         * a1  b1        b1  a2
         *          ->   
         * b2  a2        a1  b2                 
         *
         */        
        for(long attempt = 0; attempt < attempts; ++attempt) {                
            if(randGen.nextInt(2) == 0){ //choose a1 first                
                //choose a1 uniformly from all elements
                int a1 = randGen.nextInt(elems);
                
                //choose a2 uniformly from all the elements belonging to rowclass of a1
                int a2 = elem2rowIndexRange[a1][0] + randGen.nextInt(elem2rowIndexRange[a1][1] - elem2rowIndexRange[a1][0] + 1);
                
                //get the rows and columns of a1 and a2
                int row1 = elem2row[a1], row2 = elem2row[a2];
                int col1 = elem2col[a1], col2 = elem2col[a2];
                
                //check that rows and columns differ and that a1 belongs to row class of a2
                if(row1 != row2 && col1 != col2 && elem2rowIndexRange[a2][0] <= a1 && a1 <= elem2rowIndexRange[a2][1]){
                    //get B1 and B2, they can be nulls = zero class
                    boolean hasB1 = pos2elem.containsKey(elemLongPosition(row1, col2));
                    boolean hasB2 = pos2elem.containsKey(elemLongPosition(row2, col1));                    
                    
                    // check that b1 and b2 belong to the same column class
                    if(!hasB1 && !hasB2){ //they are in zero class                        
                        performSparseRowSwap(pos2elem, a1, a2, row1, row2, col1, col2); //perform swap
                    }
                    else if(hasB1 && hasB2){ //they are both in nonzero class
                        //get the elements which belong to nonzero classes
                        int b1 = pos2elem.get(elemLongPosition(row1, col2));
                        int b2 = pos2elem.get(elemLongPosition(row2, col1));
                        
                        // check the conditions
                        if(elem2colIndexRange[b1][0] <= b2 && b2 <= elem2colIndexRange[b1][1]
                        && elem2colIndexRange[b2][0] <= b1 && b1 <= elem2colIndexRange[b2][1]){  
                            //perform swap
                            performFullSwap(pos2elem, a1, a2, b1, b2, row1, row2, col1, col2);
                        }
                    }
                }
                
            } else{ //choose b1 first                
                //choose b1 uniformly from all elements
                int b1 = randGen.nextInt(elems);
                
                //choose b2 uniformly from all the elements belonging to colclass of b1
                int b2 = elem2colIndexRange[b1][0] + randGen.nextInt(elem2colIndexRange[b1][1] - elem2colIndexRange[b1][0] + 1);
                
                //get the rows and columns of a1 and a2
                int row1 = elem2row[b1], row2 = elem2row[b2];
                int col2 = elem2col[b1], col1 = elem2col[b2];
                
                //check that rows and columns differ and that a1 belongs to col class of b2
                if(row1 != row2 && col1 != col2 && elem2colIndexRange[b2][0] <= b1 && b1 <= elem2colIndexRange[b2][1]){
                    //get A1 and A2, they can be nulls = zero class                                       
                    boolean hasA1 = pos2elem.containsKey(elemLongPosition(row1, col1));
                    boolean hasA2 = pos2elem.containsKey(elemLongPosition(row2, col2));
                    
                    // check that a1 and a2 belong to the same row class
                    if(!hasA1 && !hasA2){ //they are in zero class
                        performSparseColSwap(pos2elem, b1, b2, row1, row2, col1, col2); //perform swap
                    }
                    else if(hasA1 && hasA2){ //they are both in nonzero class
                        //get the elements which belong to nonzero classes                        
                        int a1 = pos2elem.get(elemLongPosition(row1, col1));
                        int a2 = pos2elem.get(elemLongPosition(row2, col2));                        
                        
                        // check the conditions
                        if(elem2rowIndexRange[a1][0] <= a2 && a2 <= elem2rowIndexRange[a1][1]
                        && elem2rowIndexRange[a2][0] <= a1 && a1 <= elem2rowIndexRange[a2][1]){
                            //perform swap
                            performFullSwap(pos2elem, a1, a2, b1, b2, row1, row2, col1, col2);
                        }                         
                    }
                }
            }
        }
    }
    
    //performs a full swap with four nonzero elements
    private void performFullSwap(TLongIntHashMap pos2elem, int a1, int a2, int b1, int b2, int row1, int row2, int col1, int col2){        
        elem2row[a1] = row2;
        elem2row[a2] = row1;
        elem2col[b1] = col1;
        elem2col[b2] = col2;
        
        pos2elem.put(elemLongPosition(row1, col1), b1);
        pos2elem.put(elemLongPosition(row1, col2), a2);
        pos2elem.put(elemLongPosition(row2, col1), a1);
        pos2elem.put(elemLongPosition(row2, col2), b2);
        
        int[] t = elem2rowIndexRange[a1];
        elem2rowIndexRange[a1] = elem2rowIndexRange[a2];
        elem2rowIndexRange[a2] = t;
        
        t = elem2colIndexRange[b1];
        elem2colIndexRange[b1] = elem2colIndexRange[b2];
        elem2colIndexRange[b2] = t;
        
        realChanges++;        
    }
    
    //performs a swap where b1 and b2 are zeros
    private void performSparseRowSwap(TLongIntHashMap pos2elem, int a1, int a2, int row1, int row2, int col1, int col2){     
        elem2row[a1] = row2;
        elem2row[a2] = row1;
        
        pos2elem.remove(elemLongPosition(row1, col1)); //b1 = zero class
        pos2elem.put(elemLongPosition(row1, col2), a2);
        pos2elem.put(elemLongPosition(row2, col1), a1);
        pos2elem.remove(elemLongPosition(row2, col2)); //b2 = zero class
        
        int[] t = elem2rowIndexRange[a1];
        elem2rowIndexRange[a1] = elem2rowIndexRange[a2];
        elem2rowIndexRange[a2] = t;
        
        realChanges++;
    }
    
    //performs a swap where a1 and a2 are zeros
    private void performSparseColSwap(TLongIntHashMap pos2elem, int b1, int b2, int row1, int row2, int col1, int col2){
        elem2col[b1] = col1;
        elem2col[b2] = col2;
        
        pos2elem.put(elemLongPosition(row1, col1), b1);
        pos2elem.remove(elemLongPosition(row1, col2)); //a2 = zero class
        pos2elem.put(elemLongPosition(row2, col2), b2);
        pos2elem.remove(elemLongPosition(row2, col1)); //a1 = zero class
        
        int[] t = elem2colIndexRange[b1];
        elem2colIndexRange[b1] = elem2colIndexRange[b2];
        elem2colIndexRange[b2] = t;
        
        realChanges++;
    }
}
