/**
 * SwapConstrained with full matrices.
 * 
 * Randomizes a given discretization with allowed tolerance ranges while 
 * preserving the constraints. This implementation should be used with the 
 * provided Matlab interface. See more help in the Matlab methods.
 *
 * Note that the elements are the ordinal numbers, ranks, of the original 
 * values. For each element, we have two index ranges that contain the 
 * allowed fluctuation ranges for the element in the original row and 
 * column. The number of attempts is the number of times the swaps are 
 * tried to perform. 
 *
 * Implemented by Markus Ojala in 2010
 */

public class Swap{
    private long realChanges, attempts;
    private int[][] pos2elem, elem2rowIndexRange, elem2colIndexRange;
            
    //public functions to get the output
    public long getRealChanges() { return realChanges; } 
    public long getAttempts() { return attempts; } 
    public double getAcceptanceRate() { return realChanges/(double)attempts; }      
    public int[][] getPos2Elem() { return pos2elem; } 
    public int[][] getElem2RowIndexRange() { return elem2rowIndexRange; } 
    public int[][] getElem2ColIndexRange() { return elem2colIndexRange; } 
        
    /** Assumes that the elements are in increasing order, i.e., elem 1 is the smallest */    
    public Swap(int[][] pos2elem, int[][] elem2rowIndexRange, int[][] elem2colIndexRange, long attempts, MersenneTwisterFast randGen){
        this.attempts = attempts; this.pos2elem = pos2elem;
        this.elem2rowIndexRange = elem2rowIndexRange; this.elem2colIndexRange = elem2colIndexRange;
        
        //init data structures
        int rows = pos2elem.length, cols = pos2elem[0].length;
        int size = rows * cols;
        
        int[] elem2row = new int[size];
        int[] elem2col = new int[size];
        
        for(int col = 0; col < cols; col++){
            for(int row = 0; row < rows; row++){
                elem2row[pos2elem[row][col]] = row;
                elem2col[pos2elem[row][col]] = col;
            }
        }        
        
        /*
         * Tries to perform the following swaps the number of attempts times.
         *
         * a1  b1        b1  a2
         *          ->   
         * b2  a2        a1  b2                 
         *
         */        
        for(long attempt = 0; attempt < attempts; ++attempt) {
            //choose a1 uniformly from all elements
            int a1 = randGen.nextInt(size);
            
            //choose a2 uniformly from all the elements belonging to rowclass of a1
            int a2 = elem2rowIndexRange[a1][0] + randGen.nextInt(elem2rowIndexRange[a1][1] - elem2rowIndexRange[a1][0]+1);
            
            //get the rows and columns of a1 and a2
            int row1 = elem2row[a1], row2 = elem2row[a2];
            int col1 = elem2col[a1], col2 = elem2col[a2];
            
            //check that rows and columns differ and that a1 belongs to row class of a2
            if(row1 != row2 && col1 != col2 && elem2rowIndexRange[a2][0] <= a1 && a1 <= elem2rowIndexRange[a2][1]){
                //set b1 and b2
                int b1 = pos2elem[row1][col2];
                int b2 = pos2elem[row2][col1];
                
                // check the column constraints
                if(elem2colIndexRange[b1][0] <= b2 && b2 <= elem2colIndexRange[b1][1]
                && elem2colIndexRange[b2][0] <= b1 && b1 <= elem2colIndexRange[b2][1]){
                    
                    //perform swap
                    elem2row[a1] = row2;
                    elem2row[a2] = row1;
                    elem2col[b1] = col1;
                    elem2col[b2] = col2;
                    
                    pos2elem[row1][col1] = b1;
                    pos2elem[row1][col2] = a2;
                    pos2elem[row2][col1] = a1;
                    pos2elem[row2][col2] = b2;
                    
                    int[] t = elem2rowIndexRange[a1];
                    elem2rowIndexRange[a1] = elem2rowIndexRange[a2];
                    elem2rowIndexRange[a2] = t;
                    
                    t = elem2colIndexRange[b1];
                    elem2colIndexRange[b1] = elem2colIndexRange[b2];
                    elem2colIndexRange[b2] = t;
                    
                    realChanges++;
                }
            }
        }
    }
}
