function [Y,accRate,realChanges,attempts,matrixErr,localMod,origMatrix,dfltParams] = randomizeMatrix(X, varargin)
%[Y, accRate, realChanges, attempts, matrixErr, localMod, dfltParams] = randomizeMatrix(X, varargin)
%
%Randomizes the matrix X while preserving some statistics, e.g., the
%distributions of values in rows and columns. Assumes that the original
%matrix is in [0,1] and doesn't contain any NaNs. Use scale and unscale
%methods to transform the matrix into [0,1]. 
%
%
%Before using this method, MatrixRandomization.jar has to be added to java
%path by calling: javaaddpath('MatrixRandomization.jar');
%
%
%Optional parameters for randomizeMatrix are the number of attempts,
%original matrix, MatrixError, LocalModificator and defaultParamsFile. 
%These can be given in any order, e.g.,  
%
%randomizeMatrix(X, attempts, origMatrix, MatrixError, LocalModificator, defaultParamsFile)
%
%
%List of possible MatrixError classes:
% - Metropolis(errorConstant, ErrorMeasure)
% - Greedy
%
%
%List of possible ErrorMeasure classes:
% - ErrCdf
% - ErrCdfHist(bins)
% - ErrMeanStd
% - ErrNull
%
%
%List of possible LocalModificator classes:            Works with Greedy
% - ModAdd(randScale)
% - ModChange
% - ModMask(randScale)
% - ModResample
% - ModRotate
% - ModSwap(rowClasses, colClasses)                           x
%
%
%All parameters can be given in any order and they have also default
%values (applies to all classes). If defaultParamsFile are given then the
%given default values are used from file ['getParameters' defaultsParamsFile].
%
%The number of attemps is multiplied by the number of elements in X. If
%attempts is a negative value then the absolute value of that is used as
%the number of attemps (thus without multiplication). 
%
%The matrix X is the starting matrix. If origMatrix is not given, then X is
%used as the original matrix. The origMatrix is only used with Metropolis,
%and then the error in row and column statistics is calculated against origMatrix. 
%
%The output is a randomized version of X, the acceptance rate, the number
%of real changes and the number of attempts.
%
%Examples:
%
%Y = randomizeMatrix(X);
%
%which by default is the same as "SwapDiscretized":
%
%Y = randomizeMatrix(X,Greedy,ModSwap(floor(2*sqrt(size(X,2))),floor(2*sqrt(size(X,1)))
%
%GeneralMetropolis is called for example by
%
%Y = randomizeMatrix(X,Metropolis(40,ErrCdfHist(100)),ModRotate);
%
%The following uses 200*numel(X) attempts with addition mask
%(randScale=0.1) and ErrMeanStd with default errConst with Metropolis
%
%[Y, accRate, realChanges, attempts, matrixErr, localMod, dfltParams]
% = randomizeMatrix(X, ModMaskAdd, 200, Metropolis(ErrMeanStd));
%
%Note that errConst should be tuned for each data separately! The default
%values of errConst are not good in general.
%
%If MatrixErrorMeasure is given without explicite Metropolis, then it is
%wrapped inside a Metropolis. Thus
%
%Y = randomizeMatrix(X, ErrCdf)
%
%is the same as
%Y = randomizeMatrix(X, Metropolis(ErrCdf));
%
%except that ErrNull is replaced by Greedy. 
%
%Use calcDist(X,Y) and calcErrorCdf(X,Y) to validate the quality of the 
%randomized matrices.

if(min(X(:)) < 0 || max(X(:)) > 1)
    error('You should scale the data into [0,1] - use method ''scale''');    
end

persistent randGen;

try
    if(isempty(randGen))
        randGen = MersenneTwisterFast;
    end
catch
    error('You should first call: javaaddpath(''MatrixRandomization.jar'')');
end
    

[attempts, matrixErr, localMod, origMatrix, dfltParams] = getParameters(X, varargin{:});

%randomize matrix
RM = RandomizeMatrix(X,origMatrix,attempts,localMod,matrixErr,randGen);

%get output values
Y = RM.getRandomizedMatrix();
accRate = RM.getAcceptanceRate;
realChanges = RM.getRealChanges;

end




function [attempts, matrixErr, localMod, origMatrix, dfltParams, dataName] = getParameters(X, varargin)
%Parse input parameters for randomizeMatrix. There can be four input
%parameters: attempts, matrixErr, localMod and dfltParams. They can be
%given in any order and any of them can be left out. The missing parameters
%are replaced with default values. 

attempts = [];
matrixErr = [];
localMod = [];
dataName = 'default';
origMatrix = X;

%"sort" the input parameters
for i = 1:(nargin-1)
    j = varargin{i};
    if(isnumeric(j) && numel(j)==1)
        attempts = j;
    elseif(isnumeric(j) && all(size(j) == size(X)))
        origMatrix = j;
    elseif(isstruct(j))
        disp('This is currently unsupported (default params as struct)');
        %dfltParams = j;
    elseif(ischar(j))
        dataName = j;
    elseif(strcmp(class(j),'Greedy') || strcmp(class(j),'Metropolis'))
        matrixErr = j;
    elseif(strncmp(class(j),'Err',3))
        if(strcmp(class(j),'ErrNull'))
            matrixErr = Greedy();
        else
            matrixErr = Metropolis(j);
        end
    elseif(strncmp(class(j),'Mod',3))
        localMod = j;
        if(strncmp(class(j),'ModPerm',7))
            matrixErr = Greedy();
        end
    else
        disp('Unknown input parameter:'); disp(j);
    end
end

dfltParams = getDefaultParameters(X, matrixErr, localMod, dataName);

%use defaults for missing values

if(isempty(matrixErr))
    matrixErr = eval(dfltParams.matrixErr);
end

if(isempty(localMod))
    localMod = eval(dfltParams.localMod);
end

if(~strcmp(class(matrixErr),'Greedy'))
    if(matrixErr.hasErrConst() == false)
        matrixErr.setErrConst(dfltParams.errConst);
    end
end

errorMeasure = matrixErr.getMatrixErrorMeasure();
if(errorMeasure.hasParams() == false)
    errorMeasure.setParams(dfltParams.matrixErrorMeasure{:});
end

if(localMod.hasParams() == false)
    localMod.setParams(dfltParams.localModParams{:});
end

if(isempty(attempts))
    attempts = dfltParams.attempts;
end

%multiple attempts by numel(X) or take abs
if(attempts > 0)
    attempts = -numel(X)*attempts;
end
attempts = -attempts;

end



function parameters = getDefaultParameters(X, matrixErr, localMod, dataName)
%Default parameters for different combinations of matrix error measure
%and local modificator are defined here. The default parameters can be
%defined for different dataset names. 
%
%The parameters has to be defined for each matError-localMod combination.
%However, if an exact matError or localMod match is not found, the closest
%is used. 
%
%The possible matErrors are 'ErrCdf', 'ErrCdfHist', 'ErrMeanStd', 'ErrNull'
%
%And the possible localMods are 'ModAdd', 'ModChange', 'ModMask',
%'ModResample, 'ModRotate', 'ModSwap'
%
%The parameters that has to be defined are:
%
%Attempts: the number of attempts
%ErrConst: the error constant for Metropolis algorithm, 
%MatrixErr: parameter values for each matrix error measure
%LocalMod: default local modification with each error measure
%LocalModParams: parameter values for each local modification

A = {}; E = {}; L = {}; M = {}; P = {};
[defA, defE, defL, defM, defP] = getParametersDefault(X);

if(nargin>3)
    f = str2func(['getParameters' toupp(dataName)]);
    fp = functions(f);
    if(~strcmp('',fp.file)) %we used 'try' before but this is more strict
        [A,E,L,M,P] = feval(f,X);
    else
        disp('No specific default parameter values exist for given data');
    end      
end

if(isempty(matrixErr))
    if(isempty(localMod) || strcmp(class(localMod),'ModSwap'))        
        matrixErr = Greedy;
    else        
        matrixErr = Metropolis;        
    end
end

errorMeasure = class(matrixErr.getMatrixErrorMeasure());
matrixErr = class(matrixErr);

if(isempty(localMod))
    localMod = getDfltLocalMod(defL, L, matrixErr, errorMeasure);
else
    localMod = class(localMod);
end

at = getDfltParamValue(defA, A, matrixErr, errorMeasure, localMod);
ec = getDfltParamValue(defE, E, matrixErr, errorMeasure, localMod);
me = getDfltParamValue(defM, M, matrixErr, errorMeasure, localMod);
lmp = getDfltParamValue(defP, P, matrixErr, errorMeasure, localMod);

%attempts, errConst, localModification, matrixError, locMod parameters
parameters = struct('attempts', {at}, 'errConst', {ec}, 'matrixErr', {matrixErr}, 'matrixErrorMeasure', {me}, 'localMod', {localMod}, 'localModParams', {lmp});

end


function localMod = getDfltLocalMod(defD, D, matrixErr, errMeas)
try
    D = D(strcmp(D(:,1), D(getClosestName(matrixErr, D(:,1)),1)), 2:3);
    localMod = D{getClosestName(errMeas, D(:,1)), 2};
catch
    defD = defD(strcmp(defD(:,1), defD(getClosestName(matrixErr, defD(:,1)),1)), 2:3);
    localMod = defD{getClosestName(errMeas, defD(:,1)), 2};
end
end

function paramVal = getDfltParamValue(defD, D, matrixErr, errMeas, localMod)
try
    D = D(strcmp(D(:,1), D(getClosestName(matrixErr, D(:,1)),1)), 2:4);
    D = D(strcmp(D(:,1), D(getClosestName(errMeas, D(:,1)),1)), 2:3);
    paramVal = D{getClosestName(localMod, D(:,1)),2};
catch
    defD = defD(strcmp(defD(:,1), defD(getClosestName(matrixErr, defD(:,1)),1)), 2:4);
    defD = defD(strcmp(defD(:,1), defD(getClosestName(errMeas, defD(:,1)),1)), 2:3);
    paramVal = defD{getClosestName(localMod, defD(:,1)),2};
end
end

function ind = getClosestName(name, nameList)
ind = 0; maxlen = -1; 

for k = 1:length(nameList),
    l = length(nameList{k});
    if(length(name) >= l)
        if(l==0 || strncmp(nameList{k}, name, l))
            if(maxlen < l)
                maxlen = l;
                ind = k;
            end
        end   
    end
end

end
      
function upped = toupp(n)
upped = [upper(n(1)) n(2:end)];
end

