function [A, b] = formConstraints(X, Y, G, bC, bK,bS)
% form the constraints (A and b) in SDPP formulation
% X: input data matrix
% Y: response data matrix(vector)
% G: the neighborhood matrix
% bC: classification (1), regression (0)
% bK: supplied X and Y are kernels (1), supplied X and Y are samples (0)
% bS: assume that the neighborhood structure is sparse 
%     = (assume that there is enough memory available)
% assumes that X's kernel matrix has been centralized with H*X*H
% Mika Juuti et al., 2014-09-22

if nargin < 5
    bK = 0;
end
if nargin < 4
    bC = 0;
end
[n, d] = size(X);

A = zeros(d^2,d^2);
b = zeros(d^2,1);

if bK
    DY = diag(Y)*ones(1,n) + ones(n,1)*diag(Y)' - 2*Y;
else
    DY = distSqrdSelf(Y);
end

if bC
    DY(find(DY~=0))= 1;
end

%% Forming constraints while limiting memory consumption
maxMemoryLimit = 1e4;
maxNeighborhoodSize = max(sum(G,2));
if maxNeighborhoodSize*d <= maxMemoryLimit || bS;
    % The constraints for SQLP can be calculated faster if there are
    % at most maxMemoryLimit/d neighbors. The memory limit can be changed.
    fastConstraints = 1; 
else
    fastConstraints = 0;
end

printInterval = 0.05;
nextPrint = 0.00;
if fastConstraints
    for i = 1:n
        neighborsIndex = find(G(i,:));
        [A_increment, b_increment] = addToMatrix(X, i, neighborsIndex, d, G, DY);
        A = A + A_increment;
        b = b + b_increment;
        if (i/n) >= nextPrint
            fprintf('Constraints formed: %u%%.\n',round(nextPrint*100))
            nextPrint = round((nextPrint + printInterval)/0.05)*0.05;
        end
    end
else
    for i = 1:n
        neighborsIndex = find(G(i,:));
        neighborhoodSize = size(neighborsIndex,2);
        numPartitions = ceil(neighborhoodSize*d/maxMemoryLimit);
        partitionSize = floor(neighborhoodSize/numPartitions);
        for j=1:(numPartitions-1)
            subNeighborsIndex = neighborsIndex((1+(j-1)*partitionSize):(j*partitionSize));
            [A_increment, b_increment] = addToMatrix(X, i, subNeighborsIndex, d, G, DY);
            A = A + A_increment;
            b = b + b_increment;
        end
        remainderNeighborsIndex = neighborsIndex((1+partitionSize*(numPartitions-1)):end);
        [A_increment, b_increment] = addToMatrix(X, i, remainderNeighborsIndex, d, G, DY);
        A = A + A_increment;
        b = b + b_increment;
        if (i/n) >= nextPrint
            fprintf('Constraints formed: %u%%.\n',round(nextPrint*100))
            nextPrint = round((nextPrint + printInterval)/0.05)*0.05;
        end
    end
end

A = A/n;
b = -2*b/n;

end

function [A_increment, b_increment] = addToMatrix(X, i, neighborsIndex, d, G, DY)
        neighborhoodSize = size(neighborsIndex,2);
        X_center = X(i,:);
        Tau = (repmat(X_center,neighborhoodSize,1)- X(neighborsIndex,:))';
        
        Tau_interlace = repmat(Tau,d,1);
        r = vec(Tau);
        Tau0 = repmat(r',d,1);
        Tau_stack = reshape(vec(Tau0),d^2,neighborhoodSize);
        
        Tau_squared = Tau_interlace.*Tau_stack;
        D = diag(G(i,neighborsIndex));
        A_increment = Tau_squared*D*Tau_squared';
        b_increment = Tau_squared*(G(i,neighborsIndex)'.*DY(neighborsIndex,i));
end
