%% Demo for SDPP (Regression)
% Run each snippet by pressing CTRL+ENTER (Windows and Emacs defaults)
% Mika Juuti, 2015-02-20

%% Generate a simple regression problem:
% parity problem with 2 relevant dimensions, 6 unrelevant dimensions. 
x = rand(4000,8);
% x = rand(3,2);
x2 = x(:,1).^2+x(:,2).^2;
[~, idx] = sort(x2);
x = x(idx,:);
f = @(x) sin(2*pi*(x(:,1))).*sin(2*pi*(x(:,2)));

y = f(x);
figure; scatter(x(:,1),x(:,2),[],y,'.');

%% Run the SDPP with default parameters
% w: the projection matrix, orthogonalized with SVD
% z: projected data, orthogonalized with SVD
% g: the neighborhood graph %imagesc(g);
% eigs: the eigenvalues of the projected vectors
[w, z, g, eigs]  = sdpp(x,y);

% There eigenvalues for 3 to 8 are all similarly small compared to the 
% first two values, meaning that the first two projection vectors contain 
% most information about the changes in the values in the response space y:
% a strong indicator that x can be represented by the two vectors z(:,1:2).
showEigs(eigs);

% request a 2D projection of the original 8-dimensional space
x_proj = getProjection(x,w,2);

% SDPP extracts the two relevant dimensions in the first two vectors (+ rotation)
figure; scatter(x_proj(:,1),x_proj(:,2),[],y,'.');


%% Run the SDPP with custom parameters
% Specify type of neighborhood, number of neighbors, generate neighborhood.
% increasing the number of neighbors is beneficial up to a limit in the
% parity.
tic;
nStruct.Type = 'knn'; nStruct.k =100; 
[w2, z2, g2, eigs2]  = sdpp(x,y,[],[],'Neighborhood',nStruct, ...
    'Algorithm', 'sqlp');
toc
showEigs(eigs2);
figure; scatter(z2(:,1),z2(:,2),[],y,'.')

%% Run the SDPP with custom parameters
% The neighborhood matrix can also be given as a parameter.
% An initial solution can be calculated beforehand and passed to SDPP.
% Trying different neighborhood sizes is encouraged.
% Both the SQLP and CG solver are able to work with sparse neighbor matrices.
% The sparse parameter is not strictly necessary (there is an automatic check). 

nStruct.Type = 'knn'; nStruct.k = 100; 
options.Neighborhood = nStruct; G = neighborhood(x,options);
[w3, z3, g3, eigs3]  = sdpp(x,y,w,G,'Neighborhood',[],'Algorithm', ...
    'sqlp','Iterations',100,'Sparse',1);

showEigs(eigs3);
figure; scatter(z3(:,1),z3(:,2),[],y,'.')

%% Reduced target dimension
% You can request a reduced target dimension from SDPP beforehand. This can
% speed up the optimization process in very high dimensional data sets,
% when using CG.
% w: the projection matrix, orthogonalized with SVD
% z: projected data, orthogonalized with SVD
% g: the neighborhood graph %imagesc(g);
% eigs: the eigenvalues of the projected vectors

[w, z, g, eigs]  = sdpp(x,y,[],[],'Dimension',4);
showEigs(eigs);
x_proj = getProjection(x,w,2);

% SDPP extracts the two relevant dimensions in the first two vectors (+ rotation)
figure; scatter(x_proj(:,1),x_proj(:,2),[],y,'.');