% Script for analyzing word frequency distribution at different
% granularities in Pride & Prejudice

% Copyright (c) 2015, Jefrey Lijffijt
%
% Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
%
% The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
%
% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OU
T OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

%% Clear workspace
clear;

%% Read data
Text = readBook('Texts/data.txt');

%% Index text
[Words, index] = indexText(Text);

%% Compute frequencies for all words
freq = computeEventFrequencies(index);

%% Compute weibull beta for all words
weibull = computeEventDispersions(index);

%% Select bursty and non-bursty words with frequency >= 100
non_bursty = find(freq >= 100 & weibull(2,:) >= 0.97);
bursty = find(freq >= 100 & weibull(2,:) <= 0.75);

wordlist = [non_bursty bursty];

clear non_bursty bursty;

%% Selected words
Words_selected = Words(wordlist);
weibull_selected = weibull(:,wordlist);

%% Experiment
windowsizes = 1:2000;
ncols = 10000;

k = 3;
nrep = 500;
samplesize = 60;
plusplus = true;

exprep = 50;

n = length(wordlist);
err = zeros(n,exprep);
ws = cell(n,exprep);

tic_total = tic;

for i = 1:n
    word = wordlist(i);
    
    parfor j = 1:exprep
        % Compute trace matrix    
        tic_i = tic;
        Trace = computeFrequencyMatrixSample(index, word, ...
            windowsizes, ncols);
        fprintf('%s\n',['Frequency matrix ' num2str(i) ' computed in ' ...
            num2str(toc(tic_i)) ' seconds.']);

        % Compute Clara++ clustering
        tic_i = tic;
        [ws{i,j},err(i,j)] = clara(Trace,k,nrep,samplesize,plusplus);
        fprintf('%s\n',['k = ' num2str(k) ' clustering computed in ' ...
            num2str(toc(tic_i)) ' seconds.']);
    end
end

fprintf('%s\n',['Experiment finished in ' num2str(toc(tic_total)) ...
    ' seconds.']);

clear n i word Trace tic_total tic_i;

%% Save error
save dami_wordfreq.mat windowsizes ncols k nrep samplesize plusplus ...
    Words_selected weibull_selected err ws;