% Script for running experiments on DNA sequences

% Copyright (c) 2015, Jefrey Lijffijt
%
% Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
%
% The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
%
% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OU
T OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

%% Clear workspace
clear;

%% Prepare data
fprintf('%s\n','Preparing data.');

% Read data
tic;
seq1 = readFastaSequence('Texts/hs_ref_GRCh37.p9_chr1.fa');
fprintf('%s\n','Finished reading file into memory');

% Split sequence into known parts
HS = cleanDNASequence(seq1);
fprintf('%s\n','Finished splitting sequence into fully known parts');

% Index sequence
Index = cell(length(HS),1);

parfor i = 1:length(HS)
    Index{i} = indexDNA(HS{i});
end
fprintf('%s\n','Finished indexing sequence');
fprintf('%s\n',['Process finished in ' num2str(toc) ' seconds.']);

clear seq1 i HS;

%% Save processed data
save hs_seq.mat Index;

%% Compute frequencies for all words
charfreq = zeros(1,4);

for i = 1:length(Index)
    freq = computeEventFrequencies(Index{i});
    charfreq = charfreq + freq;
end

clear i freq;

%% Experiment
windowsizes = 1:10000;
ncols = 1000;
events = {1,2,3,4,[4 1],[3 2]};

k = 3;
nrep = 500;
samplesize = 60;
plusplus = true;

exprep = 1;

n = length(events);
err = zeros(n,exprep);
ws = cell(n,exprep);

tic_total = tic;

for i = 1:n
    event_i = events{i};
    parfor j = 1:exprep
        % Compute trace matrix    
        tic_j = tic;
        Trace = computeFrequencyMatrixSampleCollection(Index, event_i, ...
            windowsizes, ncols);
        fprintf('%s\n',['Frequency matrix ' num2str(i) ' computed in ' ...
            num2str(toc(tic_j)) ' seconds.']);

        % Compute k-means clustering and furthest points
        tic_j = tic;
        [ws{i,j},err(i,j)] = clara(Trace,k,nrep,samplesize,plusplus);
        fprintf('%s\n',['k = ' num2str(k) ' clustering computed in ' ...
            num2str(toc(tic_j)) ' seconds.']);
    end
end

fprintf('%s\n',['Experiment finished in ' num2str(toc(tic_total)) ...
    ' seconds.']);

clear n i Trace tic_total tic_j;

%% Save error
save dami_hs.mat err ws windowsizes ncols events k nrep samplesize plusplus;

%% Clear workspace
clear;

%% Prepare data
fprintf('%s\n','Preparing data.');

% Read data
tic;
seq2 = readFastaSequence('Texts/cfa_ref_CanFam3.1_chr1.fa');
fprintf('%s\n','Finished reading file into memory');

% Split sequence into known parts
CF = cleanDNASequence(seq2);
fprintf('%s\n','Finished splitting sequence into fully known parts');

% Index sequence
Index = cell(length(CF),1);

parfor i = 1:length(CF)
    Index{i} = indexDNA(CF{i});
end
fprintf('%s\n','Finished indexing sequence');
fprintf('%s\n',['Process finished in ' num2str(toc) ' seconds.']);

clear seq2 i CF;

%% Save processed data
save cf_seq.mat Index;

%% Compute frequencies for all words
charfreq = zeros(1,4);

for i = 1:length(Index)
    freq = computeEventFrequencies(Index{i});
    charfreq = charfreq + freq;
end

clear i freq;

%% Experiment
windowsizes = 1:10000;
ncols = 1000;
events = {1,2,3,4,[4 1],[3 2]};

k = 3;
nrep = 500;
samplesize = 60;
plusplus = true;

exprep = 100;

n = length(events);
err = zeros(n,exprep);
ws = cell(n,exprep);

tic_total = tic;

for i = 1:n
    event_i = events{i};
    parfor j = 1:exprep
        % Compute trace matrix    
        tic_j = tic;
        Trace = computeFrequencyMatrixSampleCollection(Index, event_i, ...
            windowsizes, ncols);
        fprintf('%s\n',['Frequency matrix ' num2str(i) ' computed in ' ...
            num2str(toc(tic_j)) ' seconds.']);

        % Compute k-means clustering and furthest points
        tic_j = tic;
        [ws{i,j},err(i,j)] = clara(Trace,k,nrep,samplesize,plusplus);
        fprintf('%s\n',['k = ' num2str(k) ' clustering computed in ' ...
            num2str(toc(tic_j)) ' seconds.']);
    end
end

fprintf('%s\n',['Experiment finished in ' num2str(toc(tic_total)) ...
    ' seconds.']);

clear n i Trace tic_total tic_j;

%% Save error
save dami_cf.mat err ws windowsizes ncols events k nrep samplesize plusplus;