function [Mindex,distsum,label,dist] = clara(X, k, nrep, samplesize, plusplus)
% Clustering LARge Applications algorithm (Kaufman and Rousseeuw 1990)

% Copyright (c) 2015, Jefrey Lijffijt
%
% Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
%
% The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
%
% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OU
T OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

% Init
n = size(X,1);

if(size(X,1) <= samplesize)
    samplesize = size(X,1);
    nrep = 1;
end

distsum = Inf;
label = zeros(n,1);
dist = zeros(n,1);

% Init medoids
Mindex = zeros(k,1); % Allocate space

if plusplus
    Mindex(1) = ceil(rand*n); % Select first medoid uniformly at random
    mindistances = inf(n,1);
    for i = 2:k
        % Update distances and probabilities
        mindistances = min([mindistances,computeDist(X, Mindex(i-1))],[],2);
        prob = cumsum(mindistances);
        % Select new medoid
        if(prob(end) > 0)
            Mindex(i) = find(prob > rand*prob(end),1,'first');
        else % A trivial zero-cost solution has already been found
            notselected = setdiff(1:n,Mindex);
            Mindex(i) = notselected(randi(length(notselected)));
        end
    end
else
    rp = randperm(n);
    Mindex = rp(1:k); % Random initialization of medoids
end

% Repeat nrep times
for i = 1:nrep
    % Draw 40+k objects randomly from data, plus k 'best' medoids so far
    rp = randperm(n);
    sample = union(Mindex, rp(1:(samplesize-k)));
    while(length(sample) < samplesize)
        rp = randperm(n);
        sample = union(Mindex, rp(1:(samplesize-k)));
    end
    
    % Run PAM
    Mindexnew = sample(kmedoidspp(X(sample,:), k, plusplus));
    
    % Compute cost of new solution
    distances = computeDist(X, Mindexnew);
    [distnew, labelnew] = min(distances,[],2);
    distnewsum = sum(distnew);
    
    % Compare cost of new solution to current best, store if improved
    if(distnewsum < distsum)
        distsum = distnewsum;
        Mindex = Mindexnew;
        label = labelnew;
        dist = distnew;
    end
end

end

function distances = computeDist(X, Mindex)
% Compute distance for all points in X to medoids with given indices

distances = zeros(size(X,1), length(Mindex));

for i = 1:length(Mindex)
    distances(:,i) = sum(bsxfun(@minus,X,X(Mindex(i),:)).^2,2);
end

end