% Script for analyzing word frequency distribution at different
% granularities in Pride & Prejudice

% Copyright (c) 2015, Jefrey Lijffijt
%
% Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
%
% The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
%
% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OU
T OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

%% Clear workspace
clear;

%% Load analysis 
load dami_wordfreq.mat;

%% Plot optimal k = 3 window lengths for all words
figure('Name','Word frequency histograms','Units','centimeters',...
    'OuterPosition',[3 3 26 16],'Resize','off',...
    'PaperOrientation','landscape','PaperType','A4','Renderer','painters');

lm = 0.075;
rm = 0.015;
width = 1-lm-rm;
bm = 0.18;
tm = 0.085;
height = 1-bm-tm;

symb = 's^o';
ls = ':';
os = '--';

[~,index] = sort(weibull_selected(2,:),'descend');

axes('Position',[lm bm width height],'FontSize',14.0,'TickDir','out');
hold on;

text(-0.3,max(windowsizes)*1.8,'\beta','Rotation',90,...
    'HorizontalAlignment','center','FontSize',14.0);

for i = 1:size(ws,1)
    data = sort(sort(reshape([ws{index(i),:}],k,[]),'descend'),2);
    
    cutmin = 1+floor(size(data,2)/20);
    cutmax = size(data,2)-floor(size(data,2)/20);
    mid = [floor(size(data,2)/2) ceil(size(data,2)/2)];
    
    for j = 1:k
        if(i <= size(ws,1)/2)
            plot(i,mean(data(j,mid)),symb(j),'MarkerSize',8);
            plot([i-.3 i+.3],[data(j,cutmin) data(j,cutmin)],'-');
            plot([i-.3 i+.3],[data(j,cutmax) data(j,cutmax)],'-');
            switch j
                case 1
                    plot([i i],[max(data(j,cutmin),data(j+1,cutmax)) ...
                        data(j,cutmax)],ls);
                    if(data(j,cutmin) < data(j+1,cutmax))
                        plot([i i],[data(j,cutmin) data(j+1,cutmax)],os);
                    end
                case k
                    plot([i i],[data(j,cutmin) min(data(j,cutmax), ...
                        data(j-1,cutmin))],ls);
                otherwise
                    plot([i i],[max(data(j,cutmin),data(j+1,cutmax)) ...
                        min(data(j,cutmax),data(j-1,cutmin))],...
                        ls);
                    if(data(j,cutmin) < data(j+1,cutmax))
                        plot([i i],[data(j,cutmin) data(j+1,cutmax)],...
                            os);
                    end
            end
        else
            plot(i+1,mean(data(j,mid)),symb(j),'MarkerSize',8);
            plot([i-.3 i+.3]+1,[data(j,cutmin) data(j,cutmin)],'-');
            plot([i-.3 i+.3]+1,[data(j,cutmax) data(j,cutmax)],'-');
            switch j
                case 1
                    plot([i i]+1,[max(data(j,cutmin),data(j+1,cutmax)) ...
                        data(j,cutmax)],ls);
                    if(data(j,cutmin) < data(j+1,cutmax))
                        plot([i i]+1,[data(j,cutmin) data(j+1,cutmax)],os);
                    end
                case k
                    plot([i i]+1,[data(j,cutmin) min(data(j,cutmax), ...
                        data(j-1,cutmin))],ls);
                otherwise
                    plot([i i]+1,[max(data(j,cutmin),data(j+1,cutmax)) ...
                        min(data(j,cutmax),data(j-1,cutmin))],...
                        ls);
                    if(data(j,cutmin) < data(j+1,cutmax))
                        plot([i i]+1,[data(j,cutmin) data(j+1,cutmax)],...
                            os);
                    end
            end
        end
    end
    if(i <= size(ws,1)/2)
        text(i,max(windowsizes)*1.8,num2str(weibull_selected(2,index(i)),...
            '%1.1f'),'Rotation',90,'HorizontalAlignment','center',...
            'FontSize',12.0);
        text(i,0.7,Words_selected{index(i)},'Rotation',90,...
            'HorizontalAlignment','right','FontSize',14.0);
    else
        text(i+1,max(windowsizes)*1.8,num2str(weibull_selected(2,index(i)),...
            '%1.1f'),'Rotation',90,'HorizontalAlignment','center',...
            'FontSize',12.0);
        text(i+1,0.7,Words_selected{index(i)},'Rotation',90,...
            'HorizontalAlignment','right','FontSize',14.0);
    end
end

set(gca,'YScale','log');
set(gca,'XTick',unique([1:floor(size(ws,1)/2) floor(size(ws,1)/2)+2:size(ws,1)+1]));
set(gca,'XTickLabel',{});
ylim([0.9 max(windowsizes)]);
xlim([0.4 size(ws,1)+1.6]);
box on;
ylabel('Window length');

clear lm rm width bm tm height symb index i data j cutmin cutmax mid;

%% Compute some simple statistics to compare two sets
% Select data
data_nonbursty = sort(reshape([ws{1:30,:}],k,[]),'descend');
data_bursty = sort(reshape([ws{31:60,:}],k,[]),'descend');

% Compute statistics
mean_nonbursty = mean(data_nonbursty,2);
mean_bursty = mean(data_bursty,2);
median_nonbursty = median(data_nonbursty,2);
median_bursty = median(data_bursty,2);
stdev_nonbursty = std(data_nonbursty,1,2);
stdev_bursty = std(data_bursty,1,2);

% Print means
fprintf('%s\n',['Means non-bursty: ' num2str(mean_nonbursty(1)) ', ' ...
    num2str(mean_nonbursty(2)) ', ' num2str(mean_nonbursty(3))]);
fprintf('%s\n',['Means bursty: ' num2str(mean_bursty(1)) ', ' ...
    num2str(mean_bursty(2)) ', ' num2str(mean_bursty(3))]);

% Print medians
fprintf('%s\n',['Medians non-bursty: ' num2str(median_nonbursty(1)) ', ' ...
    num2str(median_nonbursty(2)) ', ' num2str(median_nonbursty(3))]);
fprintf('%s\n',['Medians bursty: ' num2str(median_bursty(1)) ', ' ...
    num2str(median_bursty(2)) ', ' num2str(median_bursty(3))]);

% Print standard deviations
fprintf('%s\n',['Standard deviations non-bursty: ' num2str(stdev_nonbursty(1)) ', ' ...
    num2str(stdev_nonbursty(2)) ', ' num2str(stdev_nonbursty(3))]);
fprintf('%s\n',['Standard deviations bursty: ' num2str(stdev_bursty(1)) ', ' ...
    num2str(stdev_bursty(2)) ', ' num2str(stdev_bursty(3))]);

%% Compute all pairwise p-values of Wilcoxon rank-sum test
sig1 = zeros(60,60);
sig2 = zeros(60,60);
sig3 = zeros(60,60);

for i = 1:60
    data_i = sort(reshape([ws{i,:}],k,[]),'descend');
    parfor j = 1:60        
        data_j = sort(reshape([ws{j,:}],k,[]),'descend');
        sig1(i,j) = ranksum(data_i(1,:),data_j(1,:)); %#ok<PFBNS>
        sig2(i,j) = ranksum(data_i(2,:),data_j(2,:));
        sig3(i,j) = ranksum(data_i(3,:),data_j(3,:));
    end
end

clear i data_i j data_j;