#!/usr/bin/python


#TODO: closing conditions
#TODO: make sure exactly one loopvar is true
#TODO: left_i + left_f -> left_lb_f

import os
import sys
import getopt

import nusmv_yacc
from nusmv_modules import *
from expressions import *
from yices import *
import yutil
import clock_util
import ltl
import colors

_DEBUG = False


# 'fair' non-zenoness:
# Require G F reset or G F >= max for each clock
#    where reset := clk = 0
# Furthermore require G F delay



# Limited real encoding:
# Fractional part: real
# Integral part: integer 0..M
#    when an addition would result in a value > M, the integral part will simply
#    be set to M. Note that the fractional part is unnaffected meaning that e.g.
#    M+1 is equal to M under this encoding. As a result, only comparisons with
#    <=, == and >= to values up to M-1 always result in the same truth values as
#    unlimited reals

################################################################################
# Constants

SUPPORTED_SECTIONS = ["INIT", "VAR", "INVARSPEC", "LTLSPEC", "INVAR", "TRANS", "ASSIGN", "URGENT"]

################################################################################
# Warnings

_warning_count = 0
def _WARN(msg):
	global _warning_count
	msg = msg.split("\n")
	print
	print "*** WARNING: %s ***" % msg[0]
	for l in msg[1:]:
		print "***          %s ***" % l
	print
	_warning_count += 1

def _WARN_REPORT():
	if _warning_count > 0:
		print
		print "  %d warning(s) in total" % _warning_count


################################################################################
# BmcException

class BmcException(Exception):
	pass

################################################################################
# Misc model related stuff

def _contains_temporal_operators_cb(expr, _, arg):
	"Callback for traverse_and_replace; takes [False] as argument and replaces "
	"the argument by [True] if there is any temporal operator"
	if isinstance(expr, AstExpression) and len(expr) > 1 and expr[0] in TEMPORAL_LOGIC_OPERATORS:
		arg[0] = True
	
	if arg[0]:
		return expr # Finished
	else:
		return None # Continue traversal

def contains_temporal_operators(expr):
	tops = [False]
	expr.traverse_and_replace(_contains_temporal_operators_cb, tops)
	[tops] = tops
	return tops

################################################################################
# Helpers
_NOTSPEC_INTRO = [Comment(), Comment(";;;; Not spec ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;")]

################################################################################
# Actual BMC

class BMC(object):
	def __init__(self, model, incremental, liveness_method, nonzenoness_method, check_convexity, yices_exec=False):
		#Options
		assert isinstance(incremental, bool)
		self.incremental = incremental
		assert liveness_method == None or isinstance(liveness_method, basestring)
		assert nonzenoness_method == None or isinstance(nonzenoness_method, basestring)
		assert isinstance(check_convexity, bool)
		self.check_convexity = check_convexity
		
		assert isinstance(yices_exec, bool)
		assert not (incremental and yices_exec)
		self.yices_exec = yices_exec
		
		assert isinstance(model, Program)
		model.flatten(["clock"])
		assert model.modules.keys() == ["main"]
		self.module = model.modules["main"]
		
		self.liveness_method = liveness_method
		self.zeno = nonzenoness_method
		self.split_vars = self.liveness_method in ["pure", "limited"]
		self.limited_integer = self.liveness_method == "limited"
		
		#Preprocessing
		self.orgspecs = []
		for spec in self.module.body:
			if isinstance(spec, Specification):
				self.orgspecs.append(copy.deepcopy(spec))
		
		self.module.inline_definitions()
		#self.module.add_types_to_vars()
		
		#Just to make sure nothing goes wrong
		for sec in self.module.body:
			if not sec.type in SUPPORTED_SECTIONS:
				raise BmcException("Unsupported section type: %s" % sec.type)
		
		#Preprocessing
		self._init_clocks_and_variables()
		self.specs = []
		for spec in self.module.body:
			if isinstance(spec, Specification):
				self.specs.append(copy.deepcopy(spec))
		self._init_preprocess_specs()
		
		#Generate variables
		self.delayvar = Variable("delay", BOOLEAN)        # delay step ?
		self.deltavar = Variable("delta", REAL)           # time delta to next step
		self.timeleftvar = Variable("left_lb" if self.split_vars else "left", REAL)         # time left to end of trace (for "one" non-zeno condition only)
		self.nonzenovar = Variable("nonzeno", BOOLEAN)    # there is a delay step after or at current step (for non-zeno lasso paths)
		self.loopvar = Variable("loop", BOOLEAN)          # loop from last state to state i
		self.inloopvar = Variable("in_loop", BOOLEAN)
		self.fairvar = {}
		if self.zeno == 'fair':
			for clk in self.clocks:
				self.fairvar[clk] = Variable(clk.name + "_fair", BOOLEAN)
			self.fairvar[self.delayvar] = Variable("delay_fair", BOOLEAN)
		
		#Integral / fractional parts		
		self.ivar = {}
		self.fvar = {}
		maxmax = 0
		for (clk, mx) in self.clockmax.iteritems():
			mx += 1 #Note: allowing to exceed clock max by one makes things
			        #easier, as in this case it is ensured that the clock value
			        #can never become equal to clockmax, even if we do not alter
			        #the update rules for the fractional part
			maxmax = max(maxmax, mx)
			self.ivar[clk] = Variable("%s_i" % clk.name, Range(0, mx) if self.limited_integer else INTEGER)
			self.fvar[clk] = Variable("%s_f" % clk.name, REAL)
		ivartp = Range(0, maxmax) if self.limited_integer else INTEGER
		self.ivar[self.deltavar] = Variable(self.deltavar.name + "_i", ivartp)
		self.fvar[self.deltavar] = Variable(self.deltavar.name + "_f", REAL)
		
		ivarkeys = self.ivar.keys()
		ivarlst = [self.ivar[c] for c in ivarkeys]
		fvarlst = [self.fvar[c] for c in ivarkeys]
		self.replace_ivar = (ivarkeys, ivarlst)
		self.replace_fvar = (ivarkeys, fvarlst)
		if not self.check_convexity: #Causes exception if one of the variables makes it into the yices file, helps debugging
			for k in ivarkeys:
				k._yices_hint_purely_virtual = self.split_vars
		
		#Generate stuff that is always the same
		self._gen_initial_state()
		self._gen_invariants()
		self._gen_transitions()

	def at_step(self, expr, step):
		"Adds a suffix _step to all variables; obeyes (and removes) next() operators\n"
		"In comments, the strings <<step>> and <<nextstep>> are replaced by step and\n"
		"step + 1, respectively\n"
		"If step is None, no suffixes will be appended but ony preprocessing will be performed"
		
		def _at_step_callback(expr, _, step):
			if isinstance(expr, Variable):
				return Variable("%s_%d" % (expr.name, step), expr.type)
			elif isinstance(expr, AstExpression) and len(expr) == 2 and expr[0] == "next":
				return expr[1].traverse_and_replace(_at_step_callback, step + 1)
			else:
				return None
		
		if isinstance(expr, Expression):
			try:
				if self.split_vars:
					expr = clock_util.split_integral_fractional(expr, self.ivar, self.ivar, self.fvar)
				if step == None:
					return expr
				else:
					return expr.traverse_and_replace(_at_step_callback, step)
			except:
				print "\n!!! %s\n!!! %s\n\n" % (expr.string(False), expr.string(True))
				raise
				
		elif isinstance(expr, (Comment, Check)):
			if step == None:
				return expr
			else:
				return Comment(expr.text.replace("<<step>>", str(step)).replace("<<nextstep>>", str(step+1)))
		elif isinstance(expr, Assertion):
			return Assertion(self.at_step(expr.expr, step))
		else:
			assert isinstance(expr, list)
			return [self.at_step(x, step) for x in expr]
	
	def add_vars(self, a, b, c):
		"Returns constraints that ensure that a = b + c"
		if self.split_vars:
			aivar = a.replace(self.replace_ivar[0], self.replace_ivar[1])
			afvar = a.replace(self.replace_fvar[0], self.replace_fvar[1])
			
			#Find out whether lhv is limited range
			def _range_cb(expr, _, ret):
				if isinstance(expr, Variable) and isinstance(expr.type, Range):
					assert ret[0] == None
					assert expr.type.low == 0
					ret[0] = expr.type.high
			bound = [None]
			aivar.traverse_and_replace(_range_cb, bound)
			[bound] = bound
			
			#Actual addition
			bivar = b.replace(self.replace_ivar[0], self.replace_ivar[1])
			bfvar = b.replace(self.replace_fvar[0], self.replace_fvar[1])
			civar = c.replace(self.replace_ivar[0], self.replace_ivar[1])
			cfvar = c.replace(self.replace_fvar[0], self.replace_fvar[1])
			addone = NamedExpression("fractional overflow", AstExpression(">=", ("+", bfvar, cfvar), "1"))
			iret = [AstExpression("->", addone, ("=", aivar, ("+", ("+", bivar, civar), "1"))),
			        AstExpression("->", ("!", addone), ("=", aivar, ("+", bivar, civar)))]
			fret = [AstExpression("->", addone, ("=", afvar, ("-", ("+", bfvar, cfvar), "1"))),
			        AstExpression("->", ("!", addone), ("=", afvar, ("+", bfvar, cfvar)))]
			
			if bound != None:
				exceeds = AstExpression(">=", ("+", bivar, civar), str(bound))
				for i in xrange(len(iret)):
					iret[i] = AstExpression("->", ("!", exceeds), iret[i])
				iret.append(AstExpression("->", exceeds, ("=", aivar, str(bound))))
			
			return iret + fret
		else:
			return [AstExpression("=", a, ("+", b, c))]
	
	def _init_preprocess_specs(self):
		"Replaces temporal logic specifications with equivalent specifications \n"
		"if possible."
		replaced = False
		for (si, sec) in enumerate(self.specs):
			if isinstance(sec, Specification):
				if sec.type == "INVARSPEC":
					continue
				elif sec.type == "LTLSPEC":
					continue
				self.specs[si] = None
	
	def list_specs(self):
		for i in xrange(len(self.specs)):
			print "%-8d %s" % (i, self.orgspecs[i])
			if self.specs[i] == None:
				print "         (not supported)"
			elif str(self.orgspecs[i]) != str(self.specs[i]):
				print "         (checked as '%s')" % self.specs[i]
			else:
				print "         (supported)"
	
	def speccount(self):
		return len(self.specs)
	
	def _init_clocks_and_variables(self):
		#Computes values:
		self.clocks = {}
		self.clockmax = None
		self.variables = set()
		self.used_types = set()
		
		#Variables
		def _variable_callback(expr, _, (variables, types)):
			if isinstance(expr, Variable):
				if isinstance(expr.type, ModuleType) and expr.type.module == "clock":
					return None
				else:
					variables.add(expr)
					types.add(expr.type)
					return expr
			else:
				return None

		self.module.traverse_and_replace_all(_variable_callback, (self.variables, self.used_types))
		
		#Clocks + resets
		def _clock_callback(expr, _, resets):
			if isinstance(expr, Variable):
				if isinstance(expr.type, ModuleType) and expr.type.module == "clock":
					if len(expr.type.params) != 1:
						raise BmcException("Clock with not exactly one argument: %s" % expr.name)
					if expr.type.process:
						raise BmcException("Clock declared as process: %s" % expr.name)
					ret = Variable(expr.name, Real())
					resets[ret] = expr.type.params
					return ret
				else:
					return None
			else:
				return None
		
		self.module.traverse_and_replace_all(_clock_callback, self.clocks)
		
		#Clock maximums:
		self.clockmax = clock_util.get_clock_maximums(self.clocks, self.module)
	
	############################################################################
	
	def _gen_initial_state(self):
		self.initial = []

		self.initial.append(Comment(";;;; Init ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;"))
		for clk in self.clocks:
			self.initial.append(Assertion(AstExpression("=", clk, "0")))
		
		for init in self.module.get_all("INIT"):
			self.initial.append(Assertion(init.expression))
		
		for asgnments in self.module.get_all("ASSIGN"):
			for asgn in asgnments.content:
				assert asgn.modifier in ["init", "next"]
				if asgn.modifier == "init":
					self.initial.append(Assertion(AstExpression("=", asgn.name, asgn.value)))
		self.initial.append(Comment(""))

	def _gen_invariants(self):
		self.invar = []	
		self.invar.append(Comment(";;;; Invariants <<step>> ;;;;;;;;;;;;;;;;;;;;"))
		
		#Fractional / integral part
		if self.liveness_method == "mixed":
			for c in self.clocks:
				self.invar.append(Assertion(AstExpression(">=", self.fvar[c], "0")))
				self.invar.append(Assertion(AstExpression("<", self.fvar[c], "1")))
				self.invar.append(Assertion(AstExpression("=", ("+", self.ivar[c], self.fvar[c]), c)))
		if self.split_vars:
			self.invar.append(Assertion(AstExpression("<", self.fvar[self.deltavar], "1")))
			self.invar.append(Assertion(AstExpression(">=", self.fvar[self.deltavar], "0")))
		
		#Actual invariants
		if self.check_convexity:
			for invar in self.module.get_all("INVAR"):
				if not yutil.is_convex(invar.expression, self.clocks):
					raise BmcException("Non-convex invariant: %s" % str(invar))
		else:
			_WARN("Assuming convexity of all invariants without any checks")
		
		for invar in self.module.get_all("INVAR"):
			self.invar.append(Assertion(invar.expression))
		self.invar.append(Comment(""))
		
		#Urgency constraints
		for urg in self.module.get_all("URGENT"):
			if urg.expression.variables().intersection(self.clocks):
				raise BmcException("Using clock inside urgency constraint: %s" % str(urg))
			self.invar.append(Assertion(AstExpression("->", urg.expression, ("!", self.delayvar))))
		self.invar.append(Comment(""))

	def _gen_transitions(self):
		self.trans = []
		
		delayvar = self.delayvar
		deltavar = self.deltavar
		
		self.trans.append(Comment(";;;; Delay transition <<step>> -> <<nextstep>> ;;;;;;;;;;;;;;;;;;;;"))
		
		#Actually delay
		self.trans.append(Assertion(AstExpression("->", delayvar, (">", self.deltavar, "0"))))
		
		#Clock differences
		for clk in self.clocks:
			for expr in self.add_vars(AstExpression("next", clk), clk, self.deltavar):
				self.trans.append(Assertion(AstExpression("->", delayvar, expr)))
		
		#Variables
		for var in self.variables:
			self.trans.append(Assertion(AstExpression("->", delayvar, ("=", ("next", var), var))))
		self.trans.append(Comment())
		
		
		self.trans.append(Comment(";;;; Non-delay transitions <<step>> -> <<nextstep>> ;;;;;;;;;;;;;;;;;;;;"))
		
		ndelay = AstExpression("!", self.delayvar)
		
		#Do not delay
		self.trans.append(Assertion(AstExpression("->", ndelay, ("=", self.deltavar, "0"))))
		
		#Reset / do not change otherwise
		for (clk, reset) in self.clocks.iteritems():
			if self.split_vars:
				self.trans.append(Assertion(AstExpression("->",
							("&", ndelay, ("!", reset)),
							("&",
								("=", ("next", self.ivar[clk]), self.ivar[clk]),
								("=", ("next", self.fvar[clk]), self.fvar[clk])
							)
						)))
			else:
				self.trans.append(Assertion(AstExpression("->",
							("&", ndelay, ("!", reset)),
							("=", ("next", clk), clk)
						)))
			self.trans.append(Assertion(AstExpression("->",
						("&", ndelay, reset),
						("next", ("=", clk, "0"))
					)))
		
		#TRANS and ASSIGN constraints
		expressions = []
		for trans in self.module.get_all("TRANS"):
			expressions.append(AstExpression("->", ndelay, trans.expression))
		
		for asgnments in self.module.get_all("ASSIGN"):
			for asgn in asgnments.content:
				assert asgn.modifier in ["init", "next"]
				if asgn.modifier == "next":
					expressions.append(AstExpression("=", ("next", asgn.name), asgn.value))

		for expr in expressions:
			self.trans.append(Assertion(expr))
		
		self.trans.append(Comment())
		
	############################################################################
	def _gen_loop_conditions(self, indices, step):
		rp = []
		rt = [] if self.incremental else rp
		
		#Non-zenoness
		if self.zeno == "one":
			#Compute time left to end of trace
			for i in indices:
				if i > 0:
					if self.split_vars:
						expr = AstExpression("=", self.timeleftvar, ("?:",
										(">", self.ivar[self.deltavar], "0"),
										"1",
										("+", ("next", self.timeleftvar), self.fvar[self.deltavar])))
					else:
						expr = AstExpression("=", self.timeleftvar, ("+", ("next", self.timeleftvar), self.deltavar))
					rp.append(self.at_step(Assertion(expr), i-1))
			rt.append(Assertion(self.at_step(AstExpression("=", self.timeleftvar, "0"), step)))
		elif self.zeno == "fair":
			#Clocks
			for (clk, mx) in self.clockmax.iteritems():
				fv = self.fairvar[clk]
				for i in indices:
					if i > 0:
						rp.append(self.at_step(Assertion(AstExpression("=", fv, ("|", ("next", fv), ("=", clk, "0")))), i-1))
				rt.append(self.at_step(Assertion(AstExpression("=", fv, ("|", ("=", clk, "0"), (">", clk, str(mx))))), step))
			
			#delay
			for i in indices:
				if i > 0:
					fv = self.fairvar[self.delayvar]
					rp.append(self.at_step(Assertion(AstExpression("=", fv, ("|", ("next", fv), self.delayvar))), i-1))
			rt.append(self.at_step(Assertion(AstExpression("!", self.fairvar[self.delayvar])), step))
		elif self.zeno == "lasso":
			for i in indices:
				if i > 0:
					expr = AstExpression("=", self.nonzenovar, ("|", ("next", self.nonzenovar), self.delayvar))
					rp.append(Assertion(self.at_step(expr, i-1)))
			rt.append(Assertion(self.at_step(AstExpression("=", self.nonzenovar, "false"), step)))
		
		#For all steps
		for index in xrange(step):
			#Variables
			lexp = []
			for v in self.variables:
				lexp.append(AstExpression("=", self.at_step(v, index), self.at_step(v, step)))
			rt.append(Assertion(AstExpression("->", self.at_step(self.loopvar, index), reduce_to_ast_commutative("&", lexp))))
		
			#Clocks
			lexp = []
			
			if self.liveness_method == "lasso":
				for c in self.clocks:
					lexp.append(AstExpression("=", self.at_step(c, index), self.at_step(c, step)))
				
				if self.zeno == "lasso":
					lexp.append(self.at_step(self.nonzenovar, index))
				else:
					assert self.zeno == None
			elif self.liveness_method in ["mixed", "pure", "limited"]:
				assert self.zeno in ["one", "fair"]
				for (ci, c) in enumerate(self.clocks):
					ivar = self.ivar[c]
					fvar = self.fvar[c]
				
					ieq = AstExpression("=", self.at_step(ivar, index), self.at_step(ivar, step))
					f0eq = AstExpression("=", fvar, "0")
					f0eq = AstExpression("<->", self.at_step(f0eq, index), self.at_step(f0eq, step))
				
					notexceeded = NamedExpression("not exceeded %s" % c.name, AstExpression("<=", ivar, str(self.clockmax[c])))
					ieq = AstExpression("&",
								("<->", self.at_step(notexceeded, index), self.at_step(notexceeded, step)),
								("->", self.at_step(notexceeded, index), ieq)
							)
					f0eq = AstExpression("->", self.at_step(notexceeded, index), f0eq)
					lexp.append(ieq)
					lexp.append(f0eq)
				
					for d in self.clocks:
						if c != d:
							dfvar = self.fvar[d]
							divar = self.ivar[d]
							comp = AstExpression("<=", fvar, dfvar)
							comp = AstExpression("<->", self.at_step(comp, index), self.at_step(comp, step))
							dnotexceeded = NamedExpression("not exceeded %s" % d.name, AstExpression("<=", divar, str(self.clockmax[d])))
							comp = AstExpression("->",
										("&", notexceeded, dnotexceeded),
										comp
									)
							lexp.append(comp)
				
				if self.zeno == "one":
					lexp.append(self.at_step(AstExpression(">=", self.timeleftvar, "1"), index))
				elif self.zeno == "fair":
					lexp.append(self.at_step(reduce_to_ast_commutative("&", self.fairvar.values()), index))
				else:
					assert False
			else:
				assert False
			
			lv = self.at_step(self.loopvar, index)
			for l in lexp:
				rt.append(Assertion(AstExpression("->", lv, l)))
			
		#At most one loopvar true
		for i in indices: #inloopvar is true if loop goes back to previous or earlier state
			if i == 0:
				rp.append(Assertion(self.at_step(AstExpression("<->", self.inloopvar, "FALSE"), i)))
			else:
				rp.append(Assertion(self.at_step(AstExpression("<->", ("next", self.inloopvar), ("|", self.inloopvar, self.loopvar)), i-1)))
			rp.append(Assertion(self.at_step(AstExpression("->", self.inloopvar, ("!", self.loopvar)), i)))
				
		return rp, (rt if self.incremental else None)
	
	def _gen_not_spec(self, step):
		"Returns: permanent part, temporary part"
		indices = [step] if self.incremental else range(step+1)
		
		rp = copy.copy(_NOTSPEC_INTRO)
		rt = [] if self.incremental else rp
		
		#Loop variables
		if self.looping:
			grp, grt = self._gen_loop_conditions(indices, step)
			rp.extend(grp)
			if grt != None:
				rt.extend(grt)
		
		#Actual not spec
		if 0 in indices:
			for s in self.spec_initial:
				rp.append(self.at_step(s, 0))
		
		for i in indices:
			for s in self.spec_all:
				rp.append(self.at_step(s, i))
			if i > 0:
				for s in self.spec_intermediate:
					rp.append(self.at_step(s, i-1))
		
		for s in self.spec_final:
			s = s[0](step, s[1])
			if s != None:
				rt.append(s)
		
		return rp, (rt if self.incremental else None)
	
	def _prepare_spec(self, spec):
		assert isinstance(spec, Specification)
		assert spec.type in ["INVARSPEC", "LTLSPEC"]
		
		self.looping = spec.type == "LTLSPEC"
		self.spec_initial = []
		self.spec_comments = []
		self.spec_intermediate = []
		self.spec_final = []
		self.spec_all = []
		if spec.type == "INVARSPEC":
			#Invariant
			violationvar = Variable("violates", BOOLEAN)
			self.spec_all.append(AstExpression("->", violationvar, ("!", spec.expression)))
			self.spec_final.append((lambda k, _ : Assertion(reduce_to_ast_commutative("|", [self.at_step(violationvar, i) for i in xrange(k+1)])), None))
		elif spec.type == "LTLSPEC":
			#LTL
			#Positive normal form
			def dbg_cb(expr, _, __):
				if not isinstance(expr, Expression):
					raise Exception(str(expr))
			spec.expression.traverse_and_replace(dbg_cb, None)
			print spec.expression
			prop = ltl.positive_normal_form(AstExpression("!", spec.expression))
			
			#Search subexpressions
			specvar = Variable("spec", BOOLEAN)
	
			def subexp_cb(expr, _, (out, nextindex)):
				if isinstance(expr, AstExpression) and expr[0] in ltl.REDUCED_LTLOPS:
					if not expr in out:
						out[expr] = Variable("subexp_%d" % nextindex[0], BOOLEAN)
						nextindex[0] = nextindex[0] + 1
			dct = {prop:specvar}
			prop.traverse_and_replace(subexp_cb, (dct, [0]))
	
			#Replace LTL-subexpressions
			def subexp_repl_cb(expr, _, (dct, toplevel)):
				if (not toplevel[0]) and expr in dct:
					return dct[expr]
				toplevel[0] = False
			
			subexp = []
			for (ex, nm) in dct.iteritems():
				subexp.append((nm, ex.traverse_and_replace(subexp_repl_cb, (dct, [True]))))
			
			#Encode
			self.spec_initial.append(specvar)
			self.spec_final.append((lambda k, _ : Assertion(self.at_step(self.inloopvar, k)), None))
			
			for (name, exp) in subexp:
				self.spec_comments.append("%s corresponds to %s" % (name, exp))
				assert isinstance(name, Variable)
				nameb = Variable(name.name + "_b", name.type)
				
				if isinstance(exp, AstExpression) and exp[0] in ltl.REDUCED_LTLOPS:
					if exp[0] == "X":
						assert len(exp) == 2
						self.spec_intermediate.append(AstExpression("<->", name, ("next", exp[1])))
						self.spec_final.append((lambda k, (name, exp) : Assertion(AstExpression("<->", self.at_step(name, k), reduce_to_ast_commutative("|", [
										self.at_step(AstExpression("&", self.loopvar, ("next", exp[1])), i)
									for i in xrange(k)]) if k > 0 else AstExpression("FALSE"))),
								(name, exp)))
					elif exp[0] == "U" or exp[0] == "V":
						assert len(exp) == 3
						if exp[0] == "U":
							outer_op = "|"
							inner_op = "&"
						else:
							outer_op = "&"
							inner_op = "|"
						self.spec_intermediate.append(AstExpression("<->", name, (outer_op, exp[2], (inner_op, exp[1], ("next", name)))))
						self.spec_intermediate.append(AstExpression("<->", nameb, (outer_op, exp[2], (inner_op, exp[1], ("next", nameb)))))
						self.spec_final.append((lambda k, (name, nameb, outer_op, inner_op, exp) : Assertion(AstExpression("<->",
									self.at_step(name, k),
									(outer_op,
										self.at_step(exp[2], k),
										(inner_op,
											self.at_step(exp[1], k),
											reduce_to_ast_commutative("|", [
													self.at_step(AstExpression("&", self.loopvar, ("next", nameb)), i)
													for i in xrange(k)]) if k > 0 else AstExpression("FALSE"))))),
							(name, nameb, outer_op, inner_op, exp)))
						self.spec_final.append((lambda k, (nameb, exp) : Assertion(self.at_step(AstExpression("<->", nameb, exp[2]), k)), (nameb, exp)))
					else:
						assert False, exp
				else:
					#No LTL inside
					self.spec_all.append(AstExpression("<->", name, exp))
		else:
			assert False, spec
		
		self.spec_comments = map(Comment, self.spec_comments)
		self.spec_initial = map(Assertion, self.spec_initial)
		self.spec_intermediate = map(Assertion, self.spec_intermediate)
		self.spec_all = map(Assertion, self.spec_all)
	
	def _del_spec_related_things(self):
		del self.looping
		del self.spec_comments
		del self.spec_initial
		del self.spec_intermediate
		del self.spec_final
		del self.spec_all
	
	############################################################################
	
	def _check(self, yi, yf):
		"Checks yf using yices yi. Empties yf."
				
		#Check property
		yf.append(Check())
		ret = yi.check(yf)
		print colors.red("does not hold") if ret else colors.green("maybe holds")
		del yf[0:len(yf)]
		yf.append(Comment())
		yf.append(Comment())
		
		return not ret
	
	def bmc(self, num, bound, yicesfile, skip=False):
		"Checks a single specification\n"
		"spec:  the specification\n"
		"bound: Current bound\n"
		"skip:  Only print message that the specification was skipped, return None\n"
		"yicesfile:   Write commands for yices to this file\n"
		"Returns: retval, bound\n"
		"where retval is True if the specification (maybe) holds, False if it\n"
		"does not and None  if it was skipped or is not supported\n"
		"and bound is the length of the counterexample if the one was found\n"
		"Note that bound is always equal to the specified bound unless\n"
		"incremental bmc is used"
		assert isinstance(num, (int, long))
		assert isinstance(bound, (int, long))
		assert bound >= 0
		
		spec = self.specs[num]
		
		print "Property %d" % num, (colors.bold if skip else colors.bright_blue)(str(self.orgspecs[num])),
		
		if skip:
			print "was", colors.bold("skipped")
			return None, None
		elif spec == None:
			print "is", colors.yellow("not supported"), "(yet)"
			return None, None
		elif spec.type == "LTLSPEC" and self.liveness_method == None:
			print "was skipped since", colors.yellow("LTL"), "property checking was", colors.yellow("disabled")
			return None, None
		else:
			assert spec.type in ["INVARSPEC", "LTLSPEC"]
			if self.incremental:
				print
			sys.stdout.flush()
			
			#Prepare
			
			self._prepare_spec(spec)
			
			yi = Yices(yicesfile, self.yices_exec)
			yi.go()
			yi.declare_types(self.used_types)
			
			yf = []
			
			if self.spec_comments:
				yf.append(Comment("Specification remarks:"))
				yf += self.spec_comments
			
			#Go
			for iteration in xrange(bound + 1):
				yf += self.at_step(self.invar, iteration)
				if iteration == 0:
					yf += self.at_step(self.initial, iteration)
				else:
					yf += self.at_step(self.trans, iteration - 1)
				
				if self.incremental:
					print "  After %d steps ... " % iteration,
					p, t = self._gen_not_spec(iteration)
					yf += p
					yf.append(Check())
					yf.append(Push())
					yf += t
					holds = self._check(yi, yf)
					yf.append(Pop())
					if not holds:
						break
			
			#Check (unless incremental)
			if not self.incremental:
				p, t = self._gen_not_spec(bound)
				assert t == None
				yf += p
				holds = self._check(yi, yf)
			
			yi.close()
			
			#Clean up
			self._del_spec_related_things()
		
			return holds, iteration if not holds else None

################################################################################
# IO

RETVAL_WRONG_ANSWER = 3
RETVAL_USAGE = 2

def __goodbye(errmsg=None, retval=RETVAL_USAGE, mention_minus_h=True):
	if errmsg != None:
		sys.stderr.write(errmsg)
		sys.stderr.write("\n")
	if mention_minus_h:
		sys.stderr.write("Use -h for more information.\n")
	sys.exit(retval)

def __help():
	print "Usage: %s [<options>] <input-file> <k>\n" % os.path.split(sys.argv[0])[-1] +\
'''Options:
   -h or --help   : Show this screen and exit
   --lp           : List properties but do not check anything
   -n <num>       : Only check property number <num> (0-based in flattened file,
      use --lp to find  out ids.
   -y <file>      : Dump Yices input to <file>
   -i             : Incremental
   -l <condition> : Checking of liveness properties (at the moment only
       LTL properties in the form of "F <prop-logic> are supported)
       Possible arguments -- simple strategies:
          lasso   : Only consider lasso-shaped counter-examples
          mixed   : Consider non-lasso-shaped counter-examples; use mixed
             integer-real expressions
          pure    : Consider non-lasso-shaped counter-examples; use no mixed
             type expressions
          limited : Like pure, but in addition only uses limited-range integers.
   -z <encoding>  : Non-zenoness encoding. Potential values depend on the used
      liveness encoding.
      - If liveness encoding 'none' is used, then '-z' may not be used.
      - If liveness encoding 'mixed', 'pure' or 'limited' is used, then possible
         encodings are:
            one:  Require at least one time unit to pass along the loop
            fair: Require each clock is either reset along the loop or
               exceeds it's maximum infinitely often. Note that this setting may
               in rare cases lead to shorter counter-examples.
      - If liveness encoding 'lasso' is used, then possible encodings are:
            on:  Enforce non-zenoness (this is the default)
            off: Allow zeno paths
   --nochecks     : Assume convexity without checking
   --answers <ans>: Used to specify which properties hold and which ones do not.
      If the BMC result contradicts the specified result, the program exits with
      code %d. The correct answers are specified in a simple string, which
      contains 'T' for a property that holds, 'FXX' for a property that does not
      hold where XX is the minimum bound required for showing that the property
      does not hold and 'U' for a property that is not supported by the current
      settings.
   --yexec        : Use yices executable instead of API interface. Incompatible
                    with incremental BMC.
      
      
                              *** KNOWN ISSUES ***
- In LTL specifications, the '=' and '!=' operators should not contain
  LTL subexpressions. Use '<->' and 'xor' for boolean expressions instead.
''' % 	RETVAL_WRONG_ANSWER
	sys.exit(0)

#TODO: --answers --lp

LIVENESS_TYPES = {
None    : [None],
"lasso"   : ["lasso", None],
"mixed"   : ["one", "fair"],
"mixed"   : ["one", "fair"],
"pure"    : ["one", "fair"],
"limited" : ["one", "fair"]
}

if __name__ == "__main__":
	#Command line args
	args = sys.argv[1:]
	try:
		options, args = getopt.gnu_getopt(args, "hn:iy:l:z:", ["nochecks", "lp", "answers=", "help", "yexec"])
	except getopt.GetoptError, e:
		__goodbye("Error: %s" % e.msg)
	
	#Check arguments
	check_property = None
	incremental = False
	yicesfile = None
	liveness = None
	nonzenoness = None
	nochecks = False
	list_properties = False
	answers = None
	yices_exec = False
	for arg, val in options:
		if arg == "-h" or arg == "--help":
			__help()
		elif arg == "--lp":
			list_properties = True
		elif arg == "-n":
			check_property = int(val)
		elif arg == "-i":
			incremental = True
		elif arg == "-y":
			yicesfile = val
		elif arg == "-l":
			if not val in LIVENESS_TYPES:
				sys.stderr.write("Invalid liveness property argument: %s\n" % val)
				sys.stderr.write("Valid arguments: %s\n" % ", ".join(x for x in sorted(LIVENESS_TYPES) if x != None))
				__goodbye()
			if val == "none":
				liveness = None
			else:
				liveness = val
		elif arg == "-z":
			nonzenoness = val
		elif arg == "--nochecks":
			nochecks = True
		elif arg == "--answers":
			answers = []
			i = 0
			while i < len(val):
				val = val.lower()
				if val[i] == "u":
					answers.append(None)
				elif val[i] == "t":
					answers.append(True)
				elif val[i] == "f":
					i += 1
					start = i
					while i < len(val) and val[i] in "0123456789":
						i += 1
					if start >= i:
						__goodbye("Invalid answers argument: empty bound")
					answers.append((False, int(val[start:i])))
					continue
				else:
					__goodbye("Invalid character in answers string: %s" % repr(val[i]))
				i += 1
		elif arg == "--yexec":
			yices_exec = True
		else:
			assert False, (arg, val)
	
	if incremental and yices_exec:
		__goodbye("Incremental BMC not possible in when using yices in executable-mode.")
	
	#Replace certain values
	if liveness == "lasso":
		if nonzenoness in ["on", None]:
			nonzenoness = "lasso"
		elif nonzenoness == "off":
			nonzenoness = None
	
	if not nonzenoness in LIVENESS_TYPES[liveness]:
		if nonzenoness == None:
			__goodbye("Liveness encoding %s requires selecting a non-zenoness encoding (use '-z')\n" % liveness)
		else:
			__goodbye("Invalid argument for liveness encoding %s: %s\n" % (repr(liveness), repr(nonzenoness)))
	
	#Read file
	if not len(args) in [1, 2] or (len(args) == 1 and not list_properties):
		__goodbye("Invalid argument number")
	
	filename = args[0]
	inf = open(filename, "r")
	model = nusmv_yacc.parser.parse(inf.read())
	inf.close()
	
	#Do BMC
	b = BMC(model, incremental, liveness, nonzenoness, not nochecks, yices_exec)
	if answers != None and len(answers) != b.speccount():
		__goodbye("Specified %d expected results for %d specifications" % (len(answers), b.speccount()))
	
	print "File:", filename
	if list_properties:
		b.list_specs()
	else:
		bound = int(args[1])
		for i in xrange(b.speccount()):
			skip = (not check_property in [None, i])
			holds, rbound = b.bmc(i, bound, yicesfile, skip)
			if not skip and answers != None:
				expected = answers[i]
				if isinstance(expected, tuple):
					assert not expected[0]
					if expected[1] > bound:
						expected = True
					elif not incremental:
						expected = (False, bound)
				result = (holds, rbound) if holds == False else holds
				if expected != result or type(expected) != type(result):
					__goodbye("Specified result did not match actual result for property %d" % i, RETVAL_WRONG_ANSWER, False)
	print
	
	#Finish
	_WARN_REPORT()

