#include <limits.h>
#include <unistd.h> // This is used only for debugging to get sleep()

#include "dassert.h"
#include "unfold.h"
#include "monset.h"
#include "sarray.h"
#include "darray.h"
#include "eventq.h"
#include "alloc.h"
#include "dset.h"

#include "smodels.h"
#include "api.h"
#include "atomrule.h"


Unfolder::Unfolder(PTNet *in_net, BOOL in_debugging, BOOL in_do_ltl)
{
  net = in_net;
  debugging = in_debugging;
  do_ltl = in_do_ltl;

  smodels = new Smodels();
  api = new Api(&(smodels->program));
  look = TRUE;
  jump = FALSE;


  ASSERT(net->getNumPlaces() > 0);

}

Unfolder::~Unfolder()
{

  delete(api);
  delete(smodels);

}



#define DEBUG_MODEL_CONSISTENCY
//#define DEBUG_MODEL_CONSISTENCY_PRINT
//#define DEBUG_MODEL_PRINT

#define MAX(a,b) (((a) >= (b)) ? (a) : (b))

Prefix *Unfolder::Unfold(void)
{
  unsigned long i;
  unsigned long j;
  unsigned long k;
  unsigned long l;
  unsigned long t;
  unsigned long t2;
  unsigned long place;
  unsigned long transition;
  unsigned long event;
  unsigned long extended_event;
  Arc *ac;
  Arc *ac2;
  Arc *ac3;
  DARRAY *da;
  SARRAY *sa;
  BOOL dead;
  BOOL done;
  BOOL success;
  BOOL cut;
  BOOL can_stutter;
  BOOL postset_empty;
  Place **places;
  Transition **transitions;
  Condition **conditions;

  unsigned long *preset_counts;
  unsigned long numplaces;
  unsigned long numtransitions;

  unsigned long numenabled;
  unsigned long numpost;
  unsigned long gen_cond_alloc;
  unsigned long gen_event_alloc;
  unsigned long num_gen_cond;
  unsigned long num_gen_event;
  unsigned long max_trans_preset;
  unsigned long max_trans_postset;
  unsigned long model_count;
  unsigned long program_id;
#ifdef DEBUG_MODEL_CONSISTENCY
  MONSET *program_enabled_trans;
  MONSET *program_enabled_places;
  unsigned long program_gen_cond;
  BOOL first;
#endif /* DEBUG_MODEL_CONSISTENCY */

  MONSET *enabled_trans;
  MONSET *enabled_places;
  MONSET *postset_events;
  SARRAY *tr_preset;
  SARRAY *tr_postset;
  DARRAY **place_projections;
  DARRAY **place_posttrans;
  SARRAY **event_post_conds;
  DARRAY **condition_post_events;
  DSET *cut_off_events;
  DSET *cut_off_post_conds;
  unsigned long *condition_pre_events;

  Prefix *prefix;
  EventQueue *event_queue;

  Atom *atom;
  Atom *bottom_atom;
  Atom **transition_atoms;
  Atom **place_atoms;
  Atom **gen_cond_atoms;
  Atom **gen_event_atoms;
  char *buf;

#ifdef DEBUG_MODEL_CONSISTENCY
  program_enabled_trans = (MONSET *)NULL;
  program_enabled_places = (MONSET *)NULL;
#endif /* DEBUG_MODEL_CONSISTENCY */

  numplaces = net->getNumPlaces();
  numtransitions = net->getNumTransitions();
  places = net->getPlaces();
  transitions = net->getTransitions();

  ASSERT(numplaces > 0);

  // Count how many preset places of a transition do not have tokens

  preset_counts = (unsigned long *)xmalloc(numtransitions *
					   sizeof(unsigned long));

  for(i = 0; i < numtransitions; i++) {
    preset_counts[i] = 0;
  }

  // Keep track on which places have conditions

  place_projections = (DARRAY **)xmalloc(numplaces *
					 sizeof(DARRAY *));

  // Keep track on which places have postransitions

  place_posttrans = (DARRAY **)xmalloc(numplaces *
				       sizeof(DARRAY *));


  num_gen_cond = 0;
  for(i = 0; i < numplaces; i++) {

    ASSERT((places[i]->initial_marking == 0) ||
	   (places[i]->initial_marking == 1));

    place_projections[i] = (DARRAY *)NULL;
    place_posttrans[i] = (DARRAY *)NULL;

    if(places[i]->initial_marking == 0) {

      // This place is not in the initial marking,
      // therefore all its' postset transitions are
      // missing at least one token in their preset

      for(ac = places[i]->postset;
	  ac != NULL;
	  ac = ac->nexttransition) {

	preset_counts[ac->transition->id -1]++;

      }

    } else {

      num_gen_cond++;

    }

  }


  // The initial marking should be non-empty

  ASSERT(num_gen_cond > 0);

  // This is what is needed to LTL model checking

  max_trans_preset = numplaces;
  max_trans_postset = numplaces;
  ASSERT(max_trans_preset > 0);
  ASSERT(max_trans_postset > 0);


  // Allocate atoms for generated events, round up to the next
  // power of two.

  for(gen_cond_alloc = 1;
      gen_cond_alloc < num_gen_cond;
      gen_cond_alloc *= 2) {
    ;
  }

  enabled_trans = CreateMonSet(numtransitions);
  enabled_places = CreateMonSet(numplaces);
  tr_preset = CreateSArray(max_trans_preset);
  tr_postset = CreateSArray(max_trans_postset);

  transition_atoms = (Atom **)xmalloc(numtransitions * sizeof(Atom *));
  place_atoms = (Atom **)xmalloc(numplaces * sizeof(Atom *));

  // A buffer for creating atom names
  
  buf = (char *)xmalloc(256 * sizeof(char));


  // Create atoms for the initial marking conditions

  gen_cond_atoms = (Atom **)xmalloc(gen_cond_alloc * sizeof(Atom *));
  condition_post_events = (DARRAY **)xmalloc(gen_cond_alloc * sizeof(DARRAY *));
  condition_pre_events = (unsigned long *)xmalloc(gen_cond_alloc * sizeof(unsigned long));

  // Create the prefix to be returned

  prefix = new Prefix(net, do_ltl);
  event_queue = new EventQueue(prefix, do_ltl);
  prefix->SetEventQueue(event_queue);


  for(num_gen_cond = 0, i = 0; i < numplaces; i++) {

    if(places[i]->initial_marking != 0) {

      // This place is in the initial marking

      AddtoMonSet(enabled_places, i);

      place_projections[i] = CreateDArray();
      AppendDArray(place_projections[i], num_gen_cond);

      place_posttrans[i] = CreateDArray();

      prefix->AddInitialCondition(i+1);


      // This condition doesn't have a pre-event

      condition_pre_events[num_gen_cond] = ULONG_MAX;

      // A dynamic array to hold the condition postset events

      condition_post_events[num_gen_cond] = CreateDArray();

      num_gen_cond++;

    }

  }


  gen_event_alloc = 4; // Any small number >= 1 will do
  gen_event_atoms = (Atom **)xmalloc(gen_event_alloc * sizeof(Atom *));
  event_post_conds = (SARRAY **)xmalloc(gen_event_alloc * sizeof(SARRAY *));
  num_gen_event = 0; // No events on the first round
  postset_events = CreateMonSet(gen_event_alloc);



  // All those transitions with 0 preset counts are
  // potentially enabled, and must be considered.

  for(i = 0; i < numtransitions; i++) {

    if(preset_counts[i] == 0) {
      AddtoMonSet(enabled_trans, i);

      for(ac = transitions[i]->preset;
	  ac != NULL;
	  ac = ac->nextplace) {

	// Append this transition to the posttransitions of all its preplaces

	AppendDArray(place_posttrans[ac->place->id -1], i);

      }

    }

  }


  debugging = TRUE;

  // Debugging stuff: Check that the net is not
  // initially dead, and fail if so.
  // TODO: Handle also this case nicely.

  success = FALSE;

  dead = EmptyMonSet(enabled_trans);
  if(dead != FALSE) {
    goto dead_exit;
  }

  cut_off_events = CreateDSet();
  cut_off_post_conds = CreateDSet();
  model_count = 0;
  program_id = 0;
  done = FALSE;

  while((done == FALSE) &&
	(success == FALSE)) {


    // Create the bottom atom for constraint rule heads

    bottom_atom = api->new_atom();
    if(debugging != FALSE) {
      api->set_name(bottom_atom, "bottom");
    }


    // Create the place atoms

    for(i = 0, numenabled = GetMonSetSize(enabled_places); i < numenabled; i++) {

      t = GetMonSetItem(enabled_places, i);

      atom = api->new_atom();
      place_atoms[t] = atom;
      if(debugging != FALSE) {
	sprintf(buf, "p%lu", (t+1));
	api->set_name(atom, buf);
      }

    }
  

    // Create the transition atoms
    
    for(i = 0, numenabled = GetMonSetSize(enabled_trans); i < numenabled; i++) {

      t = GetMonSetItem(enabled_trans, i);

      atom = api->new_atom();
      transition_atoms[t] = atom;
      if(debugging != FALSE) {
	sprintf(buf, "t%lu", (t+1));
	api->set_name(atom, buf);
      }

    }


    // Create the condition atoms
    // (No more cut-off redundancy.)

    
    for(i = 0; i < num_gen_cond; i++) {

      gen_cond_atoms[i] = NULL;
      if(InDSet(cut_off_post_conds, i)) {
	continue;
      }
      
      atom = api->new_atom();
      gen_cond_atoms[i] = atom;
      if(debugging != FALSE) {
	sprintf(buf, "gb%lu", (i+1));
	api->set_name(atom, buf);
      }

    }


    // Create the event atoms
    // (No more cut-off redundancy.)
    
    for(i = 0; i < num_gen_event; i++) {

      gen_event_atoms[i] = NULL;
      if(InDSet(cut_off_events, i)) {
	continue;
      }

      atom = api->new_atom();
      gen_event_atoms[i] = atom;
      if(debugging != FALSE) {
	sprintf(buf, "ge%lu", (i+1));
	api->set_name(atom, buf);
      }

    }

    // Remember which events this program can handle

    program_id = num_gen_event;
#ifdef DEBUG_MODEL_CONSISTENCY
    program_gen_cond = num_gen_cond;
    program_enabled_trans = CopyMonSet(enabled_trans);
    program_enabled_places = CopyMonSet(enabled_places);
#endif /* DEBUG_MODEL_CONSISTENCY */


    // Add rules to choose exactly one transition

    // First choose a transition
    //
    // Create also all the needed atoms here

    api->begin_rule(CHOICERULE);
    for(i = 0, numenabled = GetMonSetSize(enabled_trans); i < numenabled; i++) {

      t = GetMonSetItem(enabled_trans, i);

      api->add_head(transition_atoms[t]);

    }
    api->end_rule();


    // Then require that at least one transition exists

    api->begin_rule(BASICRULE);
    api->add_head(bottom_atom);
    for(i = 0; i < numenabled; i++) {
      api->add_body(transition_atoms[GetMonSetItem(enabled_trans, i)], FALSE);
    }
    api->end_rule();

    // Then require that less than two transition exist

    if(numenabled > 1) {
      api->begin_rule(CONSTRAINTRULE);
      api->add_head(bottom_atom);
      api->set_atleast_body(2);
      for(i = 0; i < numenabled; i++) {
	api->add_body(transition_atoms[GetMonSetItem(enabled_trans, i)], TRUE);
      }
      api->end_rule();
    }



    // Add rules to require that for the place atoms at least one of the
    // postset transitions is preset.

    for(i = 0, numenabled = GetMonSetSize(enabled_places); i < numenabled; i++) {
      
      t = GetMonSetItem(enabled_places, i);
    
      da = place_posttrans[t];
      numpost = GetDArraySize(da); 

      // Check that the postset transitions really exist
      // (for initial condition places this might not be the case)

      if(numpost > 0) {

	if(numpost > 1) { 	

	  // Handle the general case

	  api->begin_rule(CONSTRAINTRULE);
	  api->add_head(place_atoms[t]);
	  api->set_atleast_body(1);

	} else {

	  // A special case: A simple implication suffices

	  api->begin_rule(BASICRULE);
	  api->add_head(place_atoms[t]);

	}

	  
	for(j = 0; j < numpost; j++) {

	  t2 = GetDArrayItem(da, j);
	  api->add_body(transition_atoms[t2], TRUE);

	}

	api->end_rule();

      }

    }


    // Add rules to choose exactly one condition for each place

    
    for(i = 0, numenabled = GetMonSetSize(enabled_places); i < numenabled; i++) {

      t = GetMonSetItem(enabled_places, i);
      
      da = place_posttrans[t];
      numpost = GetDArraySize(da); 

      // Check that the postset transitions really exist
      // (for initial condition places this might not be the case)

      if(numpost > 0) {

	// First chooce one condition if the place is present

	da = place_projections[t];

	ASSERT(GetDArraySize(da) > 0);
	numpost = GetDArraySize(da);

	for(j = 0, k = numpost; j < k; j++) {

	  t2 = GetDArrayItem(da, j);

	  if(InDSet(cut_off_post_conds, t2)) {

	    numpost--;
	  
	  }
	
	}

	if(numpost == 0) {

	  // This place has no projections, it must be FALSE!
	  
	  api->set_compute(place_atoms[t], FALSE);
	  continue;
	  
	}
	
	if(numpost > 1) {
	  
	  // The projection set contains more than one item, choosing one is needed

	  api->begin_rule(CHOICERULE);
	  for(j = 0, numpost = GetDArraySize(da); j < numpost; j++) {

	    t2 = GetDArrayItem(da, j);

	    if(InDSet(cut_off_post_conds, t2)) {
	      continue;
	    }

	    api->add_head(gen_cond_atoms[t2]);

	  }
	  api->add_body(place_atoms[t], TRUE);
	  api->end_rule();

	  // Then choose at least one if the place is preset

	  api->begin_rule(BASICRULE);
	  api->add_head(bottom_atom);
	  api->add_body(place_atoms[t], TRUE);
	  for(j = 0, numpost = GetDArraySize(da); j < numpost; j++) {

	    t2 = GetDArrayItem(da, j);

	    if(InDSet(cut_off_post_conds, t2)) {
	      continue;
	    }

	    api->add_body(gen_cond_atoms[t2], FALSE);

	  }
	  api->end_rule();


	} else {

	  // Simple implication suffices in this case

	  // Find the only non-cutoff projection condition

	  t2 = 0;
	  numpost = GetDArraySize(da);
	  for(j = 0, k = numpost; j < k; j++) {

	    t2 = GetDArrayItem(da, j);

	    if(InDSet(cut_off_post_conds, t2) == FALSE) {
	      break;
	    }

	    if((j+1) == k) {
	      ASSERT(FALSE);
	    }
	    
	  }
	      
	  if(InDSet(cut_off_post_conds, t2)) {
	    ASSERT(FALSE);
	  }

	  api->begin_rule(BASICRULE);
	  api->add_head(gen_cond_atoms[t2]);
	  api->add_body(place_atoms[t], TRUE);
	  api->end_rule();
	  

	}
	
      }
      
    }



    // Add rules to handle events
    // (No more cut-off redundancy.)
    
    for(i = 0; i < num_gen_event; i++) {

      if(InDSet(cut_off_events, i)) {
	continue;
      }
      
      api->begin_rule(CONSTRAINTRULE);
      api->add_head(gen_event_atoms[i]);
      api->set_atleast_body(1);

      // An event must exist if one of its' post conditions exist

      ClearMonSet(postset_events);
      sa = event_post_conds[i];
      for(j = 0, t = GetSArraySize(sa); j < t; j++) {
	t2 = GetSArrayItem(sa, j);
	api->add_body(gen_cond_atoms[t2], TRUE);

	// Get the postset event to a set, and thus throw away duplicates

	da = condition_post_events[t2];
	for(l = 0, k = GetDArraySize(da); l < k; l++) {

	  AddtoMonSet(postset_events, GetDArrayItem(da, l));

	}

      }

      // An event must exist if one of its' post events exist
      // All events have been collected to the postset_events.

      for(j = 0, t = GetMonSetSize(postset_events); j < t; j++) {
	t2 = GetMonSetItem(postset_events, j);
	if(gen_event_atoms[t2] != NULL) {
	  // Non-cutoff
	  api->add_body(gen_event_atoms[t2], TRUE);
	}
      }

      api->end_rule();

    }


    // Add rules to handle conditions
    // (No more cut-off redundancy.)
    
    for(i = 0; i < num_gen_cond; i++) {

      da = condition_post_events[i];
      numpost = GetDArraySize(da);

      if(numpost > 0) {

	ASSERT(InDSet(cut_off_post_conds, i) == FALSE);
		
	// Only something to do for conditions which have a postset

	// Simple case analysis on the rule gives:
	// (i)  If condition exist, none of its postset events should exist
	// (ii) If the condition does not exist, then only one of its
	//      postset events can exist

	api->begin_rule(CONSTRAINTRULE);
	api->set_atleast_body(2);
	api->add_head(bottom_atom);
	api->add_body(gen_cond_atoms[i], TRUE);

	for(j = 0; j < numpost; j++) {
	  t = GetDArrayItem(da, j);
	  if(gen_event_atoms[t] != NULL) {
	    // Non-cutoff
	    api->add_body(gen_event_atoms[t], TRUE);
	  }
	}

	api->end_rule();

      }

    }


    // Set the compute statement

    api->set_compute(bottom_atom, FALSE);


    // Get the event to extend if we are not the initial
    // program.

    extended_event = ULONG_MAX;

    if(program_id != 0) {

      // Force the smallest non-cut-off event to true

      while(event_queue->Empty() == FALSE) {

	event = event_queue->RemoveSmallest();
	prefix->AssignIDs(event);
	cut = prefix->CheckCutOff(event);

	if(cut == FALSE) {

	  // We found the next event to extend

	  extended_event = event;
	  break;

	} else {

	  // A cut-off found. Skip it!

	  AddtoDSet(cut_off_events, event);

	  if(prefix->IsSuccess(event)) {

	    // We found a successful cut-off, exit

	    success = TRUE;
	    break;

	  }

	  // Mark the cut-off postset conditions
	  
	  sa = event_post_conds[event];
	  for(i = 0, j = GetSArraySize(sa); i < j; i++) {
	    t = GetSArrayItem(sa, i);
	    AddtoDSet(cut_off_post_conds, t);
	  }
  
	}

      }

      if(extended_event == ULONG_MAX) {

	// We ran out of things to expand, we're done!

	done = TRUE;
	break;
      }

      api->set_compute(gen_event_atoms[extended_event],
		       TRUE);

      // Force all the generated but still in queue events to false

      for(i = 0, j = event_queue->GetSize(); i < j; i++) {
	api->set_compute(gen_event_atoms[event_queue->GetItem(i)],
			 FALSE);
      }


      // Set the compute statement for cut-off events
      // to force all cut-off events to be false

      for(i = 0, j = GetDSetSize(cut_off_events); i < j; i++) {
	t = GetDSetItem(cut_off_events, i);

	if(gen_event_atoms[t] != NULL) {

	  // Most cut-off atoms are NULLs, but some might still exist
	  // (Just found to be cut-offs by the code above.)
	  
	  api->set_compute(gen_event_atoms[t],
			   FALSE);
	}

      }

    }


    // We're all set!

    api->done();


#if 0
    smodels->program.print();
#endif
#if 0
    fprintf(stdout, "----- INIT -----\n");
#endif
    smodels->init();
#if 0
    smodels->program.print();
#endif

    // This loop is exited when the program can't be used any more

    while(TRUE) {

      // This loop is exited when no more models exist with this compute statement

      while(TRUE) {

	if(!smodels->model(look, jump)) {
	  break;
	}

	model_count++;

#ifdef DEBUG_MODEL_CONSISTENCY

#ifdef DEBUG_MODEL_CONSISTENCY_PRINT
	fprintf(stdout, "Model: ");
#endif /* DEBUG_MODEL_CONSISTENCY_PRINT */

	first = TRUE;
	j = ULONG_MAX;
	for(i = 0, numenabled = GetMonSetSize(program_enabled_trans), t2 = 0;
	    i < numenabled;
	    i++) {

	  t = GetMonSetItem(program_enabled_trans, i);
	  if(transition_atoms[t]->Bpos) {

	    // Transition t is in the model

#ifdef DEBUG_MODEL_CONSISTENCY_PRINT
	    fprintf(stdout, ((first != FALSE) ? ("t%lu") : (", t%lu")), (t+1));
#endif /* DEBUG_MODEL_CONSISTENCY_PRINT */
	    first = FALSE;
	    t2++;
	    j = t;
	  }

	}

	if(t2 != 1) {
#ifdef DEBUG_MODEL_CONSISTENCY_PRINT
	  fprintf(stdout, "\n");
	  fflush(stdout);
#endif /* DEBUG_MODEL_CONSISTENCY_PRINT */
	  ASSERT(FALSE);
	}
	ASSERT(j != ULONG_MAX);

	// Then scan the transition preset and calculate the size of it

	for(t2 = 0, ac = transitions[j]->preset;
	    ac != NULL;
	    ac = ac->nextplace) {

	  t2++;

	}

	for(i = 0, j = 0, numenabled = program_gen_cond;
	    i < numenabled;
	    i++) {

	  if((gen_cond_atoms[i] != NULL) &&
	     (gen_cond_atoms[i]->Bpos)) {

	    // Condition b is in the model

#ifdef DEBUG_MODEL_CONSISTENCY_PRINT
	    fprintf(stdout, ", gb%lu", (i+1));
#endif /* DEBUG_MODEL_CONSISTENCY_PRINT */
	    j++;

	  }

	}

	if(t2 != j) {
#ifdef DEBUG_MODEL_CONSISTENCY_PRINT
	  fprintf(stdout, "\n");
	  fflush(stdout);
#endif /* DEBUG_MODEL_CONSISTENCY_PRINT */
	  ASSERT(FALSE);
	}

	for(i = 0, j = 0, numenabled = GetMonSetSize(program_enabled_places);
	    i < numenabled;
	    i++) {

	  t = GetMonSetItem(program_enabled_places, i);
	  if(place_atoms[t]->Bpos) {

	    // Transition t is in the model

#ifdef DEBUG_MODEL_CONSISTENCY_PRINT
	    fprintf(stdout, ", p%lu", (t+1));
#endif /* DEBUG_MODEL_CONSISTENCY_PRINT */
	    j++;

	  }

	}

	if(t2 != j) {
#ifdef DEBUG_MODEL_CONSISTENCY_PRINT
	  fprintf(stdout, "\n");
	  fflush(stdout);
#endif /* DEBUG_MODEL_CONSISTENCY_PRINT */
	  ASSERT(FALSE);
	}


#ifdef DEBUG_MODEL_CONSISTENCY_PRINT
	fprintf(stdout, "\n");
	fflush(stdout);
#endif /* DEBUG_MODEL_CONSISTENCY_PRINT */


	// Check that all cut-off events are false

	for(i = 0, j = GetDSetSize(cut_off_events);
	    i < j;
	    i++) {

	  t = GetDSetItem(cut_off_events, i);

	  if((gen_event_atoms[t] != NULL) &&
	     (gen_event_atoms[t]->Bpos)) {

	    ASSERT(FALSE);

	  }

	}


#endif /* DEBUG_MODEL_CONSISTENCY */



	t = ULONG_MAX;
	for(i = 0, numenabled = GetMonSetSize(enabled_trans); i < numenabled; i++) {

	  t = GetMonSetItem(enabled_trans, i);

	  if(transition_atoms[t]->Bpos) {

	    // Transition t is in the model

	    break;

	  }
     
	}

	ASSERT(t != ULONG_MAX); // At least one of the transitions should exist


	// Print out the model, transition first

#ifdef DEBUG_MODEL_PRINT
	fprintf(stdout, "Model %lu: t%lu", model_count, (t+1));
#endif /* DEBUG_MODEL_PRINT */
	ClearSArray(tr_preset);
	ClearSArray(tr_postset);

	// Then scan the transition preset

	for(ac = transitions[t]->preset;
	    ac != NULL;
	    ac = ac->nextplace) {

	  // Scan the conditions projecting to that place

	  dead = TRUE;
	  da = place_projections[ac->place->id -1];
	  for(i = 0, j = GetDArraySize(da);
	      i < j;
	      i++) {

	    t2 = GetDArrayItem(da, i);
	
	    if((gen_cond_atoms[t2] != NULL) &
	       (gen_cond_atoms[t2]->Bpos)) {

	      // Found the condition which was marked true

	      AppendSArray(tr_preset, t2);

	      // Note: conditions stored as indexes (>= 1),
	      // transitions as ids (>= 0)!

#ifdef DEBUG_MODEL_PRINT
	      fprintf(stdout, ", gb%lu" , (t2+1));
#endif /* DEBUG_MODEL_PRINT */
	      dead = FALSE;
	      break;
	    }

	  }

	  // Each preset should have at least one condition

	  ASSERT(dead == FALSE);

	}

#ifdef DEBUG_MODEL_PRINT
	fprintf(stdout, "\n");
#endif /* DEBUG_MODEL_PRINT */


	// Handle the stuff which doesn't need smodels

	if((do_ltl == FALSE) ||
	   (net->isLTransition(transitions[t]->id) == FALSE)) {

	  // Easy case first, this is not an L-transition

	  can_stutter = FALSE;

	  // Create postset conditions

	  for(ac = transitions[t]->postset;
	      ac != NULL;
	      ac = ac->nextplace) {

	    place = ac->place->id;

	    // This might be a new enabled place, add it to enabled places

	    if(AddtoMonSet(enabled_places, (place -1)) == FALSE) {

	      // This place was really new, investigate its' postset
	      // for new transitions

	      place_projections[place -1] = CreateDArray();
	      place_posttrans[place -1] = CreateDArray();

	      // Some transitions in the postset of this place might become enabled

	      for(ac2 = places[place -1]->postset;
		  ac2 != NULL;
		  ac2 = ac2->nexttransition) {

		transition = ac2->transition->id; 

		if(preset_counts[transition -1] == 0) {

		  // The transition is already enabled, nothing to do

		  continue;
		}
		
		if(preset_counts[transition -1] == 1) {

		  // The transition just became enabled, add it to enabled transitions

		  AddtoMonSet(enabled_trans, (transition -1));	    

		  // Add this transition to the postset transitions of
		  // all its preset places

		  for(ac3 = transitions[transition -1]->preset;
		      ac3 != NULL;
		      ac3 = ac3->nextplace) {

		    AppendDArray(place_posttrans[ac3->place->id -1], (transition -1));

		  }
	    
		}

		// Decrement the preset counts

		preset_counts[transition -1]--;

	      }

	    }


	    // OK, now let's deal with adding stuff into the prefix

	    t2 = prefix->AddCondition(place);

	    // Store them to save some work in AddEvent

	    AppendSArray(tr_postset, t2);

	    AppendDArray(place_projections[place -1], t2);

	    if(num_gen_cond == gen_cond_alloc) {
	      gen_cond_alloc *= 2;
	      gen_cond_atoms = (Atom **)xrealloc(gen_cond_atoms,
						 (gen_cond_alloc *
						  sizeof(Atom*)));

	      condition_post_events = (DARRAY **)xrealloc(condition_post_events,
							  (gen_cond_alloc *
							   sizeof(DARRAY *)));
	      condition_pre_events = (unsigned long *)xrealloc(condition_pre_events,
							       (gen_cond_alloc *
								sizeof(unsigned long)));
	    }
	    num_gen_cond++;

	    ASSERT((num_gen_cond -1) == t2);


	    // Record the pre-event of this condition

	    condition_pre_events[t2] = num_gen_event;


	    // Allocate the data structure for post events of this condition

	    condition_post_events[t2] = CreateDArray();

	  }


	} else {

	  // This is an L-event with a dynamic pre-and postsets
	  // First expand the preset with all the remaining conditions
	  // in the cut of the (preset) local configuration

	  can_stutter = prefix->ExpandLPreset(t, tr_preset);

	  if(can_stutter == FALSE) {

	    // There is no livelock behind this event, just forget it!

	    continue;

	  }

	  // We first check that the postset is non-empty before we
	  // add the event to the prefix

	  postset_empty = TRUE;
	  conditions = prefix->getConditions();
	  for(i = 0, j = GetSArraySize(tr_preset); i < j; i++) {

	    place = conditions[GetSArrayItem(tr_preset, i)]->pl_id;

	    if(net->isInvisiblePlace(place) != FALSE) {

	      // This place is in the preset of invisible transitions,
	      // we will add it to the postset

	      postset_empty = FALSE;
	      break;

	    }
	    
	  }

	  if(postset_empty != FALSE) {

	    // Discard L-events with empty postsets

	    continue;

	  }
	  
	  // OK, the postset is a (non-empty) filtered version of the preset,
	  // obtain that one

	  for(i = 0, j = GetSArraySize(tr_preset); i < j; i++) {

	    // Add only invisible places

	    conditions = prefix->getConditions();
	    place = conditions[GetSArrayItem(tr_preset, i)]->pl_id;
	    conditions = NULL; // Do not cache this, as it might change

	    if(net->isInvisiblePlace(place) == FALSE) {

	      // This place is not in the preset of invisible transitions,
	      // we won't add it to the postset

	      continue;

	    }


	    // This might be a new enabled place, add it to enabled places

	    if(AddtoMonSet(enabled_places, (place -1)) == FALSE) {

	      // This place was really new, investigate its' postset
	      // for new transitions

	      place_projections[place -1] = CreateDArray();
	      place_posttrans[place -1] = CreateDArray();

	      // Some transitions in the postset of this place might become enabled

	      for(ac2 = places[place -1]->postset;
		  ac2 != NULL;
		  ac2 = ac2->nexttransition) {

		transition = ac2->transition->id; 

		if(preset_counts[transition -1] == 0) {

		  // The transition is already enabled, nothing to do

		  continue;
		}
		
		if(preset_counts[transition -1] == 1) {

		  // The transition just became enabled, add it to enabled transitions

		  AddtoMonSet(enabled_trans, (transition -1));	    

		  // Add this transition to the postset transitions of
		  // all its preset places

		  for(ac3 = transitions[transition -1]->preset;
		      ac3 != NULL;
		      ac3 = ac3->nextplace) {

		    AppendDArray(place_posttrans[ac3->place->id -1], (transition -1));

		  }
	    
		}

		// Decrement the preset counts

		preset_counts[transition -1]--;

	      }

	    }

	    // OK, now let's deal with adding stuff into the prefix

	    t2 = prefix->AddCondition(place);

	    // Store them to save some work in AddEvent

	    AppendSArray(tr_postset, t2);

	    AppendDArray(place_projections[place -1], t2);

	    if(num_gen_cond == gen_cond_alloc) {
	      gen_cond_alloc *= 2;
	      gen_cond_atoms = (Atom **)xrealloc(gen_cond_atoms,
						 (gen_cond_alloc *
						  sizeof(Atom*)));

	      condition_post_events = (DARRAY **)xrealloc(condition_post_events,
							  (gen_cond_alloc *
							   sizeof(DARRAY *)));
	      condition_pre_events = (unsigned long *)xrealloc(condition_pre_events,
							       (gen_cond_alloc *
								sizeof(unsigned long)));
	    }
	    num_gen_cond++;

	    ASSERT((num_gen_cond -1) == t2);


	    // Record the pre-event of this condition

	    condition_pre_events[t2] = num_gen_event;


	    // Allocate the data structure for post events of this condition

	    condition_post_events[t2] = CreateDArray();


	  }


	}

	// Create the event itself
	// Notice that transition is referenced by id

	event = prefix->AddEvent((t+1),
				 GetSArraySize(tr_preset),
				 GetSArray(tr_preset),
				 GetSArraySize(tr_postset),
				 GetSArray(tr_postset),
				 can_stutter);

	event_queue->Add(event);

	if(num_gen_event == gen_event_alloc) {
	  gen_event_alloc *= 2;
	  gen_event_atoms = (Atom **)xrealloc(gen_event_atoms,
					      (gen_event_alloc *
					       sizeof(Atom*)));
	  event_post_conds = (SARRAY **)xrealloc(event_post_conds,
						 (gen_event_alloc * sizeof(SARRAY *)));
	  DeleteMonSet(postset_events);
	  postset_events = CreateMonSet(gen_event_alloc);
	}
	num_gen_event++;
	ASSERT((num_gen_event -1) == event);


	// Remember the postset

	event_post_conds[event] = CopySArray(tr_postset);


	// Then record the condition postsets from preset

	for(i = 0, j = GetSArraySize(tr_preset); i < j; i++) {

	  // Record that the event is in the
	  // postset of each preset condition

	  AppendDArray(condition_post_events[GetSArrayItem(tr_preset, i)],
		       event);

	}

      }

      if(event_queue->Empty()) {

	// Prefix generated!

	done = TRUE;
	break;

      } else {

	// Get the next event to be handled!

	if(event_queue->PeekSmallest() >= program_id) {

	  // The event didn't exist when the program was created,
	  // need to create a new program.

	  break;

	} else {

	  // The event exists in the program, but was set to false
	  // in the compute statement.

	  ASSERT(extended_event != ULONG_MAX);

	  // Now we have to revert to undo everything after init()

#if 0
	  fprintf(stdout, "----- REVERT -----\n");
#endif
	  smodels->revert();


	  // Reset required the extension of the extended event

	  api->reset_compute(gen_event_atoms[extended_event],
			     TRUE);



	  extended_event = ULONG_MAX;

	  // Force the smallest non-cut-off event to true

	  while(event_queue->Empty() == FALSE) {

	    if(event_queue->PeekSmallest() >= program_id) {

	      // The program bacame too old!

	      break;
	    }

	    event = event_queue->RemoveSmallest();
	    prefix->AssignIDs(event);
	    cut = prefix->CheckCutOff(event);

	    if(cut == FALSE) {

	      // We found the next event to extend

	      extended_event = event;
	      break;

	    } else {

	      // A cut-off found. Skip it!

	      AddtoDSet(cut_off_events, event);

	      // No need to change the compute statement, as it is
	      // already forced to false.

	      if(prefix->IsSuccess(event)) {

		// We found a successful cut-off, exit

		success = TRUE;
		break;

	      }

	      // Mark the cut-off postset conditions
	  
	      sa = event_post_conds[event];
	      for(i = 0, j = GetSArraySize(sa); i < j; i++) {
		t = GetSArrayItem(sa, i);
		AddtoDSet(cut_off_post_conds, t);
	      }

	    }

	  }

	  if(extended_event == ULONG_MAX) {

	    // Could not find anything to extend, or
	    // the program became too old.
	    //
	    // Create a new program

	    break;

	  }

	  // Force the event to be extended to true

	  api->reset_compute(gen_event_atoms[extended_event],
			     FALSE);
	  api->set_compute(gen_event_atoms[extended_event],
			   TRUE);

	  // OK, we're ready to get the extensions of this event!

	}

      }

    }

#ifdef DEBUG_MODEL_CONSISTENCY
    DeleteMonSet(program_enabled_trans);
    program_enabled_trans = (MONSET *)NULL;
    DeleteMonSet(program_enabled_places);
    program_enabled_places = (MONSET *)NULL;
#endif /* DEBUG_MODEL_CONSISTENCY */


  
    // OK, get a fresh smodels & api for the next round

    delete(api);
    delete(smodels);

    smodels = new Smodels();
    api = new Api(&(smodels->program));

  }

#ifdef DEBUG_MODEL_CONSISTENCY
  if(program_enabled_trans != NULL) {
    DeleteMonSet(program_enabled_trans);
    program_enabled_trans = (MONSET *)NULL;
  }
  if(program_enabled_places != NULL) {
    DeleteMonSet(program_enabled_places);
    program_enabled_places = (MONSET *)NULL;
  }
#endif /* DEBUG_MODEL_CONSISTENCY */

  DeleteDSet(cut_off_post_conds);
  DeleteDSet(cut_off_events);

dead_exit:

  for(i = 0; i < num_gen_cond; i++) {
    DeleteDArray(condition_post_events[i]);
  }
  free(condition_pre_events);
  free(condition_post_events);
  free(gen_cond_atoms);
  delete(event_queue);
  DeleteMonSet(postset_events);
  for(i = 0; i < num_gen_event; i++) {
    DeleteSArray(event_post_conds[i]);
  }
  free(event_post_conds);
  free(gen_event_atoms);
  free(buf);
  free(place_atoms);
  free(transition_atoms);
  DeleteSArray(tr_postset);
  DeleteSArray(tr_preset);
  DeleteMonSet(enabled_places);
  DeleteMonSet(enabled_trans);

  for(i = 0; i < numplaces; i++) {
    if(place_posttrans[i] != NULL) {
      DeleteDArray(place_posttrans[i]);
    }
  }
  free(place_posttrans);

  for(i = 0; i < numplaces; i++) {
    if(place_projections[i] != NULL) {
      DeleteDArray(place_projections[i]);
    }
  }
  free(place_projections);

  free(preset_counts);


  if(do_ltl != FALSE) {

    if(success == FALSE) {
#if 0
      fprintf(stdout, "YES - The formula holds.\n");
#endif
      fprintf(stdout, "TRUE\n");

    } else {
#if 0
      fprintf(stdout, "NO - The formula does not hold (counterexample found).\n");
#endif
      fprintf(stdout, "FALSE\n");
      prefix->PrintCounterexample();
      
    }

  }

  return(prefix);

}

#undef MAX

