///////////////////////////////////////////////////////////
// This C++ file is derived from C code by Bob Jenkins.
// (bob_jenkins@compuserve.com)
//
//  See the following Web pages for further information.
//
//  http://www.cris.com/~ttwang/tech/inthash.htm
//  http://ourworld.compuserve.com/homepages/bob_jenkins/blockcip.htm
//
// See the file lookupa.c for the original code.
///////////////////////////////////////////////////////////
// Thanks to Bob for nice licence conditions.
///////////////////////////////////////////////////////////

/*
--------------------------------------------------------------------
lookupa.c, by Bob Jenkins, December 1996.  Same as lookup2.c
You may use this code in any way you wish.  It has no warranty.
Source is http://ourworld.compuserve.com/homepages/bob_jenkins/lookupa.c
--------------------------------------------------------------------
*/

#include "ulhash.h"

/*
--------------------------------------------------------------------
mix -- mix 3 32-bit values reversibly.
For every delta with one or two bit set, and the deltas of all three
  high bits or all three low bits, whether the original value of a,b,c
  is almost all zero or is uniformly distributed,
* If mix() is run forward or backward, at least 32 bits in a,b,c
  have at least 1/4 probability of changing.
* If mix() is run forward, every bit of c will change between 1/3 and
  2/3 of the time.  (Well, 22/100 and 78/100 for some 2-bit deltas.)
mix() takes 36 machine instructions, but only 18 cycles on a superscalar
  machine (like a Pentium or a Sparc).  No faster mixer seems to work,
  that's the result of my brute-force search.  There were about 2^^68
  hashes to choose from.  I only tested about a billion of those.
--------------------------------------------------------------------
*/
#define mix(a,b,c) \
{ \
  a -= b; a -= c; a ^= (c>>13); \
  b -= c; b -= a; b ^= (a<<8); \
  c -= a; c -= b; c ^= (b>>13); \
  a -= b; a -= c; a ^= (c>>12);  \
  b -= c; b -= a; b ^= (a<<16); \
  c -= a; c -= b; c ^= (b>>5); \
  a -= b; a -= c; a ^= (c>>3);  \
  b -= c; b -= a; b ^= (a<<10); \
  c -= a; c -= b; c ^= (b>>15); \
}

/*
--------------------------------------------------------------------
ulhash() -- hash a variable-length key into a 32-bit value
  k     : the key (the unaligned variable-length array of bytes)
  len   : the length of the key, counting by bytes
  level : can be any 4-byte value
Returns a 32-bit value.  Every bit of the key affects every bit of
the return value.  Every 1-bit and 2-bit delta achieves avalanche.
About 6len+35 instructions.

The best hash table sizes are powers of 2.  There is no need to do
mod a prime (mod is sooo slow!).  If you need less than 32 bits,
use a bitmask.  For example, if you need only 10 bits, do
  h = (h & hashmask(10));
In which case, the hash table should have hashsize(10) elements.

If you are hashing n strings (unsigned char **)k, do it like this:
  for (i=0, h=0; i<n; ++i) h = lookup( k[i], len[i], h);

By Bob Jenkins, 1996.  74512.261@compuserve.com.  You may use this
code any way you wish, private, educational, or commercial.

See http://ourworld.compuserve.com/homepages/bob_jenkins/evahash.htm
Use for hash table lookup, or anything where one collision in 2^32 is
acceptable.  Do NOT use for cryptographic purposes.
--------------------------------------------------------------------
*/

unsigned long ulhash( unsigned char *k,     /* the key */
		      unsigned long length, /* the length of the key */
		      unsigned long level   /* the previous hash, or an arbitrary value */)
{
   unsigned long a,b,c,len;

   /* Set up the internal state */
   len = length;
   a = b = 0x9e3779b9;  /* the golden ratio; an arbitrary value */
   c = level;           /* the previous hash value */

   /*---------------------------------------- handle most of the key */
   while (len >= 12)
   {
      a += (k[0] +((unsigned long)k[1]<<8) +((unsigned long)k[2]<<16) +((unsigned long)k[3]<<24));
      b += (k[4] +((unsigned long)k[5]<<8) +((unsigned long)k[6]<<16) +((unsigned long)k[7]<<24));
      c += (k[8] +((unsigned long)k[9]<<8) +((unsigned long)k[10]<<16)+((unsigned long)k[11]<<24));
      mix(a,b,c);
      k += 12; len -= 12;
   }

   /*------------------------------------- handle the last 11 bytes */
   c += length;
   switch(len)              /* all the case statements fall through */
   {
   case 11: c+=((unsigned long)k[10]<<24);
   case 10: c+=((unsigned long)k[9]<<16);
   case 9 : c+=((unsigned long)k[8]<<8);
      /* the first byte of c is reserved for the length */
   case 8 : b+=((unsigned long)k[7]<<24);
   case 7 : b+=((unsigned long)k[6]<<16);
   case 6 : b+=((unsigned long)k[5]<<8);
   case 5 : b+=k[4];
   case 4 : a+=((unsigned long)k[3]<<24);
   case 3 : a+=((unsigned long)k[2]<<16);
   case 2 : a+=((unsigned long)k[1]<<8);
   case 1 : a+=k[0];
     /* case 0: nothing left to add */
   }
   mix(a,b,c);
   /*-------------------------------------------- report the result */
   return c;
}

#undef mix
