// Copyright 1998,1999 by Keijo Heljanko
// This software is provided as is, no warranty of any kind is given.

////////////////////////////////////////////////////////////////////////
//
// Thanks to Burkhard for permission to use his code.
//
// This C++ code is based on the following C file:
// (See subdirectory mci)
//
////////////////////////////////////////////////////////////////////////

/*****************************/
/* occ_net.c                 */
/* Burkhard Graves, 26.10.94 */
/*****************************/

/*
 * Copyright Burkhard Graves, PEP project, University of Hildesheim
 * See: http://www.informatik.uni-hildesheim.de/~pep/HomePage.html 
 */

////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include "braproc.h"
#include "ulstack.h"
#include "dassert.h"

BraProcess::BraProcess ()
{

  num_events = 0;
  num_conditions = 0;
  num_edges = 0;
  num_cutoffs = 0;
  
  conditions = (Condition *) NULL;
  events = (Event *) NULL;

  cutoffs = (BITARRAY *) NULL;
  min_elements = (BITARRAY *) NULL;
  
  real_conflict_conditions = (BITARRAY *) NULL;
  real_conflict_events = (BITARRAY *) NULL;
  empty_real_postset_conditions = (BITARRAY *) NULL;
  singleton_prepreset_events = (BITARRAY *) NULL;
  translation_events = (BITARRAY *) NULL;
  translation_conditions = (BITARRAY *) NULL;
  
  pl_names = (char **) NULL;
  tr_names = (char **) NULL;
  names = (char *) NULL;

}

BraProcess::~BraProcess ()
{
  // This subroutine is based on PEP code.
  //
  // See file mci/occ_net.c

  Edge * edge, * edge_mem;
  Event * event;

  event=++events;
  while (event) {
    edge=event->preset;
    while (edge) {
      edge=(edge_mem=edge)->nextcondition;
      free(edge_mem);
    }
    edge=event->postset;
    while (edge) {
      edge=(edge_mem=edge)->nextcondition;
      free(edge_mem);
    }
    if (event->info) {
      free(event->info);
    }
    event=event->next;
  }
  if (events)
    free(events);
  if (conditions)
    free(++conditions);

  if(translation_events != NULL) {
    DeleteBitArray(translation_events);
  }
  if(translation_conditions != NULL) {
    DeleteBitArray(translation_conditions);
  }
  if(singleton_prepreset_events != NULL) {
    DeleteBitArray(singleton_prepreset_events);
  }
  if(empty_real_postset_conditions != NULL) {
    DeleteBitArray(empty_real_postset_conditions);
  }
  if(real_conflict_events != NULL) {
    DeleteBitArray(real_conflict_events);
  }
  if(real_conflict_conditions != NULL) {
    DeleteBitArray(real_conflict_conditions);
  }
  if(cutoffs != NULL) {
    DeleteBitArray(cutoffs);
  }
  
  if(min_elements != NULL) {
    DeleteBitArray(min_elements);
  }

  if(names != NULL) {
    free(names);
  }

  if(tr_names != NULL) {
    free(tr_names);
  }

  if(pl_names != NULL) {
    free(pl_names);
  }

}

// See PEP file mci/mca.c, function Min().

void BraProcess::InitMin()
{
  
  Condition *pl;

  min_elements = CreateBitArray(num_conditions);

  pl = conditions;
  ++pl;
  while (pl) {
    if (!pl->preset)
      SetBit(min_elements, ((pl->numname) - 1));
    pl = pl->next;
  }
  
}

void BraProcess::InitInfo()
{
  unsigned long i;
  Edge * edge;
  unsigned long real_postset_size;
  unsigned long old_preset_event;
  unsigned long preset_event;
  BOOL found_conflict;
  BOOL first;
  
  InitMin();
  
  real_conflict_conditions = CreateBitArray(num_conditions);

  for(i = 1; i <= num_conditions; i++) {

    if(IsCutoffPostset(i) != FALSE) {
      continue;
    }

    real_postset_size = 0;
    
    for(edge = (conditions + i)->postset; edge != NULL; edge=edge->nextevent) {
      
      if(IsCutoff(edge->event->numname) == FALSE) {
	real_postset_size++;
      }

      if(real_postset_size > 1) {
	break;
      }
    
    }

    if(real_postset_size > 1) {

      SetBit(real_conflict_conditions, (i-1));

    }
    
  }

  real_conflict_events = CreateBitArray(num_events);

  for(i = 1; i <= num_events; i++) {

    // Cutoffs are not considered, so they are skipped.
    
    if(IsCutoff(i) != FALSE) {
      continue;
    }

    found_conflict = FALSE;

    for(edge = (events + i)->preset; edge != NULL; edge=edge->nextcondition) {
      
      if(GetBit(real_conflict_conditions, (edge->condition->numname -1))) {
	found_conflict = TRUE;
	break;
      }
    
    }

    if(found_conflict != FALSE) {

      SetBit(real_conflict_events, (i-1));

    }
    
  }


  empty_real_postset_conditions = CreateBitArray(num_conditions);

  for(i = 1; i <= num_conditions; i++) {

    if(IsCutoffPostset(i) != FALSE) {
      continue;
    }

    real_postset_size = 0;
    
    for(edge = (conditions + i)->postset; edge != NULL; edge=edge->nextevent) {
      
      if(IsCutoff(edge->event->numname) == FALSE) {
	real_postset_size++;
	break;
      }
    
    }

    if(real_postset_size == 0) {

      SetBit(empty_real_postset_conditions, (i-1));

    }

  }


  singleton_prepreset_events = CreateBitArray(num_events);

  for(i = 1; i <= num_events; i++) {

    // Cutoffs are not considered, so they are skipped.
    
    if(IsCutoff(i) != FALSE) {
      continue;
    }

    found_conflict = FALSE;
    first = TRUE;
    old_preset_event = 0;
    
    for(edge = (events + i)->preset; edge != NULL; edge=edge->nextcondition) {

      if(edge->condition->preset == NULL) {
	preset_event = 0;
      } else {
	preset_event = edge->condition->preset->event->numname;
      }

      if(first == FALSE &&
	 preset_event != old_preset_event) {
	found_conflict = TRUE;
	break;
      }

      old_preset_event = preset_event;
      first = FALSE;
      
    }
      
    if(found_conflict == FALSE) {
      SetBit(singleton_prepreset_events, (i-1));
    }

  }

}


void BraProcess::InitTranslation(BITARRAY * visible_events)
{
  UL_STACK * stack;
  unsigned long i;
  Edge * edge;
  Edge * edge2;
  unsigned long preset_event;
  unsigned long preset_condition;

  // This routine gets a set of events as input,
  // and does a closure of events and conditions
  // which is the smallest set of events and conditions
  // such that:
  //
  // (a) all visible events belong to the closure
  //
  // (b) if an event "e" belongs to the closure,
  //     then all conditions in "preset(e)" also
  //     belong to the closure
  //
  // (c) if a condition "b" belongs to the closure,
  //     then:
  //
  //     1. the (unique) event "e" such that
  //        "b \in postset(e)" also belongs to
  //        the closure
  //
  //     2. all events in "postset(b)" belong to the
  //        closure

  // The idea is that only the events in the closure
  // can affect the enabledness of any of the visible
  // transitions.

  // Note: This is a lot like stubborn closure, the
  //       main difference is that we don't know which
  //       of the visible transitions are enabled and
  //       therefore have to be conservative (approximate
  //       upwards) in the closure computation.

  // For deadlocks visible events    = cutoff events
  // For reachability visible events = events which
  // can potentially change a proposition truth value


  stack = CreateULStack();
  
  translation_events = CreateBitArray(num_events);
  translation_conditions = CreateBitArray(num_conditions);

  for(i = 1; i <= num_events; i++) {

    // Only visible events are considered, rest are skipped.
    
    if(!GetBit(visible_events, (i-1)) ||
       GetBit(translation_events, (i -1))) {
      continue;
    }

    // Add the local configuration of this visible event
    // by doing a depth first search.
    
    SetBit(translation_events, (i -1));
    PushULStack(stack, i);
    
    while(!ULStackEmpty(stack)) {

      PopULStack(stack, preset_event);
      
      for(edge = (events + preset_event)->preset; edge != NULL; edge=edge->nextcondition) {
	
	preset_condition = edge->condition->numname;

	if(GetBit(translation_conditions, (preset_condition -1))) {

	  // Condition already handled, continue

	  continue;

	}

	SetBit(translation_conditions, (preset_condition -1));


	// First add condition preset event into stack,
	// if it exists and isn't handled yet.

	if(edge->condition->preset != NULL) {
	  preset_event = edge->condition->preset->event->numname;

	  if(!GetBit(translation_events, (preset_event -1))) {

	    SetBit(translation_events, (preset_event -1));
	    PushULStack(stack, preset_event);

	  }
	}

	for(edge2 = (conditions + preset_condition)->postset; edge2 != NULL; edge2=edge2->nextevent) {

	  // Handle condition postset events
	  if(!GetBit(translation_events, (edge2->event->numname -1))) {
	    SetBit(translation_events, (edge2->event->numname -1));
	    PushULStack(stack, edge2->event->numname);
	  
	  }

	}

      }

    }
    
  }

  DeleteULStack(stack);

}


void BraProcess::AddPresetEvents(BITARRAY * current_configuration)
{
  
  unsigned long i;
  Edge * edge;
  unsigned long preset_event;
  UL_STACK * stack;
  BITARRAY * original_events;
  
  // The translation might remove some events, which
  // need to be recovered by transitively adding all preset events
  // of configuration events in the configuration.
  
  stack = CreateULStack();
  original_events = CreateBitArray(num_events);
  BitArrayCopy(current_configuration, original_events);
  
  for(i = 1; i <= num_events; i++) {

    // Only events in original configuration are investigated

    if(!GetBit(original_events, (i -1))) {
      continue;
    }

    // Add the local configuration of this event
    // by doing a depth first search.
    
    PushULStack(stack, i);
    
    while(!ULStackEmpty(stack)) {

      PopULStack(stack, preset_event);
      
      for(edge = (events + preset_event)->preset; edge != NULL; edge=edge->nextcondition) {

	// Get the preset event if it exists
	
	if(edge->condition->preset != NULL) {

	  preset_event = edge->condition->preset->event->numname;

	  if(!GetBit(current_configuration, (preset_event -1))) {

	    SetBit(current_configuration, (preset_event -1));
	    PushULStack(stack, preset_event);

	  }

	}

      }

    }

  }

  DeleteBitArray(original_events);
  DeleteULStack(stack);

}

unsigned long BraProcess::ContinueToDeadlock(BITARRAY * current_configuration)
{
  unsigned long i;
  unsigned long event;
  unsigned long preset_condition;
  unsigned long postset_condition;
  unsigned long needed_preset_conditions;
  unsigned long enabled_cutoff;
  Edge * edge;
  Edge * edge2;
  BITARRAY * current_conditions;
  unsigned long * preset_counters;
  UL_STACK * stack;

  // Note! Some optimizations might give a marking
  // in which there are conflicts to resolve before the
  // net deadlocks. However all ways of resolving such
  // conflicts will lead to a deadlock.
  //
  // Currently this happens with "optimize_use_enabledness_closure"

  // Initialize with the initial conditions
  
  current_conditions = CreateBitArray(num_conditions);
  BitArrayCopy(min_elements, current_conditions);

#if 0
  fprintf(stdout, "Initial config:\n");

  for(i = 1; i <= num_conditions; i++) {
    if(GetBit(current_conditions, (i -1))) {
      fprintf(stdout, "B%ld\n", i);
    }
  }  
#endif

  // Add all postsets

  for(i = 1; i <= num_events; i++) {

    // Only events in current configuration are investigated
    
    if(!GetBit(current_configuration, (i -1))) {
      continue;
    }

    for(edge = (events + i)->postset; edge != NULL; edge=edge->nextcondition) {

      // This postset condition has been generated.

      // Ensure that it has not been generated before.
      if(GetBit(current_conditions, (edge->condition->numname -1))) {
	ASSERT(FALSE);
      }

      SetBit(current_conditions, (edge->condition->numname -1));

    }

  }

#if 0
  fprintf(stdout, "All postsets added:\n");

  for(i = 1; i <= num_conditions; i++) {
    if(GetBit(current_conditions, (i -1))) {
      fprintf(stdout, "B%ld\n", i);
    }
  }  
#endif

  // Remove all presets

  for(i = 1; i <= num_events; i++) {

    // Only events in current configuration are investigated
    
    if(!GetBit(current_configuration, (i -1))) {
      continue;
    }

    for(edge = (events + i)->preset; edge != NULL; edge=edge->nextcondition) {

      // This preset conditiond has been consumed.
#if 0
      fprintf(stdout, "B%ld -> E%ld\n", edge->condition->numname, i);
#endif
      // Ensure that it has also been generated.
      if(!GetBit(current_conditions, (edge->condition->numname -1))) {
	ASSERT(FALSE);
      }

      // Clear it so it won't be used twice!

      ClearBit(current_conditions, (edge->condition->numname -1));

    }

  }

#if 0
  fprintf(stdout, "All presets removed:\n");

  for(i = 1; i <= num_conditions; i++) {
    if(GetBit(current_conditions, (i -1))) {
      fprintf(stdout, "B%ld\n", i);
    }
  }  
#endif

  stack = CreateULStack();

  preset_counters = (unsigned long *)malloc(num_events *
					    sizeof(unsigned long));

  ASSERT(preset_counters != NULL);


  // Initialize all needed preset counters

  for(i = 1; i <= num_events; i++) {

    // Only events which are not in current configuration,
    // and are not cutoffs are initialized
    
    if(GetBit(current_configuration, (i -1)) ||
       IsCutoff(i)) {
      continue;
    }


    needed_preset_conditions = 0;

    for(edge = (events + i)->preset; edge != NULL; edge=edge->nextcondition) {

      if(!GetBit(current_conditions, (edge->condition->numname -1))) {
      
	needed_preset_conditions++;

      }

    }

    preset_counters[i -1] = needed_preset_conditions;

    if(needed_preset_conditions == 0) {

      // This event is enabled, add it to the stack

      PushULStack(stack, i);

    }

  }

  // Do forward firing until a deadlocking configuration is reached!

  while(!ULStackEmpty(stack)) {

    PopULStack(stack, event);

    if(preset_counters[event -1] == 0) {

      // This event is still enabled, fire it and
      // change the enabledness of other events
      // which are connected to its preset or
      // postset places.

      SetBit(current_configuration, (event -1));

      for(edge = (events + event)->preset; edge != NULL; edge=edge->nextcondition) {
	
	// First remove the preset condition.

	preset_condition = edge->condition->numname;

	if(!GetBit(current_conditions, (preset_condition -1))) {
	  ASSERT(FALSE);
	}

	ClearBit(current_conditions, (preset_condition -1));


	// The increases the needed preset conditions for all its postset events

	for(edge2 = (conditions + preset_condition)->postset; edge2 != NULL; edge2=edge2->nextevent) {

	  if(IsCutoff(edge2->event->numname)) {
	    continue;
	  }

	  // Handle condition postset events
	  
	  preset_counters[edge2->event->numname -1]++;

	}

      }

      // Next add postset events

      for(edge = (events + event)->postset; edge != NULL; edge=edge->nextcondition) {

	// First add the postset condition.
	
	postset_condition = edge->condition->numname;

	if(GetBit(current_conditions, (postset_condition -1))) {
	  ASSERT(FALSE);
	}

	SetBit(current_conditions, (postset_condition -1));


	// The decreases the needed preset conditions for all its postset events

	for(edge2 = (conditions + postset_condition)->postset; edge2 != NULL; edge2=edge2->nextevent) {

	  if(IsCutoff(edge2->event->numname)) {
	    continue;
	  }

	  // Handle condition postset events
	  
	  preset_counters[edge2->event->numname -1]--;

	  if(preset_counters[edge2->event->numname -1] == 0) {
	    
	    PushULStack(stack, edge2->event->numname);

	  }

	}

      }

    }

  }

#if 0
  fprintf(stdout, "Final config:\n");

  for(i = 1; i <= num_conditions; i++) {
    if(GetBit(current_conditions, (i -1))) {
      fprintf(stdout, "B%ld\n", i);
    }
  }  
#endif

  // Sanity check that all cutoff events are disabled

  enabled_cutoff = 0;

  for(i = 1; i <= num_events; i++) {

    // Only check if any cutoff is enabled
    
    if(IsCutoff(i) == FALSE) {
      continue;
    }

    needed_preset_conditions = 0;

    for(edge = (events + i)->preset; edge != NULL; edge=edge->nextcondition) {

      if(!GetBit(current_conditions, (edge->condition->numname -1))) {
      
	needed_preset_conditions++;

      }

    }

    preset_counters[i -1] = needed_preset_conditions;

    if(needed_preset_conditions == 0) {

      // Sanity check failed, the program is buggy!

      enabled_cutoff = i;

      break;

    }

  }

  free(preset_counters);  

  DeleteULStack(stack);

  DeleteBitArray(current_conditions);
  
  return enabled_cutoff;

}



Edge * BraProcess::create_edge(Condition * condition,
			       Event * event,
			       Edge * nextcondition,
			       Edge * nextevent)
{
    Edge *edge;

    edge=(Edge *)malloc(sizeof(Edge));
    edge->condition=condition;
    edge->event=event;
    edge->nextcondition=nextcondition;
    edge->nextevent=nextevent;
    return edge;
}

#define READNUM(NUMBER) fread((void *)&(NUMBER),sizeof(unsigned long),1,in)

int BraProcess::ReadMciFile(FILE * in)
{

  // This subroutine is based on PEP code.
  //
  // See file mci/occ_net.c
  //
  // It has been modified by removing some of the things
  // that aren't needed by this program.
  
  Condition * condition;
  Event * event;
  char * ptr;
  char ** calc_ptr;
  unsigned long i,num;

  READNUM(num_conditions);
  READNUM(num_events);
  conditions
    =(Condition *)malloc(sizeof(Condition) * num_conditions);
  if (num_events) {
    events
      =(Event *)malloc(sizeof(Event) * num_events);
    event=events;
    for (i=1; i<=num_events; ++i) {
      event->numname=i;
      READNUM(event->tr_numname);
      event->preset=event->postset=NULL;
      event->info=NULL;
      event=event->next=event+1;
    }
    (--event)->next=NULL;
    event=events-1;
  } else
    event=events=NULL;
  condition=conditions;
  for (i=1; i<=num_conditions; ++i) {
    condition->numname=i;
    READNUM(condition->pl_numname);
    condition->preset=condition->postset=NULL;
    READNUM(num);
    if (num) {
      (event+num)->postset=condition->preset
	=create_edge(condition,event+num,
		     (event+num)->postset,condition->preset);
      num_edges++;
    }
    READNUM(num);
    while(num) {
      (event+num)->preset=condition->postset
	=create_edge(condition,event+num,
		     (event+num)->preset,condition->postset);
      num_edges++;
      READNUM(num);
    }
    condition=condition->next=condition+1;
  }
  (--condition)->next=NULL;
  --conditions;
  --events;


  // We use our own bitarrays instead of Sets of PEP.
  // (A matter of taste.)
  
  cutoffs=CreateBitArray(num_events);
  READNUM(num);
  num_cutoffs = num;
  while (num) {
    SetBit(cutoffs, num - 1);
    i=num; READNUM(num);
    ((events+i)->info=(Info *)malloc(sizeof(Info)))
      ->corr_event=num;
    READNUM(num);
  }

  
  // We handle the initial marking differently from PEP.
  // Also lot of other info about the prefix gets initialized here.
  
  InitInfo();

  
  // We skip the maximal configurations if they exist.
  //
  // They are not needed in this program.
  
  READNUM(num);
  while (num) {
    while (num) {
      READNUM(num);
    }
    READNUM(num);
  }

  READNUM(num); /* # Places */
  pl_names=(char **)malloc(sizeof(char *)*(num+1));
  i=num;
  READNUM(num); /* # Transitions */
  tr_names=(char **)malloc(sizeof(char *)*(num+1));
  i+=num;
  READNUM(num); /* max. Laenge eines Namens */
  names=(char *)malloc(num=(num+1)*i+2);
  names=(char *)realloc(names,fread((void *)names,1,num,in));
  calc_ptr=pl_names;
  ptr=names;
  while (*ptr) {
    *calc_ptr=ptr;
    ++calc_ptr;
    while (*++ptr)
      ;
    ++ptr;
  }
  *calc_ptr=NULL;
  calc_ptr=tr_names;
  ++ptr;
  while (*ptr) {
    *calc_ptr=ptr;
    ++calc_ptr;
    while (*++ptr)
      ;
    ++ptr;
  }
  *calc_ptr=NULL;

  return 1;
  
}

#undef READNUM

/**************************************************************************/
/* Den numerischen Namen einer Stellenbeschriftung liefern.               */
/**************************************************************************/

unsigned long BraProcess::GetPlNumname(char *ptr)
{
    char **calc_ptr;

    calc_ptr=pl_names;
    while (*calc_ptr) {
        if (!strcmp(*calc_ptr,ptr))
            return ((unsigned long)(calc_ptr-pl_names))+1;
        ++calc_ptr;
    }
    return 0;
}

/**************************************************************************/
/* Die Stellenbeschriftung zu einem numerischen Namen liefern.            */
/**************************************************************************/

char * BraProcess::GetPlName(unsigned long numname)
{
    return *(pl_names+numname-1);
}



BOOL BraProcess::IsCutoffPostset (unsigned long condition_num)
{
  unsigned long event_num;
  Edge * edge;
  
  edge = (conditions + condition_num)->preset;

  if(edge == NULL) {

    return FALSE;
    
  }
  
  event_num = edge->event->numname;
 
  if(GetBit(cutoffs, (event_num - 1))) {
    return TRUE;
  } else {
    return FALSE;
  }

}

// This is a slow routine. Implement a faster
// one some day, if needed.

BOOL BraProcess::IsInInitialState(unsigned long place_num)
{

  Condition *pl;

  pl = conditions;
  ++pl;

  while (pl != NULL) {

    if (pl->preset == NULL) {

      if (pl->pl_numname == place_num) {

	return TRUE;

      }

    }

    pl = pl->next;

  }

  return FALSE;

}


// This is a slow routine. Implement a faster
// one some day, if needed.

BOOL BraProcess::IsInAnyState(unsigned long place_num)
{

  Condition *pl;

  pl = conditions;
  ++pl;

  while (pl != NULL) {

    if (pl->pl_numname == place_num) {

      return TRUE;

    }

    pl = pl->next;

  }

  return FALSE;

}

// This returns one pre-event of a net place

unsigned long BraProcess::GiveAnyPreEvent(unsigned long place_num)
{

  Condition * pl;

  pl = conditions;
  ++pl;

  while (pl != NULL) {

    if(pl->pl_numname == place_num) {

      if(pl->preset != NULL) {

	return pl->preset->event->numname;

      } else {

	return 0;

      }

    }

    pl = pl->next;

  }

  // Not reached!

  ASSERT(FALSE);
  return 0;

}


// An event is visible if its not a cutoff and has
// a needed condition in either its preset or postset.

void BraProcess::InitVisibility(BITARRAY * needed_conditions,
				BITARRAY * visible_events)

{
  Edge * edge;
  unsigned long i;
  BOOL event_visible;
  
  for(i = 1; i <= num_events; i++) {

    // Cutoffs are not considered, so they are skipped.
    
    if(IsCutoff(i) != FALSE) {
      continue;
    }

    event_visible = FALSE;

    for(edge = (events + i)->preset; edge != NULL; edge=edge->nextcondition) {
      
      if(GetBit(needed_conditions, (edge->condition->numname -1))) {
	event_visible = TRUE;
	break;
      }
    
    }

    if(event_visible == TRUE) {
      SetBit(visible_events, (i -1));
      continue;
    }

    for(edge = (events + i)->postset; edge != NULL; edge=edge->nextcondition) {
      
      if(GetBit(needed_conditions, (edge->condition->numname -1))) {
	event_visible = TRUE;
	break;
      }
    
    }

    if(event_visible == TRUE) {
      SetBit(visible_events, (i -1));
      continue;
    }
    
  }
    
}
