/*
 * Main routines:
 *
 * Copyright Keijo Heljanko <Keijo.Heljanko@hut.fi>
 *
 * Smodels:
 *
 * Copyright Patrik Simons <Patrik.Simons@hut.fi>
 *
 * PEP prefix reading code skeleton:
 *
 * Copyright Burkhard Graves, PEP project, University of Hildesheim
 * See: http://www.informatik.uni-hildesheim.de/~pep/HomePage.html 
 *
 */


#include <stdio.h>
#include <string.h>
#include "main.h"
#include "braproc.h"
#include "unf2smo.h"
#include "tree.h"
#include "scanner.h"
#include "tree2smo.h"
#include "dassert.h"

// Include the smodels stuff.

#include <iostream.h>
#include "stable.h"
#include "api.h"
#include "timer.h"
#if defined _BSD_SOURCE || defined _SVID_SOURCE || defined __SVR4
#include <sys/time.h>
#include <sys/resource.h>
#endif

static void usage(char *myname)
{
    fprintf(stderr,"============================================================================\n");
    fprintf(stderr,"Safe net model checker (Smodels reachability translation)\n");
    fprintf(stderr,"Usage: %s [OPTION]... file1 [file2]\n",myname);
    fprintf(stderr,"\n");
    fprintf(stderr,"Options:\n");
    fprintf(stderr,"        -f=fname : Model check the formula in the file fname\n");
    fprintf(stderr,"        -D       : Deadlock check the net (default)\n");
    fprintf(stderr,"        -O0      : Do not optimize the translation\n");
    fprintf(stderr,"        -O1      : Do only guaranteed linear time optimizations\n");
    fprintf(stderr,"        -O2      : Do cheap polynomial time optimizations\n");
    fprintf(stderr,"        -O3      : Do also enabledness closure (default)\n");
    fprintf(stderr,"        -O4      : Do expensive optimizations (much slower than O3)\n");
    fprintf(stderr,"        -n       : Use smodels with no lookahead (fast for easy examples)\n");
    fprintf(stderr,"        -d       : Produce smodels translation only (for debugging)\n");
    fprintf(stderr,"        -v       : Give verbose translation statistics\n");
    fprintf(stderr,"\n");
    fprintf(stderr,"        file1 : input file containing unfolding\n");
    fprintf(stderr,"                extension '.mci': binary coded format\n");
    fprintf(stderr,"        file2 : output result file\n");
    fprintf(stderr,"\n");
    fprintf(stderr,"Copyright (c) 1998-9 Keijo Heljanko <Keijo.Heljanko@hut.fi> (translation)\n");
    fprintf(stderr,"Copyright (c) 1998-9 Patrik Simons <Patrik.Simons@hut.fi> (smodels)\n");
    fprintf(stderr,"\n");
    fprintf(stderr,"Version 1.4, smodels version 2.23, (%s)\n", __DATE__);
    fprintf(stderr,"============================================================================\n");

}

// Some global variables, used in scanner.cc only!

FILE * yyin_for_parser;
BraProcess * braproc_for_parser;

int main (int argc, char *argv[])
{

  FILE * in;
  FILE * in_formula;
  char * input_file;
  char * output_file;
  char * formula_file;
  BraProcess * braproc;
  Unf2Smo * unf2smo;
  TreeNode * formula;
  unsigned long optimize_level;
  BOOL debugging;
  BOOL verbose;
  BOOL error;
  BOOL do_deadlock;
  BOOL do_formula;
  BOOL arg_deadlock;
  BOOL arg_formula;
  BOOL formula_trivial;
  BOOL negate_result;
  BOOL lookahead;
  Timer timer1;
  Timer timer2;
#if defined _BSD_SOURCE || defined _SVID_SOURCE || defined __SVR4
  struct rusage rusage1;
  struct rusage rusage2_start;
  struct rusage rusage2_stop;
#endif

  timer1.start();
  
  optimize_level = 3;
  lookahead = TRUE;
  debugging = FALSE;
  verbose = FALSE;
  error = FALSE;

  do_deadlock = TRUE;
  do_formula = FALSE;
  
  arg_deadlock = FALSE;
  arg_formula = FALSE;

  formula_trivial = FALSE;
  
  in = (FILE *) NULL;
  in_formula = (FILE *) NULL;
  input_file = (char *) NULL;
  output_file = (char *) NULL;
  formula_file = (char *) NULL;
  unf2smo = (Unf2Smo *) NULL;
  formula = (TreeNode *) NULL;

  for (int c = 1; c < argc && error == FALSE; c++) {
    if (argv[c][0] == '-') {
      if (strncmp (&argv[c][1], "f=", 2) == 0) {
	do_formula = TRUE;
	do_deadlock = FALSE;
	arg_formula = TRUE;
	formula_file = &argv[c][3];
      }
      else if (strcmp (&argv[c][1], "D") == 0) {
	do_deadlock = TRUE;
	do_formula = FALSE;
	arg_deadlock = TRUE;
      }
      else if (strcmp (&argv[c][1], "O0") == 0) {
	optimize_level = 0;
      }
      else if (strcmp (&argv[c][1], "O1") == 0) {
	optimize_level = 1;
      }
      else if (strcmp (&argv[c][1], "O2") == 0) {
	optimize_level = 2;
      }
      else if (strcmp (&argv[c][1], "O3") == 0) {
	optimize_level = 3;
      }
      else if (strcmp (&argv[c][1], "n") == 0) {
	lookahead = FALSE;
      }
      else if (strcmp (&argv[c][1], "d") == 0) {
	debugging = TRUE;
      }
      else if (strcmp (&argv[c][1], "v") == 0) {
	verbose = TRUE;
      } else {
	error = TRUE;
      }
    } else {
      if(input_file == NULL) {
	input_file = argv[c];
      } else if(output_file == NULL) {
	output_file = argv[c];
      } else {
	error = TRUE;
	break;
      }
    }
  }

  if(input_file == NULL) {
    error = TRUE;
  }


  // Only do deadlock or formula, but not both
  
  if(arg_formula != FALSE && arg_deadlock != FALSE) {
    error = TRUE;
  }
  if(do_formula != FALSE && formula_file == NULL) {
    error = TRUE;
  }
  
  if(error != FALSE) {
    usage(argv[0]);
    return 1;
  }
    
  if((in = fopen(input_file, "rb")) == NULL) {

    fprintf(stderr, "%s: No such prefix file: %s\n", argv[0], input_file);
    return 2;
   
  }

  if(output_file != NULL) {

    if(freopen(output_file, "wb", stdout) == NULL) {

      fprintf(stderr, "%s: Could not open output file: %s \n", argv[0], output_file);
      return 2;
   
    }
    
  }

  if(formula_file != NULL) {

    if((in_formula = fopen(formula_file, "rb")) == NULL) {

      fprintf(stderr, "%s: No such formula file: %s\n", argv[0], formula_file);
      return 3;
   
    }

  }
    
  braproc = new BraProcess();

  braproc->ReadMciFile(in);

  if(do_formula != FALSE) {

    // Pass two arguments through global variables for convenience.

    yyin_for_parser = in_formula;
    braproc_for_parser = braproc;

    formula = ParseFormula();

    fclose(in_formula);

    if(formula != NULL) {


//#define DEBUG_PRINT_FORMULA 1

#if DEBUG_PRINT_FORMULA
      TreeDisplay(stdout, formula);
      fprintf(stdout, "\n");
#endif

      if(IsSimpleReachabilityFormula(formula) == FALSE) {

	fprintf(stderr, "Formula is not simple reachability, aborting!\n");
	exit(1);

      }


      TreeOptimize(formula);

#if DEBUG_PRINT_FORMULA
      TreeDisplay(stdout, formula);
      fprintf(stdout, "\n");
#endif

      EvaluatePlacesInInitialState(formula);

#if DEBUG_PRINT_FORMULA
      TreeDisplay(stdout, formula);
      fprintf(stdout, "\n");
#endif

      TreeOptimize(formula);

#if DEBUG_PRINT_FORMULA
      TreeDisplay(stdout, formula);
      fprintf(stdout, "\n");
#endif

#undef DEBUG_PRINT_FORMULA

    } else {

      // Parse error in the formula file, just exit.

      exit(1);

    }

    braproc_for_parser = (BraProcess *) NULL;
    yyin_for_parser = (FILE *) NULL;

  }

  if(do_formula != FALSE ||
     braproc->num_cutoffs != 0) {

    unf2smo = new Unf2Smo(braproc);
    unf2smo->SetOptimizeLevel(optimize_level);

  }
    

  if(do_formula != FALSE) {

    // Formula handling

    formula_trivial =
      unf2smo->CreateReachabilityProgram(formula, debugging, &negate_result);

  } else if (braproc->num_cutoffs != 0) {

    // Deadlock handling

    unf2smo->CreateDeadlockProgram(debugging);

  }


  if(debugging != FALSE) {

    if(do_formula != FALSE ||
       braproc->num_cutoffs != 0) {

      unf2smo->PrintProgram();

    } else {

      // Make a fake program with one models if no cutoffs exist.
      // (saves time)
      
      fprintf(stdout, "no_cutoffs.\n");
      
    }
      
    fflush(stdout);

  } else {

    // TODO: Change non-debugging version to less
    // deadlock-checking specific!

    if(do_formula != FALSE) {

      // Compute result for formula

      timer2.start();

#if defined _BSD_SOURCE || defined _SVID_SOURCE || defined __SVR4
      getrusage (RUSAGE_SELF, &rusage2_start);
#endif
    
      if(formula_trivial != FALSE ||
	 unf2smo->ComputeModel(lookahead) != FALSE) {

	if(negate_result == FALSE) {
	  fprintf(stdout, "TRUE\n");
	} else {
	  fprintf(stdout, "FALSE\n");
	}
	  
	timer2.stop();

#if defined _BSD_SOURCE || defined _SVID_SOURCE || defined __SVR4
	getrusage (RUSAGE_SELF, &rusage2_stop);
#endif
    
	if(formula_trivial == FALSE) {

	  unf2smo->CreateReachabilityConfiguration();

	} else {

	  if(unf2smo->current_configuration != NULL) {
	    unf2smo->PrintConfiguration();
	  }

	}

      } else {

	if(negate_result == FALSE) {
	  fprintf(stdout, "FALSE\n");
	} else {
	  fprintf(stdout, "TRUE\n");
	}

	timer2.stop();
    
#if defined _BSD_SOURCE || defined _SVID_SOURCE || defined __SVR4
	getrusage (RUSAGE_SELF, &rusage2_stop);
#endif
    
      }

    } else {

      // Compute output for the deadlock program

      timer2.start();

#if defined _BSD_SOURCE || defined _SVID_SOURCE || defined __SVR4
      getrusage (RUSAGE_SELF, &rusage2_start);
#endif

      if(braproc->num_cutoffs == 0) {

	fprintf(stdout, "FALSE - EVERY\n");

	timer2.stop();

#if defined _BSD_SOURCE || defined _SVID_SOURCE || defined __SVR4
	getrusage (RUSAGE_SELF, &rusage2_stop);
#endif
    
      } else if(unf2smo->ComputeModel(lookahead) != FALSE) {
      
	fprintf(stdout, "FALSE\n");

	timer2.stop();

#if defined _BSD_SOURCE || defined _SVID_SOURCE || defined __SVR4
	getrusage (RUSAGE_SELF, &rusage2_stop);
#endif
    
	if(unf2smo->CreateDeadlockConfiguration() == FALSE) {

	  // Internal error! (A bug somewhere caught by a sanity check.)

	  exit(2);

	} else {

	  // Everything OK.
	  ;

	}

      } else {

	fprintf(stdout, "TRUE\n");

	timer2.stop();

#if defined _BSD_SOURCE || defined _SVID_SOURCE || defined __SVR4
	getrusage (RUSAGE_SELF, &rusage2_stop);
#endif

      }

    }
    

    fflush(stdout);
    
    timer1.stop();

#if defined _BSD_SOURCE || defined _SVID_SOURCE || defined __SVR4
    getrusage (RUSAGE_SELF, &rusage1);
    fprintf(stdout,
	    "\nTime needed:      %ld.%ld\n",
	    rusage1.ru_utime.tv_sec +
	    rusage1.ru_stime.tv_sec +
	    ((rusage1.ru_utime.tv_usec +
	      rusage1.ru_stime.tv_usec)
	     / 1000000),
	    ((((rusage1.ru_utime.tv_usec +
		rusage1.ru_stime.tv_usec)
	       / 1000) < 1000) ?
	     ((rusage1.ru_utime.tv_usec +
		rusage1.ru_stime.tv_usec)
	       / 1000) :
	     (((rusage1.ru_utime.tv_usec +
		rusage1.ru_stime.tv_usec)
	       / 1000) - 1000)));
#else
    fprintf(stdout, "\nTime needed:      %s s\n", timer1.print());
#endif
    
    fflush(stdout);
    
  }


  // Print statistics from smodels

  if(debugging == FALSE && verbose != FALSE) {

    fprintf(stdout, "\nSmodels statistics:\n");

    if(braproc->num_cutoffs == 0 && do_formula == FALSE) {

      fprintf(stdout, "\nProgram not generated (no cutoff events)\n");
	
    } else {
	
#if defined _BSD_SOURCE || defined _SVID_SOURCE || defined __SVR4

      rusage2_stop.ru_utime.tv_sec -= rusage2_start.ru_utime.tv_sec;
      if(rusage2_stop.ru_utime.tv_usec < rusage2_start.ru_utime.tv_usec) {
	rusage2_stop.ru_utime.tv_usec += 1000000;
	rusage2_stop.ru_utime.tv_sec--;
      }
      rusage2_stop.ru_utime.tv_usec -= rusage2_start.ru_utime.tv_usec;

      rusage2_stop.ru_stime.tv_sec -= rusage2_start.ru_stime.tv_sec;
      if(rusage2_stop.ru_stime.tv_usec < rusage2_start.ru_stime.tv_usec) {
	rusage2_stop.ru_stime.tv_usec += 1000000;
	rusage2_stop.ru_stime.tv_sec--;
      }
      rusage2_stop.ru_stime.tv_usec -= rusage2_start.ru_stime.tv_usec;

      fprintf(stdout,
	      "\nDuration: %ld.%ld\n",
	      rusage2_stop.ru_utime.tv_sec +
	      rusage2_stop.ru_stime.tv_sec +
	      ((rusage2_stop.ru_utime.tv_usec +
		rusage2_stop.ru_stime.tv_usec)
	       / 1000000),
	      ((((rusage2_stop.ru_utime.tv_usec +
		  rusage2_stop.ru_stime.tv_usec)
		 / 1000) < 1000) ?
	       ((rusage2_stop.ru_utime.tv_usec +
		 rusage2_stop.ru_stime.tv_usec)
		/ 1000) :
	       (((rusage2_stop.ru_utime.tv_usec +
		  rusage2_stop.ru_stime.tv_usec)
		 / 1000) - 1000)));
#else
      fprintf(stdout, "\nDuration: %s\n", timer2.print());
#endif

      unf2smo->PrintStatistics();

    }
	
    fflush(stdout);
      
  }


  if(output_file != NULL) {
    fclose(stdout);
  }
  
  if(do_formula != FALSE ||
     braproc->num_cutoffs != 0) {

    delete(unf2smo);

  }

  if(do_formula != FALSE) {

    if(formula != NULL) {

      TreeDelete(formula);

    }

  }
  
  delete(braproc);
  
  fclose(in);
  
  return 0;

}


