// Copyright 1998 by Keijo Heljanko
// This software is provided as is, no warranty of any kind is given.

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "dassert.h"
#include "ulstack.h"

#ifdef DEBUG
#include <unistd.h>
#endif DEBUG

#ifdef NO_BZERO_DEFINED
#define bzero(s,n) memset(s,0,n)
#endif /* NO_BZERO_DEFINED */

UL_STACK *CreateULStack(void)
{
  UL_STACK *stack;
  UL_STACK_PAGE *page;
  UL_STACK_ELEMENT *elements;

  /* The number of elements must be a power of two. */
  DEBUG_ASSERT((UL_STACK_PAGE_ELEMENTS &
		(UL_STACK_PAGE_ELEMENTS - 1)) == 0);

  if((stack = ((UL_STACK *)malloc(sizeof(UL_STACK)))) == NULL) {
    fprintf(stderr, "Could not allocate ul stack.\n");
    exit(1);
  }
  
  stack->num_pages = 1;

  if((page = ((UL_STACK_PAGE *)malloc(sizeof(UL_STACK_PAGE)))) == NULL) {
    fprintf(stderr, "Could not allocate ul stack page.\n");
    exit(1);
  }

  stack->last_page = page; /* A linked list of pages here. */
  page->prev = NULL;       /* A NULL terminated list. */

  if((elements = ((UL_STACK_ELEMENT *)malloc(sizeof(UL_STACK_ELEMENT) *
					     UL_STACK_PAGE_ELEMENTS)))
     == NULL) {
    fprintf(stderr, "Could not allocate ul stack page elements.\n");
    exit(1);
  }

  stack->curr_item = elements;

  stack->first_item = elements;
  page->first_item = elements;

  stack->last_item = ULStackLastItem(elements);
  page->last_item = ULStackLastItem(elements);

  return stack;

}

void DeleteULStack(UL_STACK *stack)
{
  UL_STACK_PAGE *page;
  UL_STACK_PAGE *prev;

  for(page = stack->last_page; page != NULL; page = prev) {

    free(page->first_item); /* Free the page elements. */

    prev = page->prev;      /* Get the next page address. */

    free(page);             /* Free the page. */
  }

  free(stack);

}

void GrowULStack(UL_STACK *stack)
{
  UL_STACK_PAGE *page;
  UL_STACK_ELEMENT *elements;

  if((page = ((UL_STACK_PAGE *)malloc(sizeof(UL_STACK_PAGE)))) == NULL) {
    fprintf(stderr, "Could not allocate ul stack page.\n");
    exit(1);
  }

  /* Add to the front of the page list. */

  page->prev = stack->last_page;
  stack->last_page = page;
  stack->num_pages++;

  if((elements = ((UL_STACK_ELEMENT *)malloc(sizeof(UL_STACK_ELEMENT) *
						UL_STACK_PAGE_ELEMENTS)))
     == NULL) {
    fprintf(stderr, "Could not allocate ul stack page elements.\n");
    exit(1);
  }

  stack->curr_item = elements;

  stack->first_item = elements;
  page->first_item = elements;

  stack->last_item = ULStackLastItem(elements);
  page->last_item = stack->last_item;

}

void ShrinkULStack(UL_STACK *stack)
{
  UL_STACK_PAGE *page;

  /* Remove from the front of the page list
     and deallocate page memory. */
  
  page = stack->last_page->prev;
  free(stack->last_page->first_item);
  free(stack->last_page);
  stack->last_page = page;
  stack->num_pages--;

  DEBUG_ASSERT(page != NULL);

  stack->curr_item = page->last_item;
  stack->first_item = page->first_item;
  stack->last_item = page->last_item;

  return;

}

