// Copyright 1998,1999 by Keijo Heljanko
// This software is provided as is, no warranty of any kind is given.

#include "combi.h"
#include "dassert.h"
#include "tree2smo.h"
#include "y_tab.h"
#include "main.h"

// Make some obvious optimizations to the formula.
// Quite a big set of trivial equivalences used.
// Note that because the formula is here a tree,
// most optimizations based on identical
// subformulas can not be implemented here, but
// a DAG-like sharing structure is needed instead.

void TreeOptimize(TreeNode *node)
{
  TreeNode *tmp;
  TreeNode *tmp2;

  tmp = (TreeNode *) NULL;
  tmp2 = (TreeNode *) NULL;

  if (node != NULL) {
    switch (node->info) {

    case PLACE:
      break;

    case FALSE_TOKEN:
      break;

    case TRUE_TOKEN:
      break;

    default:

      TreeOptimize(node->left);
      TreeOptimize(node->right);

      switch (node->info) {

      case EXISTS:
	ASSERT(FALSE);
	break;

      case DIAMOND:

	ASSERT(node->right != NULL);

	if(node->right->info == FALSE_TOKEN) {

	  node->info = FALSE_TOKEN;
	  TreeDelete(node->right);
	  node->right = (TreeNode *) NULL;

	} else if (node->right->info == TRUE_TOKEN) {

	  node->info = TRUE_TOKEN;
	  TreeDelete(node->right);
	  node->right = (TreeNode *) NULL;

	}
	break;

      case OR:

	ASSERT(node->left != NULL);
	ASSERT(node->right != NULL);

	if(node->left->info == TRUE_TOKEN ||
	   node->right->info == TRUE_TOKEN) {

	  node->info = TRUE_TOKEN;
	  TreeDelete(node->left);
	  TreeDelete(node->right);
	  node->left = (TreeNode *) NULL;
	  node->right = (TreeNode *) NULL;

	} else if(node->left->info == FALSE_TOKEN &&
		  node->right->info == FALSE_TOKEN) {

	  node->info = FALSE_TOKEN;
	  TreeDelete(node->left);
	  TreeDelete(node->right);
	  node->left = (TreeNode *) NULL;
	  node->right = (TreeNode *) NULL;

	} else if(node->left->info == FALSE_TOKEN) {

	  TreeDelete(node->left);
	  node->left = (TreeNode *) NULL;
	  tmp = node->right;
	  node->info = tmp->info;
	  node->left = tmp->left;
	  node->right = tmp->right;
	  node->place_num = tmp->place_num;
	  node->place_atom = tmp->place_atom;
	  tmp->left = (TreeNode *) NULL;
	  tmp->right = (TreeNode *) NULL;
	  TreeDelete(tmp);

	} else if(node->right->info == FALSE_TOKEN) {

	  TreeDelete(node->right);
	  node->right = (TreeNode *) NULL;
	  tmp = node->left;
	  node->info = tmp->info;
	  node->left = tmp->left;
	  node->right = tmp->right;
	  node->place_num = tmp->place_num;
	  node->place_atom = tmp->place_atom;
	  tmp->left = (TreeNode *) NULL;
	  tmp->right = (TreeNode *) NULL;
	  TreeDelete(tmp);

	} else if((node->left->info == PLACE &&
		   node->right->info == NOT &&
		   node->right->right->info == PLACE &&
		   (node->left->place_num ==
		    node->right->right->place_num)) ||
		  (node->right->info == PLACE &&
		   node->left->info == NOT &&
		   node->left->right->info == PLACE &&
		   (node->right->place_num ==
		    node->left->right->place_num))) {

	  node->info = TRUE_TOKEN;
	  TreeDelete(node->left);
	  TreeDelete(node->right);
	  node->left = (TreeNode *) NULL;
	  node->right = (TreeNode *) NULL;

	} else if(node->left->info == PLACE &&
		  node->right->info == PLACE &&
		  (node->left->place_num ==
		   node->right->place_num)) {

	  tmp = node->left;
	  tmp2 = node->right;
	  node->info = PLACE;
	  node->left = (TreeNode *) NULL;
	  node->right = (TreeNode *) NULL;
	  node->place_num = tmp->place_num;
	  node->place_atom = tmp->place_atom;
	  TreeDelete(tmp);
	  TreeDelete(tmp2);

	}

	break;

      case AND:

	ASSERT(node->left != NULL);
	ASSERT(node->right != NULL);

	if(node->left->info == FALSE_TOKEN ||
	   node->right->info == FALSE_TOKEN) {

	  node->info = FALSE_TOKEN;
	  TreeDelete(node->left);
	  TreeDelete(node->right);
	  node->left = (TreeNode *) NULL;
	  node->right = (TreeNode *) NULL;

	} else if(node->left->info == TRUE_TOKEN &&
		  node->right->info == TRUE_TOKEN) {

	  node->info = TRUE_TOKEN;
	  TreeDelete(node->left);
	  TreeDelete(node->right);
	  node->left = (TreeNode *) NULL;
	  node->right = (TreeNode *) NULL;

	} else if(node->left->info == TRUE_TOKEN) {

	  TreeDelete(node->left);
	  node->left = (TreeNode *) NULL;
	  tmp = node->right;
	  node->info = tmp->info;
	  node->left = tmp->left;
	  node->right = tmp->right;
	  node->place_num = tmp->place_num;
	  node->place_atom = tmp->place_atom;
	  tmp->left = (TreeNode *) NULL;
	  tmp->right = (TreeNode *) NULL;
	  TreeDelete(tmp);

	} else if(node->right->info == TRUE_TOKEN) {

	  TreeDelete(node->right);
	  node->right = (TreeNode *) NULL;
	  tmp = node->left;
	  node->info = tmp->info;
	  node->left = tmp->left;
	  node->right = tmp->right;
	  node->place_num = tmp->place_num;
	  node->place_atom = tmp->place_atom;
	  tmp->left = (TreeNode *) NULL;
	  tmp->right = (TreeNode *) NULL;
	  TreeDelete(tmp);

	} else if((node->left->info == PLACE &&
		   node->right->info == NOT &&
		   node->right->right->info == PLACE &&
		   (node->left->place_num ==
		    node->right->right->place_num)) ||
		  (node->right->info == PLACE &&
		   node->left->info == NOT &&
		   node->left->right->info == PLACE &&
		   (node->right->place_num ==
		    node->left->right->place_num))) {

	  node->info = FALSE_TOKEN;
	  TreeDelete(node->left);
	  TreeDelete(node->right);
	  node->left = (TreeNode *) NULL;
	  node->right = (TreeNode *) NULL;

	} else if(node->left->info == PLACE &&
		  node->right->info == PLACE &&
		  (node->left->place_num ==
		   node->right->place_num)) {

	  tmp = node->left;
	  tmp2 = node->right;
	  node->info = PLACE;
	  node->left = (TreeNode *) NULL;
	  node->right = (TreeNode *) NULL;
	  node->place_num = tmp->place_num;
	  node->place_atom = tmp->place_atom;
	  TreeDelete(tmp);
	  TreeDelete(tmp2);

	}

	break;

      case NOT:

	ASSERT(node->right != NULL);

	if(node->right->info == FALSE_TOKEN) {

	  node->info = TRUE_TOKEN;
	  TreeDelete(node->right);
	  node->right = (TreeNode *) NULL;

	} else if(node->right->info == TRUE_TOKEN) {

	  node->info = FALSE_TOKEN;
	  TreeDelete(node->right);
	  node->right = (TreeNode *) NULL;

	} else if(node->right->info == NOT) {

	  ASSERT(node->right->right != NULL);

	  tmp = node->right->right;
	  node->right->right = (TreeNode *) NULL;
	  TreeDelete(node->right);
	  node->info = tmp->info;
	  node->left = tmp->left;
	  node->right = tmp->right;
	  node->place_num = tmp->place_num;
	  node->place_atom = tmp->place_atom;
	  tmp->left = (TreeNode *) NULL;
	  tmp->right = (TreeNode *) NULL;
	  TreeDelete(tmp);

	} else if(node->right->info == AND &&
		  node->right->left->info == NOT &&
		  node->right->right->info == NOT) {

	  tmp = node->right->left->right;
	  tmp2 = node->right->right->right;
	  node->right->left->right = (TreeNode *) NULL;
	  node->right->right->right = (TreeNode *) NULL;
	  TreeDelete(node->right);
	  node->info = OR;
	  node->left = tmp;
	  node->right = tmp2;

	  // Try optimize again for this node.
	  // This is dangerous, however termination is
	  // guaranteed by the fact that optimized
	  // tree is always smaller than the non-optimized one.

	  TreeOptimize(node);

	} else if(node->right->info == OR &&
		  node->right->left->info == NOT &&
		  node->right->right->info == NOT) {

	  tmp = node->right->left->right;
	  tmp2 = node->right->right->right;
	  node->right->left->right = (TreeNode *) NULL;
	  node->right->right->right = (TreeNode *) NULL;
	  TreeDelete(node->right);
	  node->info = AND;
	  node->left = tmp;
	  node->right = tmp2;

	  // Try optimize again for this node.
	  // This is dangerous, however termination is
	  // guaranteed by the fact that optimized
	  // tree is always smaller than the non-optimized one.

	  TreeOptimize(node);

	}

	break;

      default:

	ASSERT(FALSE);

	break;
      }

    }

  }
  
}


// Recursively count all the diamond modalities
// in the formula.


unsigned long CountDiamonds(TreeNode * node) {

  unsigned long diamonds = 0;

  if (node != NULL) {

    switch (node->info) {

    case PLACE:
      break;

    case FALSE_TOKEN:
      break;

    case TRUE_TOKEN:
      break;

    case DIAMOND:

      diamonds = 1;

      // Fall through to the default case below.

    default:

      diamonds = diamonds +
	CountDiamonds(node->left) + 
	CountDiamonds(node->right);

      break;

    }
  }

  return diamonds;

}


// Recursively count all the place tokens
// in the formula tree.


unsigned long CountPlaces(TreeNode * node) {

  unsigned long places = 0;

  if (node != NULL) {

    switch (node->info) {

    case PLACE:

      places = 1;
      
      break;

    case FALSE_TOKEN:
      break;

    case TRUE_TOKEN:
      break;

    default:

      places = places + 
	CountPlaces(node->left) + 
	CountPlaces(node->right);

      break;

    }
  }

  return places;

}

// This checks if the formula is of the form
// which can be solved by only one reachability
// test.

BOOL IsSimpleReachabilityFormula(TreeNode *node) {

  unsigned long diamonds;

  // Only formulas with one diamond modality can be handled.

  diamonds = CountDiamonds(node);

  // fprintf(stderr, "Diamonds: %ld\n", diamonds);

  if(diamonds > 1) {

    return FALSE;

  }

  return TRUE;

}

// The genral approach:
// 
// Split the formula into two parts:
//
// 1. part that should hold in the initial state
// 2. part that should hold in some reachable state
//
// If the second part is empty, the prefix isn't needed
// at all, and the formula can be evaluated based on the
// initial state only!


// Change place formulas evaluated in the initial state
// to true or false depending on the initial state.
//
// The following routine needs the global variable:
//
// braproc_for_parser
//
// needs to be properly initialized!

void EvaluatePlacesInInitialState(TreeNode * node) {

  if (node != NULL) {

    switch (node->info) {

    case PLACE:

      if(braproc_for_parser->
	 IsInInitialState(node->place_num) != FALSE) {

	// The place is in initial state.

	node->info = TRUE_TOKEN;

      } else {

	// The place is not in initial state.

	node->info = FALSE_TOKEN;

      }

      // Clear the placename

      node->place_num = 0L;
      node->place_atom = (Atom *) NULL;

      break;

    case FALSE_TOKEN:
      break;

    case TRUE_TOKEN:
      break;

    case DIAMOND:

      // Do not evaluate places under diamond modalities
      // in initial state, just check that each place
      // really exists in the prefix with the call below.

      EvaluatePlacesInAnyState(node->right);
      
      break;

    default:

      EvaluatePlacesInInitialState(node->left);
      EvaluatePlacesInInitialState(node->right);

      break;

    }

  }

}

// Change place formulas which never appear in
// the prefix into false in formulas behind a
// diamond modality.
//
// This should not be called directly, call the routine
// above to do the full stuff.
//
// The following routine needs the global variable:
//
// braproc_for_parser
//
// needs to be properly initialized!

void EvaluatePlacesInAnyState(TreeNode * node) {

  if (node != NULL) {

    switch (node->info) {

    case PLACE:

      if(braproc_for_parser->
	 IsInAnyState(node->place_num) == FALSE) {

	// The place is not true in any state.

	node->info = FALSE_TOKEN;

	// Clear the placename

	node->place_num = 0L;
	node->place_atom = (Atom *) NULL;

      }

      break;

    case FALSE_TOKEN:
      break;

    case TRUE_TOKEN:
      break;

    case DIAMOND:

      // Should not happen!

      ASSERT(false);
      
      break;

    default:

      EvaluatePlacesInAnyState(node->left);
      EvaluatePlacesInAnyState(node->right);

      break;

    }

  }

}




