// Copyright 1998,1999 by Keijo Heljanko
// This software is provided as is, no warranty of any kind is given.

#include <stdlib.h>
#include "unf2smo.h"
#include "bitarr.h"
#include "ulstack.h"
#include "shash.h"
#include "tree.h"
#include "y_tab.h"
#include "dassert.h"
#include "tree2smo.h"

#include "smodels.h"
#include "api.h"

#define MAX(a,b) (((a) >= (b)) ? (a) : (b))

BOOL Unf2Smo::CreateReachabilityProgram(TreeNode * formula, BOOL debugging, BOOL *negate)
{
  Api * api;
  Edge * edge;
  Edge * preset;
  Atom ** places;
  BITARRAY * needed_conditions;
  BITARRAY * visible_events;
  BITARRAY * preset_array;
  BITARRAY * postset_array;
  UL_STACK * stack;
  SIMPLEHASH * hash;
  BOOL formula_trivial;
  BOOL negate_result;
  unsigned long * created_rules;
  unsigned long * created;
  unsigned long * created_heads;
  unsigned long * heads;
  unsigned long * rule_start;
  unsigned long * old_rule;
  unsigned long ** sorted_rules;
  unsigned long * place_nums;
  unsigned long preset_event;
  unsigned long postset_event;
  unsigned long replacing_event;
  unsigned long num_rules;
  unsigned long headcount;
  unsigned long head;
  unsigned long head_index;
  unsigned long rule_size;
  unsigned long bottom_atom_index;
  unsigned long atoms_per_event;
  unsigned long num_places;
  unsigned long only_place;
  unsigned long i;
  unsigned long j;
    
  static char buf[256];

  created_rules = (unsigned long *) NULL;
  created = (unsigned long *) NULL;
  created_heads = (unsigned long *) NULL;
  heads = (unsigned long *) NULL;
  rule_start = (unsigned long *) NULL;
  sorted_rules = (unsigned long **) NULL;
  hash = (SIMPLEHASH *) NULL;

  places = (Atom **) NULL;
  place_nums = (unsigned long *) NULL;
  num_places = 0;

  // Assert that the top-level formula is
  // of one of the supported types.

  ASSERT(formula->info == TRUE_TOKEN ||
	 formula->info == FALSE_TOKEN ||
	 formula->info == DIAMOND ||
	 (formula->info == NOT &&
	  formula->right->info == DIAMOND));
  

  // Formula atom is a kind of comment
  // left for debugging purposes (and
  // also using the translator through
  // the debugging interface).



  api = new Api(&(smodels->program));

  conflict_atom = api->new_atom();
  if(optimize_remove_bottom_event == FALSE) {
    bottom_event_atom = api->new_atom();
  } else {
    bottom_event_atom = (Atom *) NULL;
  }

  if(debugging != FALSE) {
  
    api->set_name(conflict_atom, "conflict");
    if(optimize_remove_bottom_event == FALSE) {
      api->set_name(bottom_event_atom, "e0");
    }

  }


  formula_atom = api->new_atom();

  if(debugging != FALSE) {
  
    api->set_name(formula_atom, "formula");

  }

  negate_result = FALSE;
  
  if(formula->info == TRUE_TOKEN ||
     formula->info == FALSE_TOKEN ||
     (formula->info == DIAMOND &&
      formula->right->info == PLACE) ||
     (formula->info == NOT &&
      formula->right->info == DIAMOND &&
      formula->right->right->info == PLACE)) {

    formula_trivial = TRUE;

    true_atom = api->new_atom();

    if(debugging != FALSE) {
      api->set_name(true_atom, "true");
    }

    api->begin_rule(BASICRULE);
    api->add_head(formula_atom);


    only_place = 0;
    if(formula->info == DIAMOND) {
      only_place = formula->right->place_num;
    } else if(formula->info == NOT) {
      only_place = formula->right->right->place_num;
    }

    if(only_place != 0 &&
       braproc->IsInAnyState(only_place) != FALSE) {

      current_configuration = CreateBitArray(braproc->num_events);

      preset_event = braproc->GiveAnyPreEvent(only_place);

      if(preset_event != 0) {

	SetBit(current_configuration, (preset_event -1));
	braproc->AddPresetEvents(current_configuration);

      }

    } else {

      current_configuration = (BITARRAY *) NULL;

    }

    if(formula->info == TRUE_TOKEN ||
       (formula->info == DIAMOND &&
	braproc->IsInAnyState(only_place) != FALSE) ||
       (formula->info == NOT &&
	braproc->IsInAnyState(only_place) == FALSE)) {

      api->add_body(true_atom, TRUE);
      negate_result = FALSE;

    } else {

      api->add_body(true_atom, FALSE);
      negate_result = TRUE;

    }
    api->end_rule();

    api->begin_rule(BASICRULE);
    api->add_head(true_atom);
    api->end_rule();
    delete(api);
  
    *negate = negate_result;

    return formula_trivial;

  } else {

    formula_trivial = FALSE;

  }

  // A bitmap to mark the needed conditions.
  // (Used by optimizations.)
  
  needed_conditions = CreateBitArray(braproc->num_conditions);
  replaced_events = CreateBitArray(braproc->num_events);
  visible_events = CreateBitArray(braproc->num_events);
  
  
  // Create the place atoms for the original net
  
  num_places = CountPlaces(formula);

  places = (Atom **) malloc(sizeof(Atom *) * (num_places + 1));
  place_nums = (unsigned long *) malloc(sizeof(unsigned long) * (num_places + 1));

  for(i = 0; i < (num_places + 1); i++) {
    places[i] = (Atom *) NULL;
    place_nums[i] = 0;
  }

  // No general DAG-based optimizations used in formula translation,
  // only make sure that the place atoms are created exactly
  // once, because their duplication is potentially very constly.
  //
  // TODO: Add DAG-based (as opposed to tree-based)
  //       optimizations here.

  AddPlaceAtoms(debugging,
		api,
		places,
		place_nums,
		formula);

  num_places = 0;
  for(i = 0; ; i++) {

    if(places[i] == NULL) {
      break;
    }

    num_places++;

  }

  ASSERT(num_places > 0);

  
  // Check which conditions will be needed by projection rules
  
  MarkNeededConditions(places,
		       place_nums,
		       needed_conditions);

  // Initialize the visibility of events.
  // Event is visible iff it has a needed condition in either
  // its preset or postset, and the event is not a cutoff
  // event.

  braproc->InitVisibility(needed_conditions,
			  visible_events);

  
  if(optimize_use_enabledness_closure != FALSE) {
    braproc->InitTranslation(visible_events);
  }

  // Add the bottom event atom for nonoptimized translation

  if(optimize_remove_bottom_event == FALSE) {
    api->begin_rule(BASICRULE);
    api->add_head(bottom_event_atom);
    api->end_rule();
  }

  // TODO: Minimize memory allocation by calloc:ing when needed

  if(optimize_use_choice_rules == FALSE) {
    atoms_per_event = 2;
  } else {
    atoms_per_event = 1;
  }

  atoms = (Atom **) malloc(sizeof(Atom *) * ((atoms_per_event *
					      braproc->num_events) +
					     (braproc->num_conditions) +
					     2));

  // Initialize the first index to NULL to catch some coding errors

  num_atoms = 0;
  atoms[num_atoms] = NULL;
  num_atoms++;
  atoms[num_atoms] = bottom_event_atom; // This can be (Atom *)NULL
  bottom_atom_index = num_atoms;
  num_atoms++;

  events = (unsigned long *) malloc(sizeof(unsigned long) *
				    (braproc->num_events));
  if(optimize_use_choice_rules == FALSE) {
    blocked_events = (unsigned long *) malloc(sizeof(unsigned long) *
					      (braproc->num_events));
  }
  conditions = (unsigned long *) malloc(sizeof(unsigned long) *
					(braproc->num_conditions));
  replacement_events = (unsigned long *) malloc(sizeof(unsigned long) *
						(braproc->num_events));


  // Add the needed atoms (events, blocking atoms, conditions)
    
  for(i = 1; i <= braproc->num_events; i++) {

    events[i -1] = 0;
    if(optimize_use_choice_rules == FALSE) {
      blocked_events[i -1] = 0;
    }
    replacement_events[i -1] = 0;
    
    if(optimize_use_enabledness_closure != FALSE &&
       braproc->IsTranslationEvent(i) == FALSE) {

      // This event is not needed in reachability checking
	
      continue;

    }
      
    if(braproc->IsCutoff(i) == FALSE) {
	
      // Only non-cutoff events needed in this translation

      if(optimize_replace_events_by_events != FALSE) {

	// If we have an event which has only one event
	// in the preset of its preset, and has no events
	// conflicting with it, then this event is not needed.
	// It can then be replaced by its prepreset event.
	//
	// Note: Do the optimization only for invisible
	//       transitions.
	
	if(!GetBit(visible_events, (i -1)) &&
	   braproc->EventHasSingletonPrepreset(i) == FALSE &&
	   braproc->EventHasRealConflict(i) == FALSE) {
	
	  // Do special case handling for events which have
	  // several events in their prepreset, all of which have
	  // been replaced by the same event.

	  edge = (braproc->events + i)->preset;
	  preset = edge->condition->preset;
	  
	  if(preset != NULL) {
	    preset_event = preset->event->numname;
	  } else {
	    preset_event = 0;
	  }

	  if(preset_event != 0 &&
	     GetBit(replaced_events, preset_event -1)) {
	    
	    preset_event = replacement_events[preset_event -1];

	  }

	  replacing_event = preset_event;
	  
	  for( ; edge != NULL; edge = edge->nextcondition) {

	    preset = edge->condition->preset;

	    if(preset != NULL) {
	      preset_event = preset->event->numname;
	    } else {
	      preset_event = 0;
	    }

	    if(preset_event != 0 &&
	       GetBit(replaced_events, preset_event -1)) {

	      preset_event = replacement_events[preset_event -1];

	    }

	    if(preset_event != replacing_event) {

	      replacing_event = preset_event +1;
	      break;
	  
	    }

	  }

	  if(replacing_event == preset_event &&
	     replacing_event < i) {

	    // All Prepreset events have been replaced by the same event,
	    // replace also this by the replacing event.

	    replacement_events[i -1] = replacing_event;
	    if(replacing_event != 0) {
	      events[i -1] = events[replacing_event -1];
	    } else {
	      events[i -1] = bottom_atom_index;
	    }
	    SetBit(replaced_events, (i -1));
	    continue;
	    
	  }

	}


	// Do this optimization only for invisible events.
	
	if(!GetBit(visible_events, (i -1)) &&
	   braproc->EventHasSingletonPrepreset(i) != FALSE &&
	   braproc->EventHasRealConflict(i) == FALSE) {

	  if((braproc->events +i)->preset->condition->preset == NULL) {
	    
	    replacement_events[i -1] = 0;
	    events[i -1] = bottom_atom_index;
	    SetBit(replaced_events, (i -1));
	    continue;
	    
	  } else {

	    if((braproc->events +i)->preset->condition->preset->event->numname < i) {

	      // This event is done.

	      replacing_event = (braproc->events +i)->preset->condition
		->preset->event->numname;

	      // It might be the case that the replacing event is itself replaced.

	      if(replacing_event != 0 &&
		 GetBit(replaced_events, (replacing_event -1))) {

		// In that case follow the pointer.
		replacing_event = replacement_events[replacing_event -1];

	      }
	      
	      replacement_events[i -1] = replacing_event;
	      if(replacing_event != 0) {
		events[i -1] = events[replacing_event -1];
	      } else {
		events[i -1] = bottom_atom_index;
	      }
	      SetBit(replaced_events, (i -1));
	      continue;
	      
	    } else {

	      // Event has a smaller numbered event in it's preset,
	      // skip this optimization.
	      //
	      // (This shouldn't happen in nets created by PEP.
	      //  It's not fatal if it does, only an optimization is missed.
	      //  If this happens often it can be fixed. - kepa@cc.hut.fi)
	      
	      ;
	      
	    }
	    
	    
	  }
	  
	}

      }


      atoms[num_atoms] = api->new_atom();
      events[i -1] = num_atoms;
      num_atoms++;
  
      if(debugging != FALSE) {
  
	sprintf(buf, "e%ld", i);
	api->set_name(atoms[events[i -1]], buf);

      }

      // Make it possible for visible events to always
      // block.
      
      if((optimize_use_choice_rules == FALSE) &&
	 (optimize_remove_blocking == FALSE ||
	  GetBit(visible_events, (i -1)) ||
	  braproc->EventHasRealConflict(i))) {
	
	atoms[num_atoms] = api->new_atom();
	blocked_events[i -1] = num_atoms;
	num_atoms++;

	if(debugging != FALSE) {
  
	  sprintf(buf, "be%ld", i);
	  api->set_name(atoms[blocked_events[i -1]], buf);

	}
	  
      }

    }

  }


  // Add the conditions
  
  for(i = 1; i <= braproc->num_conditions; i++) {

    conditions[i -1] = 0;
    
    if(GetBit(needed_conditions, (i -1))) {

	atoms[num_atoms] = api->new_atom();
	conditions[i -1] = num_atoms;
	num_atoms++;
	
	if(debugging != FALSE) {
  
	  sprintf(buf, "b%ld", i);
	  api->set_name(atoms[conditions[i -1]], buf);

	}	  
      
    }

  }


  // Create a hash table if duplicate rules need to be
  // removed.

  // TODO: This hash table is created "Big enough",
  //       add dynamic memory management!
  
  if(optimize_use_multihead_rules != FALSE ||
     optimize_remove_duplicate_rules != FALSE) {
    hash = CreateSimpleHash(MAX(braproc->num_events, braproc->num_conditions));
  }

  // Allocate memory to store heads of
  // rules.

  // TODO: Add dynamic memory management.

  created_heads = (unsigned long *)malloc((sizeof(unsigned long)) * 2 *
					  MAX(braproc->num_events,
					      braproc->num_conditions));
  heads = created_heads;
  headcount = 0;
  
  // Allocate memory to store already created
  // rules.

  // TODO: Add dynamic memory management.
  
  created_rules = (unsigned long *)
    malloc((sizeof(unsigned long)) *
	   (braproc->num_edges +     // Storage for all edges
	    MAX((3*braproc->num_events),
		(2*braproc->num_conditions)) // Storage for rule sizes etc.
	    ));

  created = created_rules;
  num_rules = 0;

  
  // Add basic rules about presets and blocking.

  stack = CreateULStack();
  preset_array = CreateBitArray((braproc->num_events) +1);

  for(i = 1; i <= braproc->num_events; i++) {

    if(events[i -1] != 0 &&
       !GetBit(replaced_events, (i -1))) {

      rule_size = 0;
      
      // Do this only for non-cutoff events

      for(edge = (braproc->events + i)->preset;
	  edge != NULL;
	  edge = edge->nextcondition) {

	preset = edge->condition->preset;

	if(preset != NULL) {
	  preset_event = preset->event->numname;
	} else {
	  preset_event = 0;
	}

	if(preset_event != 0 &&
	   GetBit(replaced_events, preset_event -1)) {
	  preset_event = replacement_events[preset_event -1];
	}

	if(optimize_remove_bottom_event != FALSE &&
	   preset_event == 0) {

	  // Remove the bottom event atom from presets
	  
	  continue;
	  
	}
	
	// Remove duplicates and reverse preset order for a nicer
	// appereance in printout.
	  
	if(!GetBit(preset_array, preset_event)) {

	  SetBit(preset_array, preset_event);
	  PushULStack(stack, preset_event);
	  rule_size++;
	}
	    
      }

      // Remember the rules in created as following tuple:
      // <headnum, rulesize+1, body1, ..., bodyn, ruletype>

      rule_start = created;
      *heads++ = i;
      *heads++ = 0; // A linked list next link!
      headcount++;
      *created++ = headcount;

      *created++ = (rule_size+1);
      
      while(!ULStackEmpty(stack)) {

	PopULStack(stack, preset_event);
	ClearBit(preset_array, preset_event);

	// Add preset as positive

	*created++ = preset_event;
	
      }  

      // Remember the rule type as the last "body atom"
      
      if(optimize_use_choice_rules == FALSE ||
	 (!GetBit(visible_events, (i -1)) &&
	  braproc->EventHasRealConflict(i) == FALSE)) {
	*created++ = (unsigned long)BASICRULE;
      } else {
	*created++ = (unsigned long)CHOICERULE;
      }

      if(optimize_use_multihead_rules != FALSE &&
	 created[-1] != BASICRULE) {

	if((old_rule = InsertDataIfNew(hash, &rule_start[1])) != NULL) {

	  // The body is a duplicate of some earlier rule.
	  // Add the new head as the first head in the headlist.
	  
	  heads[-1] = old_rule[-1];
	  old_rule[-1] = *rule_start;

	  created = rule_start;
	  continue;

	}

      }

      num_rules++;

    }

  }


  // Feed all preset rules to smodels
  
  rule_start = created_rules;
    
  for(i = 0; i < num_rules; i++) {

    // Feed the rule to smodels.

    head_index = rule_start[0];
    head = created_heads[2 * (head_index -1)];
    j = rule_start[1];

    // Reverse the head list for nicer output

    PushULStack(stack, head);
    headcount = 1;

    while((head_index = (created_heads[(2*(head_index -1)) +1])) != 0) {
      head = created_heads[2 * (head_index -1)];
      PushULStack(stack, head);
      headcount++;
    }

    // Recall what the rule type was supposed to be

    if(headcount > 1 &&
       ((RuleType) rule_start[j +1]) == BASICRULE) {
      
      // Optimization disabled 04/05/1999, generaterules
      // are going away from smodels soon.
      
      ASSERT(FALSE);
#if 0      
      // BASICRULES with several heads are GENERATERULES!

      api->begin_rule(GENERATERULE);
      api->set_atleast_head(headcount);
#endif      

    } else {
      api->begin_rule((RuleType) rule_start[j +1]);
    }

    while(!ULStackEmpty(stack)) {

      PopULStack(stack, head);

      api->add_head(atoms[events[head -1]]);

    }


    // Skip head
    rule_start++;

    // Read body length, substract the ruletype of size 1
    j = (*rule_start++) -1;
    
    for( ; j > 0; j--) {

      preset_event = *rule_start++;
      if(preset_event == 0) {
	api->add_body(bottom_event_atom, TRUE);
      } else {
	api->add_body(atoms[events[preset_event -1]], TRUE);
      }
	  
    }
      
    // Skip rule type
    rule_start++;
    
  
    // Add blocking condition as negative
    // Skip this if optimizing and the event has no real conflicts.
      
    if((optimize_use_choice_rules == FALSE) &&
       (optimize_remove_blocking == FALSE ||
	GetBit(visible_events, (i -1)) ||
	braproc->EventHasRealConflict(head))) {

      ASSERT(optimize_use_multihead_rules == FALSE);
      
      api->add_body(atoms[blocked_events[head -1]], FALSE);
      api->end_rule();

      // Add the rule for blocking condition

      api->begin_rule(BASICRULE);
      api->add_head(atoms[blocked_events[head -1]]);
      api->add_body(atoms[events[head -1]], FALSE);
      
    } 

    api->end_rule();

    
  }
      
  if(optimize_use_multihead_rules != FALSE ||
     optimize_remove_duplicate_rules != FALSE) {
    ClearSimpleHash(hash);
  }

  created = created_rules;
  num_rules = 0;


  // Add rules about conflict sets.

  // Create a bit-array to find duplicates within rules.
  // (Replacing events by events might create duplicates.)
 
  postset_array = CreateBitArray((braproc->num_events) +1);


  for(i = 1; i <= braproc->num_conditions; i++) {

    // This had: if(condition[i -1] != 0 && ...
    //
    // which is invalid here!
    
    if(braproc->ConditionHasRealConflict(i)) {

      rule_size = 0;
	
      for(edge = ((braproc->conditions)+i)->postset;
	  edge != NULL;
	  edge = edge->nextevent) {

	if(braproc->IsCutoff(edge->event->numname) != FALSE) {

	  // Skip cutoffs.
	    
	  continue;
	    
	}

	postset_event = edge->event->numname;

	if(postset_event != 0 &&
	   GetBit(replaced_events, postset_event -1)) {
	  
	  postset_event = replacement_events[postset_event -1];

	}
	
	if(optimize_remove_bottom_event != FALSE &&
	   postset_event == 0) {

	  // Remove the bottom event atom from presets
	  // (Shouldn't happen, but is not fatal.)
	  
	  continue;
	  
	}

	if(postset_event != 0 &&
	   events[postset_event -1] == 0) {

	  // This postset event has been removed!
	  
	  continue;

 	}
	
	// Remove duplicates and reverse postset order for a nicer
	// appereance in printout.
	  
	if(!GetBit(postset_array, postset_event)) {

	  SetBit(postset_array, postset_event);
	  PushULStack(stack, postset_event);
	  rule_size++;
	}
	
      }

      if(rule_size <= 1) {

	// This might happen due to event to event replacement.

	if(rule_size == 0) {

	  // Shouldn't happen, but is non-fatal.
	  
	  continue;

	}
	
	PopULStack(stack, postset_event);
	ClearBit(postset_array, postset_event);
	continue;
	
      }
      
      rule_start = created;
      *created++ = rule_size;

      while(!ULStackEmpty(stack)) {

	PopULStack(stack, postset_event);
	ClearBit(postset_array, postset_event);

	*created++ = postset_event;
	
      }  

      // This condition has a real conflict in it's postset.
      // Add a conflict rule.

      if(optimize_remove_duplicate_rules != FALSE ||
	 optimize_remove_sub_and_supersets != FALSE) {

	// First sort the rule atoms so that rules will be
	// canonical.
	//
	// I use insertion sort instead of heap sort because
	// the rules will be almost sorted. Only through the
	// optimizations can some out of order elements be added.

	InsertionSort(rule_start[0], &(rule_start[1]));
	
	if(optimize_remove_duplicate_rules != FALSE) {
	  
	  if(InsertDataTellIfNew(hash, rule_start) == FALSE) {

	    // This is a duplicate of some earlier rule.
	    // Try to generate a new one with better luck.
	  
	    created = rule_start;
	    continue;

	  }

	}

      }

      num_rules++;

    }

  }


  if(optimize_sort_rules != FALSE ||
     optimize_remove_sub_and_supersets != FALSE) {

    sorted_rules = (unsigned long **)malloc(num_rules *
					    (sizeof(unsigned long *)));

    rule_start = created_rules;

    for(i = 0; i < num_rules; i++) {

      sorted_rules[i] = rule_start;

      rule_start = &(rule_start[rule_start[0] +1]);

    }

    if(optimize_remove_sub_and_supersets != FALSE) {

      num_rules = OptimizeRemoveSubAndSupersets(num_rules,
						sorted_rules,
						FALSE);

    }
    
    if(optimize_sort_rules != FALSE) {
      
      HeapSortRules(num_rules, sorted_rules);

    }
    
  }


  // Feed all conflict rules to smodels
  
  rule_start = created_rules;
    
  for(i = 0; i < num_rules; i++) {

    if(optimize_sort_rules != FALSE ||
       optimize_remove_sub_and_supersets != FALSE) {

      rule_start = sorted_rules[i];

    }

    // Feed the rule to smodels.

    j = *rule_start++;
    
    if(j == 2) {

      api->begin_rule(BASICRULE);

    } else {

      api->begin_rule(CONSTRAINTRULE);
//      api->set_atleast_body(2, TRUE); // Old smodels 2 api
      api->set_atleast_body(2);
      
    }
      
    api->add_head(conflict_atom);

    for( ; j > 0; j--) {

      api->add_body(atoms[events[(*rule_start++) -1]], TRUE);
	  
    }
      
    api->end_rule();
      
  }

  
  if(optimize_sort_rules != FALSE) {

    free(sorted_rules);
    sorted_rules = (unsigned long **)NULL;

  }


  if(optimize_remove_duplicate_rules != FALSE) {

    ClearSimpleHash(hash);

  }

  // Add rules to model the conditions.

  for(i = 1; i <= braproc->num_conditions; i++) {

    if(conditions[i -1] != 0) {

      api->begin_rule(BASICRULE);
      api->add_head(atoms[conditions[i -1]]);

      if((braproc->conditions + i)->preset == NULL) {

	if(optimize_remove_bottom_event == FALSE) {
	  api->add_body(bottom_event_atom, TRUE);
	}
	  
      } else {

	if (optimize_remove_bottom_event == FALSE ||
	    events[((braproc->conditions + i)
		    ->preset->event->numname) -1] !=
	    bottom_atom_index) {
	  
	  api->add_body(atoms[events[((braproc->conditions + i)
				      ->preset->event->numname) -1]],
			TRUE);
	  
	}

      }
	
      for(edge = ((braproc->conditions)+i)->postset;
	  edge != NULL;
	  edge = edge->nextevent) {

	if(braproc->IsCutoff(edge->event->numname) != FALSE) {

	  // Skip cutoffs.
	    
	  continue;
	    
	}

	postset_event = edge->event->numname;
	  
	if(postset_event != 0 &&
	   GetBit(replaced_events, postset_event -1)) {
	    
	  postset_event = replacement_events[postset_event -1];

	  // Should not happen

	  ASSERT(FALSE);
	  
	}

	if(optimize_remove_bottom_event != FALSE &&
	   postset_event == 0) {

	  // Remove the bottom event atom from postsets. (Shouldn't happen.)
	  
	  ASSERT(FALSE);

	  continue;
	  
	}
	  
	if(postset_event != 0 &&
	   events[postset_event -1] == 0) {

	  // This postset event has been removed!
	  
	  // Should not happen

	  ASSERT(FALSE);
	  
	  continue;

	}
	  
	// Remove duplicates and reverse postset order for a nicer
	// appereance in printout.
	  
	if(!GetBit(postset_array, postset_event)) {

	  SetBit(postset_array, postset_event);
	  PushULStack(stack, postset_event);
	    
	}
	    
      }

      while(!ULStackEmpty(stack)) {

	PopULStack(stack, postset_event);
	ClearBit(postset_array, postset_event);
	  
	api->add_body(atoms[events[postset_event -1]], FALSE);
	  
      }  

      api->end_rule();

    }

  }	

  
  // Add the projection rules which project the prefix
  // configuration on the places of the original net.

  GenProjectionRules(api,
		     places,
		     place_nums);


  num_formula_atoms = 0;

  ASSERT(formula->info == DIAMOND ||
	 (formula->info == NOT &&
	  formula->right->info == DIAMOND));


  negate_result = FALSE;
  
  diamond_atom = GenFormulaProgram(debugging,
				   api,
				   formula,
				   &negate_result);


  // If we do debugging, then add a rule to
  // show the negation polarity of the top-level formula.
  //
  // E.g. If we get the rule:
  //
  // formula :- diamond.
  //
  // and there is no stable model containing diamond,
  // then the formula is false.
  //
  // The other case is:
  //
  // formula :- not diamond.
  //
  // and there is no stable model containing diamond,
  // then the formula is true.
  //
  // We do this also for non-debugging to not to
  // change the number of rules from the debugging
  // version into the non-debugging one.

  api->begin_rule(BASICRULE);
  api->add_head(formula_atom);

  if(negate_result == FALSE) {
    api->add_body(diamond_atom, TRUE);
  } else {
    api->add_body(diamond_atom, FALSE);
  }
  
  api->end_rule();


  // Add the compute statement
    
  api->set_compute(conflict_atom, FALSE);
  api->set_compute(diamond_atom, TRUE);
    
  api->done();

//  fprintf(stdout, "%% Program done!\n");
//  fflush(stdout);
  
  smodels->init();

  if(hash != NULL) {
    DeleteSimpleHash(hash);
  }
  
  DeleteBitArray(postset_array);
  DeleteBitArray(preset_array);
  DeleteULStack(stack);

  delete(api);

  DeleteBitArray(visible_events);
  DeleteBitArray(needed_conditions);

  if(created_heads != NULL) {
    free(created_heads);
  }
  
  if(created_rules != NULL) {
    free(created_rules);
  }

  if(place_nums != NULL) {
    free(place_nums);
  }

  if(places != NULL) {
    free(places);
  }

  *negate = negate_result;
  return formula_trivial;

}

#undef MAX


void Unf2Smo::AddPlaceAtoms(BOOL debugging,
			    Api * api,
			    Atom ** places,
			    unsigned long * place_nums,
			    TreeNode * node)

{
  unsigned long i;
  char * buf;

  if(node != NULL) {

    switch (node->info) {

    case PLACE:

      for(i = 0; ; i++) {

	if(places[i] == NULL) {
	  break;
	}

	if(place_nums[i] == node->place_num) {
	  
	  // The place atom already exists, return

	  node->place_atom = places[i];

	  return;

	}

      }

      place_nums[i] = node->place_num;
      places[i] = api->new_atom();
      node->place_atom = places[i];

      if(debugging != FALSE) {
  
	buf = (char *)malloc(sizeof(char) * 256);

	sprintf(buf, "s%ld", place_nums[i]);
	api->set_name(places[i], buf);

	free(buf);

      }

      break;

    case FALSE_TOKEN:
      break;

    case TRUE_TOKEN:
      break;

    default:

      AddPlaceAtoms(debugging,
		    api,
		    places,
		    place_nums,
		    node->left);

      AddPlaceAtoms(debugging,
		    api,
		    places,
		    place_nums,
		    node->right);

      break;

    }

  }

}


Atom * Unf2Smo::GenFormulaProgram(BOOL debugging,
				  Api * api,
				  TreeNode * node,
				  BOOL * negate_result)

{

  char * buf;
  Atom * atom;
  Atom * left_atom;
  Atom * right_atom;
  BOOL left_child_result;
  BOOL right_child_result;
  
  *negate_result = FALSE;
  left_child_result = FALSE;
  right_child_result = FALSE;

  if(node != NULL) {

    switch (node->info) {

    case FALSE_TOKEN:

      ASSERT(FALSE);
      break;

    case TRUE_TOKEN:

      ASSERT(FALSE);
      break;

    case PLACE:

      return node->place_atom;
      break;

    case NOT:

      right_atom = GenFormulaProgram(debugging,
				     api,
				     node->right,
				     &right_child_result);

      ASSERT(right_child_result == FALSE);

      *negate_result = TRUE;

      return right_atom;

      break;

    case DIAMOND:

      right_atom = GenFormulaProgram(debugging,
				     api,
				     node->right,
				     &right_child_result);

      num_formula_atoms++;

      atom = api->new_atom();

      if(debugging != FALSE) {
  
	api->set_name(atom, "diamond");

      }

      api->begin_rule(BASICRULE);
      api->add_head(atom);

      if(right_child_result == FALSE) {
	api->add_body(right_atom, TRUE);
      } else {
	api->add_body(right_atom, FALSE);
      }

      api->end_rule();

      return atom;

      break;

    case AND:
    case OR:
      
      left_atom = GenFormulaProgram(debugging,
				    api,
				    node->left,
				    &left_child_result);

      right_atom = GenFormulaProgram(debugging,
				     api,
				     node->right,
				     &right_child_result);

      num_formula_atoms++;

      atom = api->new_atom();

      if(debugging != FALSE) {
  
	buf = (char *)malloc(sizeof(char) * 256);

	sprintf(buf, "f%ld", num_formula_atoms);
	api->set_name(atom, buf);

	free(buf);

      }

      if(node->info == AND) {
	api->begin_rule(BASICRULE);
      } else {
	api->begin_rule(CONSTRAINTRULE);
	api->set_atleast_body(1);
      }

      api->add_head(atom);

      if(left_child_result == FALSE) {
	api->add_body(left_atom, TRUE);
      } else {
	api->add_body(left_atom, FALSE);
      }

      if(right_child_result == FALSE) {
	api->add_body(right_atom, TRUE);
      } else {
	api->add_body(right_atom, FALSE);
      }

      api->end_rule();

      return atom;

      break;


    default:

      ASSERT(FALSE);
      break;

    }

  }

  // NOTREACHED!
  
  ASSERT(FALSE);

  return NULL;

}


// This is a slow routine with long formulas, fixme.

void Unf2Smo::MarkNeededConditions(Atom ** places,
				   unsigned long * place_nums,
				   BITARRAY * needed_conditions)
{
  Condition * pl;
  unsigned long i;
  unsigned long pl_numname;
  
  pl = braproc->conditions;
  ++pl;

  while (pl != NULL) {

    pl_numname = pl->pl_numname;

    for(i = 0; ; i++) {

      if(places[i] == NULL) {
	break;
      }

      if(place_nums[i] == pl_numname &&
	 braproc->IsCutoffPostset(pl->numname) == FALSE) {

	SetBit(needed_conditions, (pl->numname -1));
	break;
	
      }

    }
    
    pl = pl->next;

  }

}


// This is a slow routine with long formulas, fixme.

void Unf2Smo::GenProjectionRules(Api * api,
				 Atom ** places,
				 unsigned long * place_nums)

{
  Condition * pl;
  unsigned long num_proj;
  unsigned long i;

  for(i = 0; ; i++) {

    if(places[i] == NULL) {
      break;
    }

    num_proj = 0;

    pl = braproc->conditions;
    ++pl;

    while (pl != NULL) {

      if (pl->pl_numname == place_nums[i] &&
	  braproc->IsCutoffPostset(pl->numname) == FALSE) {

	num_proj++;

      }

      pl = pl->next;

    }
    
    ASSERT(num_proj > 0);

    if(num_proj < 2) {

      api->begin_rule(BASICRULE);

    } else {

      api->begin_rule(CONSTRAINTRULE);
      api->set_atleast_body(1);

    }

    api->add_head(places[i]);


    pl = braproc->conditions;
    ++pl;

    while (pl != NULL) {

      if (pl->pl_numname == place_nums[i] &&
	  braproc->IsCutoffPostset(pl->numname) == FALSE) {

	api->add_body(atoms[conditions[pl->numname -1]], TRUE);

      }

      pl = pl->next;

    }

    api->end_rule();

  }

}


// Reconstruct the configuration reached
// and then print it.

void Unf2Smo::CreateReachabilityConfiguration(void)

{

  unsigned long i;

  current_configuration = CreateBitArray(braproc->num_events);

  // Find out the current configuration.

  for(i = 1; i <= braproc->num_events; i++) {

    if(events[i -1] != 0) {

      if(atoms[events[i -1]] == bottom_event_atom ||
	 atoms[events[i -1]]->Bpos) {

	SetBit(current_configuration, (i -1));

      }

    }

  }

  braproc->AddPresetEvents(current_configuration);

  PrintConfiguration();

}
