// Copyright 1998 by Keijo Heljanko
// This software is provided as is, no warranty of any kind is given.

#include <stdlib.h>
#include "unf2smo.h"
#include "bitarr.h"
#include "ulstack.h"
#include "shash.h"

#include "smodels.h"
#include "api.h"

Unf2Smo::Unf2Smo(BraProcess *braprocess)
{
  braproc = braprocess;

  smodels = new Smodels();

  atoms = (Atom **) NULL;
  events = (unsigned long *) NULL;
  blocked_events = (unsigned long *) NULL;
  replacement_events = (unsigned long *) NULL;
  conditions = (unsigned long *) NULL;

  replaced_events = (BITARRAY *) NULL;
  current_configuration = (BITARRAY *) NULL;
  
  num_atoms = 0;

  // Turn off all optimizations
  
  optimize_remove_bottom_event = FALSE;
  optimize_remove_conditions = FALSE;
  optimize_remove_blocking = FALSE;
  optimize_replace_conditions_by_events = FALSE;
  optimize_replace_events_by_events = FALSE;
  optimize_use_choice_rules = FALSE;
  optimize_use_deadlock_property = FALSE;
  optimize_remove_duplicate_rules = FALSE;
  optimize_sort_rules = FALSE;
  optimize_remove_sub_and_supersets = FALSE;
  
}

void Unf2Smo::SetOptimizeLevel(unsigned long optimize_level)
{

  // Always remove the bottom element
  
  optimize_remove_bottom_event = TRUE;

  if(optimize_level == 0) {
    return;
  }

  // These are linear time optimizations
  
  optimize_remove_conditions = TRUE;
  optimize_remove_blocking = TRUE;
  optimize_replace_conditions_by_events = TRUE;
  optimize_replace_events_by_events = TRUE;
  optimize_use_choice_rules = TRUE;
#if 0    
  optimize_use_deadlock_property = TRUE;  // Don't use yet!
                                          // TODO: Prove correct!!
#endif

  if(optimize_level == 1) {
    return;
  }

  // These are polynomial time optimizations
  
  optimize_remove_duplicate_rules = TRUE;
  optimize_sort_rules = TRUE;

  if(optimize_level == 2) {
    return;
  }

  // These are too slow optimizations compared to the gain
  
  optimize_remove_sub_and_supersets = TRUE; // TODO: Implement me efficiently
                                            // Currently VERY slow  
  

}

Unf2Smo::~Unf2Smo()
{
  if(current_configuration != NULL) {
    DeleteBitArray(current_configuration);
  }
  if(replaced_events != NULL) {
    DeleteBitArray(replaced_events);
  }
  if(atoms != NULL) {
    free(atoms);
  }
  if(conditions != NULL) {
    free(conditions);
  }
  if(replacement_events != NULL) {
    free(replacement_events);
  }
  if(blocked_events != NULL) {
    free(blocked_events);
  }
  if(events != NULL) {
    free(events);
  }
  
  braproc = (BraProcess *) NULL;

  delete(smodels);
  
}

void Unf2Smo::InsertionSort(unsigned long n, unsigned long *a)
{
  unsigned long i;
  unsigned long j;
  unsigned long v;

  // Good old insertion sort.
  //
  // It is linear time for already sorted sequences.
  //
  // (And quadratic for sequences in the worst case.)

  
  if(n < 2) {
    return;
  }
  
  for(i = 1; i < n; i++) {

    v = a[i];
    j = i;

    while (j != 0 && a[j -1] > v) {
      a[j] = a[j -1];
      j--;
    }

    a[j] = v;
    
  }

}

long Unf2Smo::RuleCompare(unsigned long *rule1,
			  unsigned long *rule2)
{
  unsigned long i;
  unsigned long n1;
  unsigned long n2;
  long smaller;

  n1=*rule1++;
  n2=*rule2++;

  smaller = ((n1 < n2) ? (-1) : ((n1 > n2) ? (1) : (0)));
  i = ((n1 < n2) ? (n1) : (n2));

  for(; i != 0; i--) {

    n1=*rule1++;
    n2=*rule2++;

    if(n1 < n2) {
      return -1;
    }
    if(n1 > n2) {
      return 1;
    }

  }

  return smaller;

}



void Unf2Smo::RuleDownHeap(unsigned long **rules,
			   unsigned long n,
			   unsigned long k)
{
  unsigned long j;
  unsigned long *v;

  v = rules[k];

  while((j = ((2*k) +1)) < n) {

    if((j +1) < n) {

      if(RuleCompare(rules[j], rules[j +1]) < 0) {

	j++;

      }


    }

    if(RuleCompare(v, rules[j]) >= 0) {

      break;

    }

    rules[k] = rules[j];
    k = j;
      
  }

  rules[k] = v;
  
}


void Unf2Smo::HeapSortRules(unsigned long m, unsigned long **rules)
{
  unsigned long n;
  unsigned long k;
  unsigned long * t;

  // Heap Sort - Guaranteed O(n log n) time
  
  if(m <= 1) {
    return;
  }

  n = m;

  for(k = (n/2); k >= 1;) {

    k--;
    RuleDownHeap(rules, n, k);

  }

  for(;;) {

    n--;
    t = rules[0];
    rules[0] = rules[n];
    rules[n] = t;

    if(n <= 1) {
      break;
    }

    RuleDownHeap(rules, n, 0);

  }

}

unsigned long Unf2Smo::OptimizeRemoveSubAndSupersets(unsigned long num_rules,
						     unsigned long **rules,
						     BOOL supersets)
{
  unsigned long i;
  unsigned long j;
  unsigned long k;
  unsigned long l;
  unsigned long m;
  unsigned long n;
  unsigned long num_removed;
  unsigned long *t;
  
  // TODO: Implement an efficient one instead of this (really) slow one

  if(num_rules > 2) {

    num_removed = 0;
	
    for(i = 0; i < num_rules; i++) {

      l = rules[i][0];

      for(j = 1; j < num_rules; j++) {

	// The rules are internally sorted, remove
	// a rule if another rule exists which has
	// a superset of atoms of our current rule.

	// Do modulo indexing for candidates
	
	k = i+j;
	if(k >= num_rules) {
	  k = k - num_rules;
	}

	if(rules[k] == NULL) {

	  // Candidate rule already deleted

	  continue;

	}

	m = rules[k][0];

	if(supersets == FALSE) {

	  // Remove subset rules if a superset rule exists
	  
	  if(m <= l) {

	    // The candidate replacement is too small to be a superset 
	  
	    continue;
	    
	  }

	  n = l;
	  
	  for(; m > 0; m--) {
	    
	    if(rules[k][m] > rules[i][n]) {

	      // If the last atom of the rule is too big, skip it
	      
	      continue;
	      
	    }

	    if(rules[k][m] < rules[i][n]) {

	      // If the last atom of the rule is too small, the
	      // candiate isn't a superset

	      break;
	      
	    }

	    // The last atoms match, one less atoms to find

	    n--;

	    if(n == 0) {

	      // All original rule atoms found, remove rule
	      
	      rules[i] = (unsigned long *)NULL;
	      num_removed++;
	      break;
	      
	    }
	    
	  }

	} else {

	  // Remove superset rules if a subset rule exists
	  
	  if(m >= l) {

	    // The candidate replacement is too big to be a subset 
	  
	    continue;
	    
	  }

	  n = l;
	  
	  for(; n > 0; n--) {
	    
	    if(rules[i][n] > rules[k][m]) {

	      // If the last atom of the rule is too big, skip it
	      
	      continue;
	      
	    }

	    if(rules[i][n] < rules[k][m]) {

	      // If the last atom of the rule is too small, the
	      // candiate isn't a subset

	      break;
	      
	    }

	    // The last atoms match, one less atoms to find

	    m--;

	    if(m == 0) {

	      // All candidate rule atoms found, remove rule
	      
	      rules[i] = (unsigned long *)NULL;
	      num_removed++;
	      break;
	      
	    }
	    
	  }
	  
	}

	if(rules[i] == NULL) {
	  break;
	}
	
      }
      
    }

    if(num_removed > 0) {

      // Compress rule pointer array by removing unneeded ones
      
      j = 0;
      for(i = 0; i < num_rules; i++) {
	t = rules[i];

	if(t != NULL) {
	  rules[j] = t;
	  j++;
	}
      }
    }
    
    num_rules = num_rules - num_removed;
	
  }
	  
  return num_rules;

}

void Unf2Smo::CreateDeadlockProgram(BOOL debugging)
{
  Api * api;
  Edge * edge;
  Edge * preset;
  BITARRAY * needed_conditions;
  BITARRAY * preset_array;
  BITARRAY * postset_array;
  BITARRAY * seen_atoms;
  UL_STACK * stack;
  SIMPLEHASH * hash;
  BOOL only_bottom_atom_seen;
  unsigned long * created_rules;
  unsigned long * created;
  unsigned long * rule_start;
  unsigned long ** sorted_rules;
  unsigned long preset_event;
  unsigned long preset_condition;
  unsigned long preset_atom;
  unsigned long postset_event;
  unsigned long replacing_event;
  unsigned long num_rules;
  unsigned long rule_size;
  unsigned long bottom_atom_index;
  unsigned long i;
  unsigned long j;
    
  static char buf[256];

  created_rules = (unsigned long *) NULL;
  created = (unsigned long *) NULL;
  rule_start = (unsigned long *) NULL;
  sorted_rules = (unsigned long **) NULL;
  hash = (SIMPLEHASH *) NULL;
  
  // A bitmap to mark the needed conditions.
  // (Used by optimizations.)
  
  needed_conditions = CreateBitArray(braproc->num_conditions);
  replaced_events = CreateBitArray(braproc->num_events);

  if(optimize_remove_conditions == FALSE) {

    // No optimizations.
    
    // All except cutoff postsets will be created.
    
    for(i = 1; i <= braproc->num_conditions; i++) {

      if(optimize_use_deadlock_property != FALSE &&
	 braproc->IsDeadlockTranslationCondition(i) == FALSE) {

	// This condition is not needed in deadlock checking
	
	continue;

      }

      if(braproc->IsCutoffPostset(i) == FALSE) {
	SetBit(needed_conditions, (i -1));
      }
      
    }

  } else {

    // Add only conditions which are needed by the liveness
    // property to be checked, and which can't be replaced
    // by the generating event.

    // Check the conditions needed by the liveness rules,
    // which can't be replaced by the event itself.

    for(i = 1; i <= braproc->num_events; i++) {

      // Do this for all events (including cutoff events)

      for(edge = (braproc->events + i)->preset;
	  edge != NULL;
	  edge = edge->nextcondition) {

	if(optimize_use_deadlock_property != FALSE &&
	   braproc->IsDeadlockTranslationCondition(edge->condition->numname) == FALSE) {

	  // This condition is not needed in deadlock checking
	
	  continue;

	}
      
	// Cutoff presets always included.
	
	if(braproc->IsCutoff(i) ||
	   optimize_remove_blocking == FALSE ||
	   braproc->EventHasRealConflict(i)) {
	  
	  SetBit(needed_conditions, (edge->condition->numname -1));
	  
	}
	
      }

    }

  }
      
  
  api = new Api(&(smodels->program));

  false_atom = api->new_atom();
  live_atom = api->new_atom();
  if(optimize_remove_bottom_event == FALSE) {
    bottom_event_atom = api->new_atom();
  } else {
    bottom_event_atom = (Atom *) NULL;
  }

  if(debugging != FALSE) {
  
    sprintf(buf, "false");
    api->set_name(false_atom, buf);
    sprintf(buf, "live");
    api->set_name(live_atom, buf);
    if(optimize_remove_bottom_event == FALSE) {
      sprintf(buf, "e0");
      api->set_name(bottom_event_atom, buf);
    }

  }
  
  if(optimize_remove_bottom_event == FALSE) {
    api->begin_rule(BASICRULE);
    api->add_head(bottom_event_atom);
    api->end_rule();
  }

  // TODO: Minimize memory allocation by calloc:ing when needed
  
  atoms = (Atom **) malloc(sizeof(Atom *) * ((2 *braproc->num_events) +
					     (braproc->num_conditions) +
					     2));

  // Initialize the first index to NULL to catch some coding errors

  num_atoms = 0;
  atoms[num_atoms] = NULL;
  num_atoms++;
  atoms[num_atoms] = bottom_event_atom; // This can be (Atom *)NULL
  bottom_atom_index = num_atoms;
  num_atoms++;
  
  events = (unsigned long *) malloc(sizeof(unsigned long) *
				    (braproc->num_events));
  if(optimize_use_choice_rules == FALSE) {
    blocked_events = (unsigned long *) malloc(sizeof(unsigned long) *
					      (braproc->num_events));
  }
  conditions = (unsigned long *) malloc(sizeof(unsigned long) *
					(braproc->num_conditions));

  replacement_events = (unsigned long *) malloc(sizeof(unsigned long) *
						(braproc->num_events));


  // Add the needed atoms (events, blocking atoms, conditions)
    
  for(i = 1; i <= braproc->num_events; i++) {

    events[i -1] = 0;
    if(optimize_use_choice_rules == FALSE) {
      blocked_events[i -1] = 0;
    }
    replacement_events[i -1] = 0;
    
    if(optimize_use_deadlock_property != FALSE &&
       braproc->IsDeadlockTranslationEvent(i) == FALSE) {

      // This event is not needed in deadlock checking
	
      continue;

    }
      
    if(braproc->IsCutoff(i) == FALSE) {
	
      // Only non-cutoff events needed in this translation

      if(optimize_replace_events_by_events != FALSE) {

	// If we have an event which has only one event
	// in the preset of its preset, and has no events
	// conflicting with it, then this event is not needed.
	// It can then be replaced by its prepreset event.

	if(braproc->EventHasSingletonPrepreset(i) == FALSE &&
	   braproc->EventHasRealConflict(i) == FALSE) {
	
	  // Do special case handling for events which have
	  // several events in their prepreset, all of which have
	  // been replaced by the same event.

	  edge = (braproc->events + i)->preset;
	  preset = edge->condition->preset;
	  
	  if(preset != NULL) {
	    preset_event = preset->event->numname;
	  } else {
	    preset_event = 0;
	  }

	  if(preset_event != 0 &&
	     GetBit(replaced_events, preset_event -1)) {
	    
	    preset_event = replacement_events[preset_event -1];

	  }

	  replacing_event = preset_event;
	  
	  for( ; edge != NULL; edge = edge->nextcondition) {

	    preset = edge->condition->preset;

	    if(preset != NULL) {
	      preset_event = preset->event->numname;
	    } else {
	      preset_event = 0;
	    }

	    if(preset_event != 0 &&
	       GetBit(replaced_events, preset_event -1)) {

	      preset_event = replacement_events[preset_event -1];

	    }

	    if(preset_event != replacing_event) {

	      replacing_event = preset_event+1;
	      break;
	  
	    }

	  }

	  if(replacing_event == preset_event &&
	     replacing_event < i) {

	    // All Prepreset events have been replaced by the same event,
	    // replace also this by the replacing event.

	    replacement_events[i -1] = replacing_event;
	    if(replacing_event != 0) {
	      events[i -1] = events[replacing_event -1];
	    } else {
	      events[i -1] = bottom_atom_index;
	    }
	    SetBit(replaced_events, (i -1));
	    continue;
	    
	  }

	}

	
	if(braproc->EventHasSingletonPrepreset(i) != FALSE &&
	   braproc->EventHasRealConflict(i) == FALSE) {

	  if((braproc->events +i)->preset->condition->preset == NULL) {
	    
	    replacement_events[i -1] = 0;
	    events[i -1] = bottom_atom_index;
	    SetBit(replaced_events, (i -1));
	    continue;
	    
	  } else {

	    if((braproc->events +i)->preset->condition->preset->event->numname < i) {

	      // This event is done.

	      replacing_event = (braproc->events +i)->preset->condition
		->preset->event->numname;

	      // It might be the case that the replacing event is itself replaced.

	      if(replacing_event != 0 &&
		 GetBit(replaced_events, (replacing_event -1))) {

		// In that case follow the pointer.
		replacing_event = replacement_events[replacing_event -1];

	      }
	      
	      replacement_events[i -1] = replacing_event;
	      if(replacing_event != 0) {
		events[i -1] = events[replacing_event -1];
	      } else {
		events[i -1] = bottom_atom_index;
	      }
	      SetBit(replaced_events, (i -1));
	      continue;
	      
	    } else {

	      // Event has a smaller numbered event in it's preset,
	      // skip this optimization.
	      //
	      // (This shouldn't happen in nets created by PEP.
	      //  It's not fatal if it does, only an optimization is missed.
	      //  If this happens often it can be fixed. - kepa@cc.hut.fi)
	      
	      ;
	      
	    }
	    
	    
	  }
	  
	}

      }


      atoms[num_atoms] = api->new_atom();
      events[i -1] = num_atoms;
      num_atoms++;
  
      if(debugging != FALSE) {
  
	sprintf(buf, "e%ld", i);
	api->set_name(atoms[events[i -1]], buf);

      }
	
      if((optimize_use_choice_rules == FALSE) &&
	 (optimize_remove_blocking == FALSE ||
	  braproc->EventHasRealConflict(i))) {
	
	atoms[num_atoms] = api->new_atom();
	blocked_events[i -1] = num_atoms;
	num_atoms++;

	if(debugging != FALSE) {
  
	  sprintf(buf, "be%ld", i);
	  api->set_name(atoms[blocked_events[i -1]], buf);

	}
	  
      }

    }

  }


  // Add the conditions
  
  for(i = 1; i <= braproc->num_conditions; i++) {

    conditions[i -1] = 0;
    
    if(GetBit(needed_conditions, (i-1))) {

      if(optimize_replace_conditions_by_events == FALSE ||
	 braproc->ConditionHasEmptyRealPostset(i) == FALSE) {

	edge = ((braproc->conditions)+i)->postset;

	if(edge != NULL) {

	  postset_event = edge->event->numname;

	  if(postset_event != 0 &&
	     GetBit(replaced_events, postset_event -1)) {

	    // Postset event replaced, this condition isn't needed
	    // after all.

	    continue;

	  }

	}

	atoms[num_atoms] = api->new_atom();
	conditions[i -1] = num_atoms;
	num_atoms++;
      
	if(debugging != FALSE) {
  
	  sprintf(buf, "b%ld", i);
	  api->set_name(atoms[conditions[i -1]], buf);

	}
	  
      } else {

	// Replace condition by it's preset event.
	// (Condition exists when it's preset is in configuration.)

	preset = (braproc->conditions +i)->preset;

	if(preset != NULL) {
	  preset_event = preset->event->numname;
	  conditions[i -1] = events[preset_event -1];  // Note: This might be bottom_atom_index
	} else {
	  conditions[i-1] = bottom_atom_index;
	}
	
      }
	
    }

  }



  // Add basic rules about presets and blocking.

  stack = CreateULStack();
  preset_array = CreateBitArray((braproc->num_events) +1);

  for(i = 1; i <= braproc->num_events; i++) {

    if(events[i -1] != 0 &&
       !GetBit(replaced_events, (i -1))) {

      // Do this only for non-cutoff events

      for(edge = (braproc->events + i)->preset;
	  edge != NULL;
	  edge = edge->nextcondition) {

	preset = edge->condition->preset;

	if(preset != NULL) {
	  preset_event = preset->event->numname;
	} else {
	  preset_event = 0;
	}

	if(preset_event != 0 &&
	   GetBit(replaced_events, preset_event -1)) {
	  preset_event = replacement_events[preset_event -1];
	}

	if(optimize_remove_bottom_event != FALSE &&
	   preset_event == 0) {

	  // Remove the bottom event atom from presets
	  
	  continue;
	  
	}
	
	// Remove duplicates and reverse preset order for a nicer
	// appereance in printout.
	  
	if(!GetBit(preset_array, preset_event)) {

	  SetBit(preset_array, preset_event);
	  PushULStack(stack, preset_event);
	}
	    
      }

      if(optimize_use_choice_rules == FALSE ||
	 braproc->EventHasRealConflict(i) == FALSE) {
	 api->begin_rule(BASICRULE);
      } else {
	 api->begin_rule(CHOICERULE);
      }

      api->add_head(atoms[events[i -1]]);
	
      while(!ULStackEmpty(stack)) {

	PopULStack(stack, preset_event);
	ClearBit(preset_array, preset_event);

	// Add preset as positive
	  
	if(preset_event == 0) {
	  api->add_body(bottom_event_atom, TRUE);
	} else {
	  api->add_body(atoms[events[preset_event -1]], TRUE);
	}
	  
      }  

      // Add blocking condition as negative
      // Skip this if optimizing and the event has no real conflicts.
      
      if((optimize_use_choice_rules == FALSE) &&
	 (optimize_remove_blocking == FALSE ||
	  braproc->EventHasRealConflict(i))) {
	
	api->add_body(atoms[blocked_events[i -1]], FALSE);
	api->end_rule();

	// Add the rule for blocking condition

	api->begin_rule(BASICRULE);
	api->add_head(atoms[blocked_events[i -1]]);
	api->add_body(atoms[events[i -1]], FALSE);
      } 

      api->end_rule();

    }
      
  }



  // Add rules about conflict sets.

  // Create a hash table if duplicate rules need to be
  // removed.

  // TODO: This hash table is created "Big enough",
  //       add dynamic memory management!
  
  if(optimize_remove_duplicate_rules != FALSE) {
    hash = CreateSimpleHash(braproc->num_conditions);
  }
  
  // Create a bit-array to find duplicates within rules.
  // (Replacing events by events might create duplicates.)
 
  postset_array = CreateBitArray((braproc->num_events) +1);


  // Allocate memory to store already created
  // rules.

  // TODO: Add dynamic memory management.
  
  created_rules = (unsigned long *)
    malloc((sizeof(unsigned long)) *
	   (braproc->num_edges +     // Storage for all edges
	    braproc->num_conditions  // Storage for rule size
	    ));

  created = created_rules;
  num_rules = 0;
  
  for(i = 1; i <= braproc->num_conditions; i++) {

    if(conditions[i -1] != 0 &&
       braproc->ConditionHasRealConflict(i)) {

      rule_size = 0;
	
      for(edge = ((braproc->conditions)+i)->postset;
	  edge != NULL;
	  edge = edge->nextevent) {

	if(braproc->IsCutoff(edge->event->numname) != FALSE) {

	  // Skip cutoffs.
	    
	  continue;
	    
	}

	postset_event = edge->event->numname;

	if(postset_event != 0 &&
	   GetBit(replaced_events, postset_event -1)) {
	  
	  postset_event = replacement_events[postset_event -1];

	}
	
	if(optimize_remove_bottom_event != FALSE &&
	   postset_event == 0) {

	  // Remove the bottom event atom from presets
	  // (Shouldn't happen, but is not fatal.)
	  
	  continue;
	  
	}

	if(postset_event != 0 &&
	   events[postset_event -1] == 0) {

	  // This postset event has been removed!
	  
	  continue;

 	}
	
	// Remove duplicates and reverse postset order for a nicer
	// appereance in printout.
	  
	if(!GetBit(postset_array, postset_event)) {

	  SetBit(postset_array, postset_event);
	  PushULStack(stack, postset_event);
	  rule_size++;
	}
	
      }

      if(rule_size < 1) {

	// This might happen due to event to event replacement.

	if(rule_size == 0) {

	  // Shouldn't happen, but is non-fatal.
	  
	  continue;

	}
	
	PopULStack(stack, postset_event);
	ClearBit(postset_array, postset_event);
	continue;
	
      }
      
      rule_start = created;
      *created++ = rule_size;

      while(!ULStackEmpty(stack)) {

	PopULStack(stack, postset_event);
	ClearBit(postset_array, postset_event);

	*created++ = postset_event;
	
      }  

      // This condition has a real conflict in it's postset.
      // Add a conflict rule.

      if(optimize_remove_duplicate_rules != FALSE ||
	 optimize_remove_sub_and_supersets != FALSE) {

	// First sort the rule atoms so that rules will be
	// canonical.
	//
	// I use insertion sort instead of heap sort because
	// the rules will be almost sorted. Only through the
	// optimizations can some out of order elements be added.

	InsertionSort(rule_start[0], &(rule_start[1]));
	
	if(optimize_remove_duplicate_rules != FALSE) {
	  
	  if(InsertDataTellIfNew(hash, rule_start) == FALSE) {

	    // This is a duplicate of some earlier rule.
	    // Try to generate a new one with better luck.
	  
	    created = rule_start;
	    continue;

	  }

	}

      }

      num_rules++;

    }

  }


  if(optimize_sort_rules != FALSE ||
     optimize_remove_sub_and_supersets != FALSE) {

    sorted_rules = (unsigned long **)malloc(num_rules *
					    (sizeof(unsigned long *)));

    rule_start = created_rules;

    for(i = 0; i < num_rules; i++) {

      sorted_rules[i] = rule_start;

      rule_start = &(rule_start[rule_start[0] +1]);

    }

    if(optimize_remove_sub_and_supersets != FALSE) {

      num_rules = OptimizeRemoveSubAndSupersets(num_rules,
						sorted_rules,
						FALSE);

    }
    
    if(optimize_sort_rules != FALSE) {
      
      HeapSortRules(num_rules, sorted_rules);

    }
    
  }


  // Feed all conflict rules to smodels
  
  rule_start = created_rules;
    
  for(i = 0; i < num_rules; i++) {

    if(optimize_sort_rules != FALSE ||
       optimize_remove_sub_and_supersets != FALSE) {

      rule_start = sorted_rules[i];

    }

    // Feed the rule to smodels.

    j = *rule_start++;
    
    if(j == 2) {

      api->begin_rule(BASICRULE);

    } else {

      api->begin_rule(CONSTRAINTRULE);
      api->set_atleast_body(2, TRUE);
      
    }
      
    api->add_head(false_atom);

    for( ; j > 0; j--) {

      api->add_body(atoms[events[(*rule_start++) -1]], TRUE);
	  
    }
      
    api->end_rule();
      
  }

  
  if(optimize_sort_rules != FALSE) {

    free(sorted_rules);
    sorted_rules = (unsigned long **)NULL;

  }


  if(optimize_remove_duplicate_rules != FALSE) {

    ClearSimpleHash(hash);

  }


  // Add rules to model the conditions.

  for(i = 1; i <= braproc->num_conditions; i++) {

    if(conditions[i -1] != 0) {

      if(optimize_remove_bottom_event != FALSE &&
	 conditions[i -1] == bottom_atom_index) {

	// Condition is always true, and can be remove from program
	
	continue;
	
      }

      if(optimize_replace_conditions_by_events == FALSE ||
	 braproc->ConditionHasEmptyRealPostset(i) == FALSE) {

	// Only add rules for those conditions which are needed.

	api->begin_rule(BASICRULE);
	api->add_head(atoms[conditions[i -1]]);

	if((braproc->conditions + i)->preset == NULL) {

	  if(optimize_remove_bottom_event == FALSE) {
	    api->add_body(bottom_event_atom, TRUE);
	  }
	  
	} else {

	  if (optimize_remove_bottom_event == FALSE ||
	      events[((braproc->conditions + i)
		      ->preset->event->numname) -1] !=
	      bottom_atom_index) {
	    
	    api->add_body(atoms[events[((braproc->conditions + i)
					->preset->event->numname) -1]],
			  TRUE);
	  
	  }
	  
	}

	
	for(edge = ((braproc->conditions)+i)->postset;
	    edge != NULL;
	    edge = edge->nextevent) {

	  if(braproc->IsCutoff(edge->event->numname) != FALSE) {

	    // Skip cutoffs.
	    
	    continue;
	    
	  }

	  postset_event = edge->event->numname;
	  
	  if(postset_event != 0 &&
	     GetBit(replaced_events, postset_event -1)) {
	    
	    postset_event = replacement_events[postset_event -1];
	    
	  }

	  if(optimize_remove_bottom_event != FALSE &&
	     postset_event == 0) {

	    // Remove the bottom event atom from postsets. (Shouldn't happen.)
	  
	    continue;
	  
	  }
	  
	  if(postset_event != 0 &&
	     events[postset_event -1] == 0) {

	    // This postset event has been removed!
	  
	    continue;

	  }
	  
	  // Remove duplicates and reverse postset order for a nicer
	  // appereance in printout.
	  
	  if(!GetBit(postset_array, postset_event)) {

	    SetBit(postset_array, postset_event);
	    PushULStack(stack, postset_event);
	    
	  }
	    
	}

	while(!ULStackEmpty(stack)) {

	  PopULStack(stack, postset_event);
	  ClearBit(postset_array, postset_event);
	  
	  api->add_body(atoms[events[postset_event -1]], FALSE);
	  
	}  

	api->end_rule();

      }
    }	
  }
    

  // Add liveness rules.


  // Reuse the same memory as used before for conflicts
  
  created = created_rules;
  num_rules = 0;
  
  // We will use atoms instead of conditions because conditions
  // might have been replaced by events.

  seen_atoms = CreateBitArray(num_atoms);

  for(i = 1; i <= braproc->num_events; i++) {

    // Do this for all events (including cutoff events)
    // Skip this if optimizing and the event has no real conflicts,
    // and it is not a cutoff event.
    // (It will not be needed to forced forward because it can't block.)

    // Cutoff presets always included.
    
    only_bottom_atom_seen = TRUE;

    if(braproc->IsCutoff(i) ||
       (events[i -1] != 0 &&
	(optimize_remove_blocking == FALSE ||
	 braproc->EventHasRealConflict(i)))) {

      if(i != 0 &&
	 GetBit(replaced_events, i -1)) {

	// No liveness rules for replaced events.

	continue;

      }

      rule_size = 0;
      
      for(edge = (braproc->events + i)->preset;
	  edge != NULL;
	  edge = edge->nextcondition) {

	// We reverse the postset order so it will be alphabetical
	// (nicer looking) in the printout.

	preset_condition = edge->condition->numname;

	if(conditions[preset_condition -1] == 0) {
	  only_bottom_atom_seen = FALSE;
	  continue;
	}
	
	if(optimize_remove_bottom_event != FALSE &&
	   conditions[preset_condition -1] == bottom_atom_index) {

	  // Remove the unneeded bottom event atom
	  
	  continue;
	  	  
	}
	
	// Remove duplicates within the rules if ones exist.
	// (Replacing conditions by events might create duplicates.)
	
	if(!GetBit(seen_atoms, conditions[preset_condition -1])) {

	  SetBit(seen_atoms, conditions[preset_condition -1]);
	  PushULStack(stack, conditions[preset_condition -1]);
	  rule_size++;
	}
	
      }


      if(rule_size == 0 &&
	 only_bottom_atom_seen == FALSE) {

	// This might happen if a cutoff preset is replaced by
	// the preset event.

	// The correctness of this case is tricky and needs
	// to be proved by hand. (Replacing cutoff preset condition
	// with its preset event.)
	
	// Note: Do not remove those liveness rules which have
	// all preset conditions replaced by the bottom event.
	
	continue;

      }

      rule_start = created;
      *created++ = rule_size;
      
      while(!ULStackEmpty(stack)) {

	PopULStack(stack, preset_atom);
	ClearBit(seen_atoms, preset_atom);

	*created++ = preset_atom;
	
      }  

      if(optimize_remove_duplicate_rules != FALSE ||
	 optimize_remove_sub_and_supersets != FALSE) {

	// First sort the rule atoms so that rules will be
	// canonical.
	//
	// I use insertion sort instead of heap sort because
	// the rules will be almost sorted. Only through the
	// optimizations can some out of order elements be added.

	InsertionSort(rule_start[0], &(rule_start[1]));
	
	if(optimize_remove_duplicate_rules != FALSE) {

	  if(InsertDataTellIfNew(hash, rule_start) == FALSE) {

	    // This is a duplicate of some earlier rule.
	    // Try to generate a new one with better luck.
	  
	    created = rule_start;
	    continue;

	  }

	}

	
      }


      num_rules++;

    }

  }

  if(optimize_sort_rules != FALSE ||
     optimize_remove_sub_and_supersets != FALSE) {
    
    sorted_rules = (unsigned long **)malloc(num_rules *
					    (sizeof(unsigned long *)));

    rule_start = created_rules;

    for(i = 0; i < num_rules; i++) {

      sorted_rules[i] = rule_start;

      rule_start = &(rule_start[rule_start[0] +1]);

    }

    if(optimize_remove_sub_and_supersets != FALSE) {

      num_rules = OptimizeRemoveSubAndSupersets(num_rules,
						sorted_rules,
						TRUE);

    }
       
    if(optimize_sort_rules != FALSE) {
      
      HeapSortRules(num_rules, sorted_rules);
      
    }
    
  }


  // Feed all liveness rules to smodels
  
  rule_start = created_rules;
    
  for(i = 0; i < num_rules; i++) {

      if(optimize_sort_rules != FALSE ||
	 optimize_remove_sub_and_supersets != FALSE) {

	rule_start = sorted_rules[i];

      }

      // Feed the rule to smodels.

      api->begin_rule(BASICRULE);
      api->add_head(live_atom);
      
      for(j = *rule_start++; j > 0; j--) {

	api->add_body(atoms[*rule_start++], TRUE);
	  
      }
      
      api->end_rule();

  }


  if(optimize_sort_rules != FALSE) {

    free(sorted_rules);
    sorted_rules = (unsigned long **)NULL;

  }
  
  //if(optimize_remove_duplicate_rules != FALSE) {
  //
  //   ClearSimpleHash(hash);
  //
  //}

  // Add the compute statement
    
  api->set_compute(false_atom, FALSE);
  api->set_compute(live_atom, FALSE);
    
  api->done();

//  fprintf(stdout, "%% Program done!\n");
//  fflush(stdout);
  
  smodels->init();

  DeleteBitArray(seen_atoms);

  if(hash != NULL) {
    DeleteSimpleHash(hash);
  }
  
  DeleteBitArray(postset_array);
  DeleteBitArray(preset_array);
  DeleteULStack(stack);

  delete(api);

  DeleteBitArray(needed_conditions);

  if(created_rules != NULL) {
    free(created_rules);
  }
  
}


BOOL Unf2Smo::ComputeModel(void)
{
  int model;

  model = smodels->model(TRUE, FALSE);

  if(model) {

    return TRUE;

  } else {

    return FALSE;

  }

}


BOOL Unf2Smo::CreateDeadlockConfiguration(void)

{

  unsigned long i;
  unsigned long enabled_cutoff;

  // Drive tokens down until the net deadlocks.
  // This is guaranteed if the translation is correct,
  // we have some error checking for translation
  // bugs in the code below.

  current_configuration = CreateBitArray(braproc->num_events);

  // Find out the current configuration.

  for(i = 1; i <= braproc->num_events; i++) {

    if(events[i -1] != 0) {

      if(atoms[events[i -1]] == bottom_event_atom ||
	 atoms[events[i -1]]->Bpos) {

	SetBit(current_configuration, (i -1));

      }

    }

  }

  braproc->AddPresetEvents(current_configuration);
  enabled_cutoff = braproc->ContinueToDeadlock(current_configuration);

  if(enabled_cutoff != 0) {

    fprintf(stdout, "Deadlock configuration sanity check failed:\n");
    fprintf(stdout, "Cutoff event E%ld is still enabled!\n", enabled_cutoff);
    fprintf(stdout, "Please contact the author by email: Keijo.Heljanko@hut.fi\n");
    fflush(stdout);
    fprintf(stdout, "\nThe error occurred in the non-deadlocking configuration:\n");

    PrintConfiguration();
    return FALSE;

  }

  PrintConfiguration();
  return TRUE;

}


void Unf2Smo::PrintProgram(void)
{

  smodels->program.print();

}  

void Unf2Smo::PrintStatistics(void)
{

  fprintf(stdout, "Number of choice points: %ld\n",
	  smodels->number_of_choice_points);
  fprintf(stdout, "Number of wrong choices: %ld\n",
	  smodels->number_of_wrong_choices);
  fprintf(stdout, "Number of atoms: %ld\n",
	  smodels->program.number_of_atoms);
  fprintf(stdout, "Number of rules: %ld\n",
	  smodels->program.number_of_rules);
  fprintf(stdout, "Number of picked atoms: %ld\n",
	  smodels->number_of_picked_atoms);
  fprintf(stdout, "Number of forced atoms: %ld\n",
	  smodels->number_of_forced_atoms);
  fprintf(stdout, "Number of truth assignments: %ld\n",
	  smodels->number_of_assignments);
}


void Unf2Smo::PrintConfiguration(void)

{

  unsigned long i;
  unsigned long num_events;
  unsigned long tr_numname;
  const char *format;

  num_events = 0;
  format = (const char *)NULL;
  
  for(i = 1; i <= braproc->num_events; i++) {

    if(GetBit(current_configuration, (i -1))) {

      if(num_events == 0) {
	format = "E%ld(%s)";
      } else if (num_events & 0x1) {
	format = ", E%ld(%s)";
      } else {
	format = ",\n  E%ld(%s)";
      }

      tr_numname = (braproc->events +i)->tr_numname;
      fprintf(stdout, format, i, braproc->tr_names[tr_numname -1]);

      num_events++;

    }

  }

  if(num_events > 0) {
    fprintf(stdout, ".");
  }

  fprintf(stdout, "\n");
  fflush(stdout);
  
}
