// Copyright 1998 by Patrik Simons
// This software is provided as is, no warranty of any kind is given.
//
// File:   atomrule.cc
// Author: Patrik Simons
//
#include <iostream.h>
#ifdef USEDOUBLE
#include <math.h>
#endif
#include "atomrule.h"
#include "program.h"

Atom::Atom ()
{
  endHead = 0;
  endPos = 0;
  endNeg = 0;
  endPos2 = 0;
  endNeg2 = 0;
  end = 0;
  headof = 0;
  head = 0;
  pos = 0;
  neg = 0;
  source = 0;
  posScore = 0;
  negScore = 0;
  name = 0;
  closure = false;
  Bpos = false;
  Bneg = false;
  visited = false;
  guess = false;
  isnant = false;
  dependsonTrue = false;
  dependsonFalse = false;
  computeTrue = false;
  computeFalse = false;
  backtracked = false;
  forced = false;
  in_etrue_queue = false;
  in_efalse_queue = false;
}

Atom::~Atom ()
{
  delete[] head;
  delete[] name;
}

void 
Atom::backchainTrue ()
{
  Follows *f;
  // Find the only active rule.
  for (f = head; f != endHead; f++)
    if (!f->r->isInactive ())
      break;
  f->r->backChainTrue ();
}

//
// Note that since backchain depends on the truth values of the
// atoms in the body of the rules, we must set Bpos first.
// To keep everything safe we also handle inactivation before 
// we fire any rules.
//
void
Atom::setBTrue ()
{
  Follows *f;
  Bpos = true;
  for (f = neg; f != endNeg; f++)
    f->r->inactivateInFalse (f->a);
  for (f = pos; f != endPos; f++)
    f->r->mightFireTrue (f->a);
  // This atom could have been forced.
  if (headof == 1)
    backchainTrue ();
}
 
void
Atom::setBFalse ()
{
  Follows *f;
  Bneg = true;
  for (f = pos; f != endPos; f++)
    f->r->inactivateInTrue (f->a);
  for (f = neg; f != endNeg; f++)
    f->r->mightFireFalse (f->a);
  closure = false;
  source = 0;
  // This atom could have been forced.
  if (headof)          // There are active rules.
    backchainFalse (); // Might backchain already backchained rules
		       // in mightFireFalse.
}

void
Atom::setTrue ()
{
  for (Follows *f = pos; f != endPos; f++)
    f->r->mightFireInitTrue (f->a);
  for (Follows *f = neg; f != endNeg; f++)
    f->r->mightFireInitFalse (f->a);
  closure = true;
}

void
Atom::backtrackFromBTrue ()
{
  Follows *f;
  for (f = pos; f != endPos; f++)
    f->r->backtrackFromTrueInPos (f->a);
  for (f = neg; f != endNeg; f++)
    f->r->backtrackFromTrueInNeg (f->a);
  Bpos = false;
  closure = true;
}

void
Atom::backtrackFromBFalse ()
{
  Follows *f;
  for (f = neg; f != endNeg; f++)
    f->r->backtrackFromFalseInNeg (f->a);
  for (f = pos; f != endPos; f++)
    f->r->backtrackFromFalseInPos (f->a);
  Bneg = false;
  closure = true;
}

void 
Atom::backchainFalse ()
{
  for (Follows *f = head; f != endHead; f++)
    f->r->backChainFalse ();
}

void
Atom::reduce_head ()
{
  Follows t;
  Follows *g = head;
  for (Follows *f = head; f != endHead; f++)
    {
      f->r->swapping (f, g);
      t = *g;
      *g = *f;
      *f = t;
      if (!g->r->isInactive ())
	g++;
    }
  endHead = g;
}

void
Atom::reduce_pbody (bool strong)
{
  Follows t;
  Follows *g = pos;
  for (Follows *f = pos; f != endPos; f++)
    {
      f->r->swapping (f, g);
      t = *g;
      *g = *f;
      *f = t;
      if ((strong == false || Bpos == false) &&
	  Bneg == false && !g->r->isInactive ())
	g++;
    }
  endPos = g;
  endPos2 = g;
}

void
Atom::reduce_nbody (bool strong)
{
  Follows t;
  Follows *g = neg;
  for (Follows *f = neg; f != endNeg; f++)
    {
      f->r->swapping (f, g);
      t = *g;
      *g = *f;
      *f = t;
      // Only WeightRule must retain BPos == true when strong == false.
      if (((strong == false && f->r->type == WEIGHTRULE) ||
	   Bpos == false) &&
	  Bneg == false && !g->r->isInactive ())
	g++;
    }
  endNeg = g;
  g = neg;
  for (Follows *f = neg; f != endNeg; f++)
    if (f->r->type == WEIGHTRULE)
      {
	if (f != g)
	  {
	    f->r->swapping (f, g);
	    t = *g;
	    *g = *f;
	    *f = t;
	  }
	g++;
      }
  endNeg2 = g;
}

void
Atom::reduce (bool strongly)
{
  reduce_head ();
  reduce_pbody (strongly);
  reduce_nbody (strongly);
}

void
Atom::unreduce ()
{
  endHead = pos;
  endPos = neg;
  endPos2 = neg;
  endNeg = end;
}

const char *
Atom::atom_name ()
{
  if (name)
    return name;
  else
    return "#noname#";
}

void
Rule::visit (Atom *a)
{
  p->queue.push (a);
  a->visited = true;
}

void
Rule::etrue_push (Atom *a)
{
  if (!a->in_etrue_queue && !a->in_efalse_queue)
    p->equeue.push (a);
  a->in_etrue_queue = true;
  if (a->Bneg || a->in_efalse_queue)
    p->conflict = true;
}

void
Rule::efalse_push (Atom *a)
{
  if (!a->in_etrue_queue && !a->in_efalse_queue)
    p->equeue.push (a);
  a->in_efalse_queue = true;
  if (a->Bpos || a->in_etrue_queue)
    p->conflict = true;
}

inline void
Rule::queue_push (Atom *a)
{
  p->queue.push (a);
}

BasicRule::BasicRule (Program *p0)
  : HeadRule (p0)
{
  head = 0;
  neg = 0;
  pos = 0;
  pos2 = 0;
  nbody = 0;
  pbody = 0;
  nend = 0;
  pend = 0;
  end = 0;
  inactive = 0;
  visited = false;
  type = BASICRULE;
}

BasicRule::~BasicRule ()
{
  delete[] nbody;  // pbody is allocated after nbody
}

bool
BasicRule::isInactive ()
{
  return inactive;
}

bool
BasicRule::isUpperActive ()
{
  return pos2 == 0 && !inactive;
}

void
BasicRule::inactivate (Atom **a)
{
  inactive++;
  if (inactive == 1)
    {
      Atom *b = head;
      b->headof--;
      if (b->Bneg == false)
	{
	  if (b->headof && (b->source == 0 || b->source == this))
	    {
	      b->source = 0;
	      queue_push (b);
	    }
	  if (b->headof == 0)
	    efalse_push (b);
	  else if (b->headof == 1 && b->Bpos && *a != b)
	    b->backchainTrue ();
	}
    }
}

void
BasicRule::inactivateInTrue (Atom **a)
{
  if ((*a)->closure)
    pos2++;
  inactivate (a);
}

void
BasicRule::inactivateInFalse (Atom **a)
{
  inactivate (a);
}

void
BasicRule::fireInit ()
{
  if (pos == 0 && neg == 0)
    {
      etrue_push (head);
      queue_push (head);
      head->source = this;
    }
  else if (pos2 == 0)
    {
      queue_push (head);
      if (head->source == 0)
	head->source = this;
    }
}

void
BasicRule::mightFireInitTrue (Atom **)
{
  pos2--;
  if (pos2 == 0 && !inactive)
    {
      queue_push (head);
      if (head->source == 0)
	head->source = this;
    }
}

void
BasicRule::unInit ()
{
  pos2 = pos;
}

void
BasicRule::mightFire ()
{
  if (pos == 0)
    {
      if (neg == 0 && head->Bpos == false)
	etrue_push (head);
      else if (neg == 1 && head->Bneg)
	backChainFalse ();
    }
  else if (pos == 1 && neg == 0 && head->Bneg)
    backChainFalse ();
}

void
BasicRule::mightFireTrue (Atom **)
{
  pos--;
  mightFire ();
}

void
BasicRule::mightFireFalse (Atom **)
{
  neg--;
  mightFire ();
}

void
BasicRule::backChainTrue ()
{
  Atom **b;
  if (neg)
    for (b = nbody; b != nend; b++)
      if ((*b)->Bneg == false)
	efalse_push (*b);
  if (pos)
    for (b = pbody; b != pend; b++)
      if ((*b)->Bpos == false)
	etrue_push (*b);
}

void
BasicRule::backChainFalse ()
{
  Atom **b;
  if (inactive)
    return;
  if (pos == 0 && neg == 1)
    {
      for (b = nbody; b != nend; b++)
	if ((*b)->Bneg == false)
	  {
	    etrue_push (*b);
	    break;
	  }
    }
  else if (pos == 1 && neg == 0)
    for (b = pbody; b != pend; b++)
      if ((*b)->Bpos == false)
	{
	  efalse_push (*b);
	  break;
	}
}

void
BasicRule::backtrackFromTrueInPos (Atom **)
{
  pos++;
}

void
BasicRule::backtrackFromFalseInPos (Atom **)
{
  pos2--;
  inactive--;
  if (inactive == 0)
    head->headof++;
}

void
BasicRule::backtrackFromTrueInNeg (Atom **)
{
  inactive--;
  if (inactive == 0)
    head->headof++;
}

void
BasicRule::backtrackFromFalseInNeg (Atom **)
{
  neg++;
}

void
BasicRule::propagateFalse (Atom **)
{
  pos2++;
  if (pos2 == 1 && !inactive && head->closure != false &&
      (head->source == 0 || head->source == this))
    {
      head->source = 0;
      queue_push (head);
    }
}

void
BasicRule::propagateTrue (Atom **)
{
  pos2--;
  if (pos2 == 0 && !inactive && head->closure == false)
    {
      if (head->source == 0)
	head->source = this;
      queue_push (head);
    }
}

void
BasicRule::backtrackUpper (Atom **)
{
  pos2--;
}

void
BasicRule::search (Atom *)
{
  if (!head->visited && head->Bneg == false)
    visit (head);
  for (Atom **a = nbody; a != nend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      visit (*a);
  for (Atom **a = pbody; a != pend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      visit (*a);
  visited = true;
}

void
BasicRule::reduce (bool strongly)
{
  Atom **a;
  Atom **b;
  Atom *t;
  b = nbody;
  for (a = nbody; a != nend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((*b)->Bpos == false && (*b)->Bneg == false)
	b++;
    }
  nend = b;
  b = pbody;
  for (a = pbody; a != pend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((strongly == false || (*b)->Bpos == false)
	  && (*b)->Bneg == false)
	b++;
    }
  pend = b;
}

void
BasicRule::unreduce ()
{
  nend = pbody;
  pend = end;
}

void
BasicRule::setup ()
{
  head->head->r = this;
  head->head->a = &head;
  head->head++;
  Atom **a;
  for (a = nbody; a != nend; a++)
    {
      (*a)->negScore--;
      (*a)->neg[(*a)->negScore].r = this;
      (*a)->neg[(*a)->negScore].a = a;
    }
  for (a = pbody; a != pend; a++)
    {
      (*a)->posScore--;
      (*a)->pos[(*a)->posScore].r = this;
      (*a)->pos[(*a)->posScore].a = a;
    }
}

void
BasicRule::print ()
{
  cout << head->atom_name ();
  if (nbody)
    cout << " :- ";
  Atom **a;
  int comma = 0;
  for (a = pbody; a != pend; a++)
    {
      if (comma)
	cout << ", ";
      cout << (*a)->atom_name ();
      comma = 1;
    }
  for (a = nbody; a != nend; a++)
    {
      if (comma)
	cout << ", ";
      cout << "not " << (*a)->atom_name ();
      comma = 1;
    }
  cout << '.' << endl;
}


ConstraintRule::ConstraintRule (Program *p0)
  : HeadRule (p0)
{
  head = 0;
  neg = 0;
  pos = 0;
  pos2 = 0;
  nbody = 0;
  pbody = 0;
  nend = 0;
  pend = 0;
  end = 0;
  inactivePos = 0;
  inactiveNeg = 0;
  visited = false;
  type = CONSTRAINTRULE;
}

ConstraintRule::~ConstraintRule ()
{
  delete[] nbody;
}

bool
ConstraintRule::isInactive ()
{
  return inactivePos > 0 || inactiveNeg > 0;
}

bool
ConstraintRule::isUpperActive ()
{
  return pos2 <= 0 && !isInactive ();
}

void
ConstraintRule::inactivate (Atom **a)
{
  Atom *b = head;
  b->headof--;
  if (b->Bneg == false)
    {
      if (b->headof && (b->source == 0 || b->source == this))
	{
	  b->source = 0;
	  queue_push (b);
	}
      if (b->headof == 0)
	efalse_push (b);
      else if (b->headof == 1 && b->Bpos && *a != b)
	b->backchainTrue ();
    }
}

void
ConstraintRule::inactivateInTrue (Atom **a)
{
  if ((*a)->closure)
    pos2++;
  inactivePos++;
  if (inactivePos == 1 && inactiveNeg <= 0)
    inactivate (a);
}

void
ConstraintRule::inactivateInFalse (Atom **a)
{
  inactiveNeg++;
  if (inactiveNeg == 1 && inactivePos <= 0)
    inactivate (a);
}

void
ConstraintRule::fireInit ()
{
  if (pos <= 0 && neg <= 0)
    {
      etrue_push (head);
      queue_push (head);
      head->source = this;
    }
  else if (pos2 <= 0)
    {
      queue_push (head);
      if (head->source == 0)
	head->source = this;
    }
}

void
ConstraintRule::mightFireInitTrue (Atom **)
{
  pos2--;
  if (pos2 == 0 && !isInactive ())
    {
      queue_push (head);
      if (head->source == 0)
	head->source = this;
    }
}

void
ConstraintRule::unInit ()
{
  pos2 = pos;
}

void
ConstraintRule::mightFireTrue (Atom **)
{
  pos--;
  if (pos == 0)
    {
      if (neg <= 0 && head->Bpos == false)
	etrue_push (head);
      else if (neg == 1 && head->Bneg)
	backChainFalse ();
    }
  else if (pos == 1 && neg <= 0 && head->Bneg)
    backChainFalse ();
}

void
ConstraintRule::mightFireFalse (Atom **)
{
  neg--;
  if (neg <= 0)
    {
      if (pos <= 0 && head->Bpos == false)
	etrue_push (head);
      else if (pos == 1 && head->Bneg)
	backChainFalse ();
    }
  else if (neg == 1 && pos <= 0 && head->Bneg)
    backChainFalse ();
}

void
ConstraintRule::backChainTrue ()
{
  if (inactivePos != 0 && inactiveNeg != 0)
    return;
  Atom **b;
  if (neg && inactiveNeg == 0)
    for (b = nbody; b != nend; b++)
      if ((*b)->Bneg == false && (*b)->Bpos == false)
	efalse_push (*b);
  if (pos && inactivePos == 0)
    for (b = pbody; b != pend; b++)
      if ((*b)->Bneg == false && (*b)->Bpos == false)
	etrue_push (*b);
}

void
ConstraintRule::backChainFalse ()
{
  Atom **b;
  if (isInactive ())
    return;
  if (pos <= 0 && neg == 1)
    {
      for (b = nbody; b != nend; b++)
	if ((*b)->Bneg == false && (*b)->Bpos == false)
	  etrue_push (*b);
    }
  else if (pos == 1 && neg <= 0)
    for (b = pbody; b != pend; b++)
      if ((*b)->Bneg == false && (*b)->Bpos == false)
	efalse_push (*b);
}

void
ConstraintRule::backtrackFromTrueInPos (Atom **)
{
  pos++;
}

void
ConstraintRule::backtrackFromFalseInPos (Atom **)
{
  pos2--;
  inactivePos--;
  if (inactivePos == 0 && inactiveNeg <= 0)
    head->headof++;
}

void
ConstraintRule::backtrackFromTrueInNeg (Atom **)
{
  inactiveNeg--;
  if (inactiveNeg == 0 && inactivePos <= 0)
    head->headof++;
}

void
ConstraintRule::backtrackFromFalseInNeg (Atom **)
{
  neg++;
}

void
ConstraintRule::propagateFalse (Atom **)
{
  pos2++;
  if (pos2 == 1 && !isInactive () && head->closure != false &&
      (head->source == 0 || head->source == this))
    {
      head->source = 0;
      queue_push (head);
    }
}

void
ConstraintRule::propagateTrue (Atom **)
{
  pos2--;
  if (pos2 == 0 && !isInactive () && head->closure == false)
    {
      if (head->source == 0)
	head->source = this;
      queue_push (head);
    }
}

void
ConstraintRule::backtrackUpper (Atom **)
{
  pos2--;
}

void
ConstraintRule::search (Atom *)
{
  if (!head->visited && head->Bneg == false)
    visit (head);
  for (Atom **a = nbody; a != nend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      visit (*a);
  for (Atom **a = pbody; a != pend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      visit (*a);
  visited = true;
}

void
ConstraintRule::reduce (bool strongly)
{
  Atom **a;
  Atom **b;
  Atom *t;
  b = nbody;
  for (a = nbody; a != nend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((*b)->Bpos == false && (*b)->Bneg == false)
	b++;
    }
  nend = b;
  b = pbody;
  for (a = pbody; a != pend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((strongly == false || (*b)->Bpos == false) &&
	  (*b)->Bneg == false)
	b++;
    }
  pend = b;
}

void
ConstraintRule::unreduce ()
{
  nend = pbody;
  pend = end;
}

void
ConstraintRule::setup ()
{
  head->head->r = this;
  head->head->a = &head;
  head->head++;
  Atom **a;
  for (a = nbody; a != nend; a++)
    {
      (*a)->negScore--;
      (*a)->neg[(*a)->negScore].r = this;
      (*a)->neg[(*a)->negScore].a = a;
    }
  for (a = pbody; a != pend; a++)
    {
      (*a)->posScore--;
      (*a)->pos[(*a)->posScore].r = this;
      (*a)->pos[(*a)->posScore].a = a;
    }
}

void
ConstraintRule::print ()
{
  cout << head->atom_name () << " :- ";
  Atom **a;
  int comma = 0;
  if (pbody != pend)
    {
      long atleast = pos;
      for (a = pbody; a != pend; a++)
	if ((*a)->Bpos)
	  atleast++;
      cout << atleast << " { ";
      for (a = pbody; a != pend; a++)
	{
	  if (comma)
	    cout << ", ";
	  cout << (*a)->atom_name ();
	  comma = 1;
	}
      cout << " }";
    }
  if (nbody != nend)
    {
      if (comma)
	cout << ' ';
      comma = 0;
      long atleast = neg;
      for (a = nbody; a != nend; a++)
	if ((*a)->Bneg)
	  atleast++;
      cout << atleast << " { ";
      for (a = nbody; a != nend; a++)
	{
	  if (comma)
	    cout << ", ";
	  cout << "not " << (*a)->atom_name ();
	  comma = 1;
	}
      cout << " }";
    }
  cout << '.' << endl;
}

ChoiceRule::ChoiceRule (Program *p0)
  : Rule (p0)
{
  head = 0;
  hend = 0;
  neg = 0;
  pos = 0;
  pos2 = 0;
  nbody = 0;
  pbody = 0;
  nend = 0;
  pend = 0;
  end = 0;
  inactive = 0;
  visited = false;
  type = CHOICERULE;
}

ChoiceRule::~ChoiceRule ()
{
  delete[] head;
}

bool
ChoiceRule::isInactive ()
{
  return inactive;
}

bool
ChoiceRule::isUpperActive ()
{
  return pos2 == 0 && !inactive;
}

void
ChoiceRule::inactivate (Atom **a)
{
  inactive++;
  if (inactive == 1)
    {
      for (Atom **h = head; h != hend; h++)
	{
	  Atom *b = *h;
	  b->headof--;
	  if (b->Bneg == false)
	    {
	      if (b->headof && (b->source == 0 || b->source == this))
		{
		  b->source = 0;
		  queue_push (b);
		}
	      if (b->headof == 0)
		efalse_push (b);
	      else if (b->headof == 1 && b->Bpos && *a != b)
		b->backchainTrue ();
	    }
	}
    }
}

void
ChoiceRule::inactivateInTrue (Atom **a)
{
  if ((*a)->closure)
    pos2++;
  inactivate (a);
}

void
ChoiceRule::inactivateInFalse (Atom **a)
{
  inactivate (a);
}

void
ChoiceRule::fireInit ()
{
  if ((pos == 0 && neg == 0) || pos2 == 0)
    for (Atom **h = head; h != hend; h++)
      {
	queue_push (*h);
	if ((*h)->source == 0)
	  (*h)->source = this;
      }
}

void
ChoiceRule::mightFireInitTrue (Atom **)
{
  pos2--;
  if (pos2 == 0 && !inactive)
    for (Atom **h = head; h != hend; h++)
      {
	queue_push (*h);
	if ((*h)->source == 0)
	  (*h)->source = this;
      }
}

void
ChoiceRule::unInit ()
{
  pos2 = pos;
}

void
ChoiceRule::mightFire ()
{
}

void
ChoiceRule::mightFireTrue (Atom **)
{
  pos--;
}

void
ChoiceRule::mightFireFalse (Atom **)
{
  neg--;
}

void
ChoiceRule::backChainTrue ()
{
  Atom **b;
  if (neg)
    for (b = nbody; b != nend; b++)
      if ((*b)->Bneg == false)
	efalse_push (*b);
  if (pos)
    for (b = pbody; b != pend; b++)
      if ((*b)->Bpos == false)
	etrue_push (*b);
}

void
ChoiceRule::backChainFalse ()
{
}

void
ChoiceRule::backtrackFromTrueInPos (Atom **)
{
  pos++;
}

void
ChoiceRule::backtrackFromFalseInPos (Atom **)
{
  pos2--;
  inactive--;
  if (inactive == 0)
    for (Atom **h = head; h != hend; h++)
      (*h)->headof++;
}

void
ChoiceRule::backtrackFromTrueInNeg (Atom **)
{
  inactive--;
  if (inactive == 0)
    for (Atom **h = head; h != hend; h++)
      (*h)->headof++;
}

void
ChoiceRule::backtrackFromFalseInNeg (Atom **)
{
  neg++;
}

void
ChoiceRule::propagateFalse (Atom **)
{
  pos2++;
  if (pos2 == 1 && !inactive)
    for (Atom **h = head; h != hend; h++)
      if ((*h)->closure != false &&
	  ((*h)->source == 0 || (*h)->source == this))
	{
	  (*h)->source = 0;
	  queue_push (*h);
	}
}

void
ChoiceRule::propagateTrue (Atom **)
{
  pos2--;
  if (pos2 == 0 && !inactive)
    for (Atom **h = head; h != hend; h++)
      if ((*h)->closure == false)
	{
	  if ((*h)->source == 0)
	    (*h)->source = this;
	  queue_push (*h);
	}
}

void
ChoiceRule::backtrackUpper (Atom **)
{
  pos2--;
}

void
ChoiceRule::search (Atom *)
{
  for (Atom **h = head; h != hend; h++)
    if (!(*h)->visited && (*h)->Bneg == false)
      visit (*h);
  for (Atom **a = nbody; a != nend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      visit (*a);
  for (Atom **a = pbody; a != pend; a++)
    if (!(*a)->visited && (*a)->Bneg == false)
      visit (*a);
  visited = true;
}

void
ChoiceRule::reduce (bool strongly)
{
  Atom **a;
  Atom **b;
  Atom *t;
  b = head;
  for (a = head; a != hend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((strongly == false || (*b)->Bpos == false)
	  && (*b)->Bneg == false)
	b++;
    }
  hend = b;
  b = nbody;
  for (a = nbody; a != nend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((*b)->Bpos == false && (*b)->Bneg == false)
	b++;
    }
  nend = b;
  b = pbody;
  for (a = pbody; a != pend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((strongly == false || (*b)->Bpos == false)
	  && (*b)->Bneg == false)
	b++;
    }
  pend = b;
}

void
ChoiceRule::unreduce ()
{
  hend = nbody;
  nend = pbody;
  pend = end;
}

void
ChoiceRule::setup ()
{
  Atom **a;
  for (a = head; a != hend; a++)
    {
      (*a)->head->r = this;
      (*a)->head->a = a;
      (*a)->head++;
    }
  for (a = nbody; a != nend; a++)
    {
      (*a)->negScore--;
      (*a)->neg[(*a)->negScore].r = this;
      (*a)->neg[(*a)->negScore].a = a;
    }
  for (a = pbody; a != pend; a++)
    {
      (*a)->posScore--;
      (*a)->pos[(*a)->posScore].r = this;
      (*a)->pos[(*a)->posScore].a = a;
    }
}

void
ChoiceRule::print ()
{
  Atom **a;
  bool comma = false;
  cout << "{ ";
  for (a = head; a != hend; a++)
    {
      if (comma)
	cout << ", ";
      cout << (*a)->atom_name ();
      comma = true;
    }
  cout << " }";
  comma = false;
  if (pbody != pend || nbody != nend)
    cout << " :- ";
  for (a = pbody; a != pend; a++)
    {
      if (comma)
	cout << ", ";
      cout << (*a)->atom_name ();
      comma = true;
    }
  for (a = nbody; a != nend; a++)
    {
      if (comma)
	cout << ", ";
      cout << "not " << (*a)->atom_name ();
      comma = true;
    }
  cout << '.' << endl;
}


WeightRule::WeightRule (Program *p0)
  : HeadRule (p0)
{
  head = 0;
  bend = 0;
  body = 0;
  end = 0;
  reverse = 0;
  weight = 0;
  positive = 0;
  minweight = 0;
  maxweight = 0;
  upper_max = 0;
  upper_min = 0;
  atmost = 0;
  atleast = 0;
  lower = 0;
  upper = 0;
  lower_shadow = 0;
  upper_shadow = 0;
  visited = false;
  type = WEIGHTRULE;
}

WeightRule::~WeightRule ()
{
  delete[] body;
  delete[] weight;
  delete[] positive;
  delete[] reverse;
}

bool
WeightRule::isInactive ()
{
  return maxweight < atleast || minweight > atmost;
}

bool
WeightRule::isUpperActive ()
{
  return upper_max <= atmost && upper_min >= atleast;
}

bool
WeightRule::fired ()
{
  return maxweight <= atmost && minweight >= atleast;
}

void
WeightRule::change (Atom *a)
{
  if (isInactive ())
    {
      Atom *b = head;
      b->headof--;
      if (b->Bneg == false)
	{
	  if (b->headof && (b->source == 0 || b->source == this))
	    {
	      b->source = 0;
	      queue_push (b);
	    }
	  if (b->headof == 0)
	    efalse_push (b);
	  else if (b->headof == 1 && b->Bpos && a != b)
	    b->backchainTrue ();
	}
    }
  else if (fired ())
    {
      if (head->Bpos == false)
	etrue_push (head);
    }
  else if (head->Bneg)
    backChainFalse ();
  else if (head->Bpos && head->headof == 1)
    backChainTrue ();
}

void
WeightRule::upper_lower (Atom **a)
{
  if (upper == a)
    do 
      upper++;
    while (upper != bend && ((*upper)->Bpos || (*upper)->Bneg));
  if (lower == a)
    do
      lower++;
    while (lower != bend && ((*lower)->Bpos || (*lower)->Bneg));
}

void
WeightRule::inactivate (Atom **a)
{
  bool inactive = isInactive ();
  bool frd = fired ();
  Weight w = weight[a-body];
  if (w >= 0)
    maxweight -= w;
  else
    minweight -= w;
  upper_lower (a);
  if (!frd && !inactive)
    change (*a);
}

void
WeightRule::inactivateInTrue (Atom **a)
{
  if ((*a)->closure)
    {
      Weight w = weight[a-body];
      if (w > 0)
	upper_min -= w;
      else
	upper_max -= w;
    }
  inactivate (a);
}

void
WeightRule::inactivateInFalse (Atom **a)
{
  inactivate (a);
}

void
WeightRule::fireInit ()
{
  if (fired ())
    {
      etrue_push (head);
      queue_push (head);
      head->source = this;
    }
  else if (isUpperActive () && !isInactive ())
    {
      queue_push (head);
      if (head->source == 0)
	head->source = this;
    }
}

void
WeightRule::mightFireInitTrue (Atom **a)
{
  bool active = isUpperActive ();
  Weight w = weight[a-body];
  if (w > 0)
    upper_min += w;
  else
    upper_max += w;
  if (!active && isUpperActive () && !isInactive ())
    {
      queue_push (head);
      if (head->source == 0)
	head->source = this;
    }
}

void
WeightRule::mightFireInitFalse (Atom **a)
{
  bool active = isUpperActive ();
  Weight w = weight[a-body];
  if (w > 0)
    upper_max -= w;
  else
    upper_min -= w;
  if (!active && isUpperActive () && !isInactive ())
    {
      queue_push (head);
      if (head->source == 0)
	head->source = this;
    }
}

void
WeightRule::unInit ()
{
  upper_min = 0;
  upper_max = 0;
  for (Atom **a = body; a != bend; a++)
    if (!positive[a-body])
      {
	Weight w = weight[a-body];
	upper_min += w;
	upper_max += w;
      }
}

void
WeightRule::mightFire (Atom **a)
{
  bool inactive = isInactive ();
  bool frd = fired ();
  Weight w = weight[a-body];
  if (w < 0)
    maxweight += w;
  else
    minweight += w;
  upper_lower (a);
  if (!frd && !inactive)
    change (*a);
}

void
WeightRule::mightFireTrue (Atom **a)
{
  mightFire (a);
}

void
WeightRule::mightFireFalse (Atom **a)
{
  if ((*a)->closure)
    {
      bool active = isUpperActive ();
      Weight w = weight[a-body];
      if (w > 0)
	upper_max += w;
      else
	upper_min += w;
      if (active && !isInactive () && !isUpperActive ())
	{
	  Atom *b = head;
	  if (b->headof && (b->source == 0 || b->source == this))
	    {
	      b->source = 0;
	      queue_push (b);
	    }
	}
    }
  mightFire (a);
}

void
WeightRule::backChainTrue ()
{
  if (fired ())
     return;
  for (; upper_shadow != bend; upper_shadow++)
    {
      if ((*upper_shadow)->Bpos || (*upper_shadow)->Bneg)
	continue;
      if (weight[upper_shadow-body] < 0)
	{
	  if (maxweight + weight[upper_shadow-body] < atleast)
	    {
	      if (positive[upper_shadow-body])
		efalse_push (*upper_shadow);
	      else
		etrue_push (*upper_shadow);
	    }
	  else
	    break;
	}
      else
	{
	  if (maxweight - weight[upper_shadow-body] < atleast)
	    {
	      if (positive[upper_shadow-body])
		etrue_push (*upper_shadow);
	      else
		efalse_push (*upper_shadow);
	    }
	  else
	    break;
	}
    }
  for (; lower_shadow != bend; lower_shadow++)
    {
      if ((*lower_shadow)->Bpos || (*lower_shadow)->Bneg)
	continue;
      if (weight[lower_shadow-body] > 0)
	{
	  if (minweight + weight[lower_shadow-body] > atmost)
	    {
	      if (positive[lower_shadow-body])
		efalse_push (*lower_shadow);
	      else
		etrue_push (*lower_shadow);
	    }
	  else
	    break;
	}
      else
	{
	  if (minweight - weight[lower_shadow-body] > atmost)
	    {
	      if (positive[lower_shadow-body])
		etrue_push (*lower_shadow);
	      else
		efalse_push (*lower_shadow);
	    }
	  else
	    break;
	}
    }
}

void
WeightRule::backChainFalse ()
{
  if (isInactive () || fired ())
    return;
  for (; upper_shadow != bend; upper_shadow++)
    {
      if ((*upper_shadow)->Bpos || (*upper_shadow)->Bneg)
	continue;
      if (weight[upper_shadow-body] < 0)
	{
	  if (maxweight + weight[upper_shadow-body] <= atmost &&
	      minweight >= atleast)
	    {
	      if (positive[upper_shadow-body])
		efalse_push (*upper_shadow);
	      else
		etrue_push (*upper_shadow);
	    }
	  else
	    break;
	}
      else
	{
	  if (maxweight - weight[upper_shadow-body] <= atmost &&
	      minweight >= atleast)
	    {
	      if (positive[upper_shadow-body])
		etrue_push (*upper_shadow);
	      else
		efalse_push (*upper_shadow);
	    }
	  else
	    break;
	}
    }
  for (; lower_shadow != bend; lower_shadow++)
    {
      if ((*lower_shadow)->Bpos || (*lower_shadow)->Bneg)
	continue;
      if (weight[lower_shadow-body] > 0)
	{
	  if (minweight + weight[lower_shadow-body] >= atleast &&
	      maxweight <= atmost)
	    {
	      if (positive[lower_shadow-body])
		efalse_push (*lower_shadow);
	      else
		etrue_push (*lower_shadow);
	    }
	  else
	    break;
	}
      else
	{
	  if (minweight - weight[lower_shadow-body] >= atleast &&
	      maxweight <= atmost)
	    {
	      if (positive[lower_shadow-body])
		etrue_push (*lower_shadow);
	      else
		efalse_push (*lower_shadow);
	    }
	  else
	    break;
	}
    }
}

void
WeightRule::backtrack (Atom **a)
{
  if (upper > a)
    upper = a;
  upper_shadow = upper;
  if (lower > a)
    lower = a;
  lower_shadow = lower;
}

void
WeightRule::backtrackFromTrueInPos (Atom **a)
{
  bool inactive = isInactive ();
  Weight w = weight[a-body];
  if (w < 0)
    maxweight -= w;
  else
    minweight -= w;
  if (inactive && !isInactive ())
    head->headof++;
  backtrack (a);
}

void
WeightRule::backtrackFromFalseInPos (Atom **a)
{
  bool inactive = isInactive ();
  Weight w = weight[a-body];
  if (w > 0)
    {
      maxweight += w;
      upper_min += w;
    }
  else
    {
      minweight += w;
      upper_max += w;
    }
  if (inactive && !isInactive ())
    head->headof++;
  backtrack (a);
}

void
WeightRule::backtrackFromTrueInNeg (Atom **a)
{
  bool inactive = isInactive ();
  Weight w = weight[a-body];
  if (w > 0)
    maxweight += w;
  else
    minweight += w;
  if (inactive && !isInactive ())
    head->headof++;
  backtrack (a);
}

void
WeightRule::backtrackFromFalseInNeg (Atom **a)
{
  bool inactive = isInactive ();
  Weight w = weight[a-body];

  if (w > 0)
    upper_max -= w;
  else
    upper_min -= w;
  if (w < 0)
    maxweight -= w;
  else
    minweight -= w;
  if (inactive && !isInactive ())
    head->headof++;
  backtrack (a);
}

void
WeightRule::propagateFalse (Atom **a)
{
  bool active = isUpperActive ();
  bool inactive = isInactive ();
  Weight w = weight[a-body];
  if (w > 0)
    upper_min -= w;
  else
    upper_max -= w;
  if (!inactive && active && !isUpperActive () &&
      head->closure != false &&
      (head->source == 0 || head->source == this))
    {
      head->source = 0;
      queue_push (head);
    }
}

void
WeightRule::propagateFalseInNeg (Atom **a)
{
  bool active = isUpperActive ();
  bool inactive = isInactive ();
  Weight w = weight[a-body];
  if (w > 0)
    upper_max += w;
  else
    upper_min += w;
  if (!inactive && active && !isUpperActive () &&
      head->closure != false &&
      (head->source == 0 || head->source == this))
    {
      head->source = 0;
      queue_push (head);
    }
}

void
WeightRule::propagateTrue (Atom **a)
{
  bool active = isUpperActive ();
  bool inactive = isInactive ();
  Weight w = weight[a-body];
  if (w > 0)
    upper_min += w;
  else
    upper_max += w;
  if (!inactive && !active && isUpperActive () &&
      head->closure == false)
    {
      if (head->source == 0)
	head->source = this;
      queue_push (head);
    }
}

void
WeightRule::propagateTrueInNeg (Atom **a)
{
  bool active = isUpperActive ();
  bool inactive = isInactive ();
  Weight w = weight[a-body];
  if (w > 0)
    upper_max -= w;
  else
    upper_min -= w;
  if (!inactive && !active && isUpperActive () &&
      head->closure == false)
    {
      if (head->source == 0)
	head->source = this;
      queue_push (head);
    }
}

void
WeightRule::backtrackUpper (Atom **a)
{
  Weight w = weight[a-body];
  if (w > 0)
    upper_min += w;
  else
    upper_max += w;
}

void
WeightRule::backtrackUpperInNeg (Atom **a)
{
  Weight w = weight[a-body];
  if (w > 0)
    upper_max -= w;
  else
    upper_min -= w;
}

void
WeightRule::search (Atom *a)
{
  if (a->Bneg || (a->Bpos && a != head))
    return;
  if (!head->visited && head->Bneg == false)
    {
      queue_push (head);
      head->visited = true;
    }
  for (Atom **b = body; b != bend; b++)
    if (!(*b)->visited && (*b)->Bneg == false)
      {
	queue_push (*b);
	(*b)->visited = true;
      }
  visited = true;
}

void
WeightRule::reduce (bool strongly)
{
  Atom **b = body;
  for (Atom **a = body; a != bend; a++)
    {
      swap (b-body, a-body);
      if ((strongly == false || (*b)->Bpos == false) &&
	  (*b)->Bneg == false)
	b++;
    }
  bend = b;
  backtrack (body);
}

void
WeightRule::unreduce ()
{
  bend = end;
  backtrack (body);
  sort ();
}

void
WeightRule::initWeight (Weight w)
{
  if (w > 0)
    maxweight += w;
  else
    minweight += w;
}

void
WeightRule::swap (long a, long b)
{
  Weight tw = weight[a];
  weight[a] = weight[b];
  weight[b] = tw;
  Atom *ta = body[a];
  body[a] = body[b];
  body[b] = ta;
  bool tb = positive[a];
  positive[a] = positive[b];
  positive[b] = tb;
  if (reverse[a])
    reverse[a]->a = body+b;
  if (reverse[b])
  reverse[b]->a = body+a;
}

bool
WeightRule::smaller (long a, long b)
{
#ifdef USEDOUBLE
  if (fabs (weight[a]) < fabs (weight[b]))
    return true;
  else
    return false;
#else
  Weight wa, wb;
  if (weight[a] < 0)
    wa = -weight[a];
  else
    wa = weight[a];
  if (weight[b] < 0)
    wb = -weight[b];
  else
    wb = weight[b];
  if (wa < wb)
    return true;
  else
    return false;
#endif
}

void
WeightRule::heap (long k, long e)
{
  long a = 2*k+1;
  while (a < e)
    {
      if (smaller (a+1, a))
	a++;
      if (smaller (a, k))
	swap (a, k);
      else
	break;
      k = a;
      a = 2*k+1;
    }
  if (a == e && smaller (a, k))
    swap (a, k);
}

void
WeightRule::sort ()
{
  long i;
  long e = bend-body-1;
  for (i = (e-1)/2; i >= 0; i--)
    heap (i, e);
  i = e;
  while (i > 0)
    {
      swap (i, 0);
      i--;
      heap (0, i);
    }
}

void
WeightRule::swapping (Follows *f, Follows *g)
{
  reverse[f->a - body] = g;
  reverse[g->a - body] = f;
}

void
WeightRule::setup ()
{
  head->head->r = this;
  head->head->a = &head;
  head->head++;
  Atom **a;
  Weight *w = weight;
  sort ();
  for (a = body; a != bend; a++)
    if (positive[a-body])
      {
	(*a)->posScore--;
	(*a)->pos[(*a)->posScore].r = this;
	(*a)->pos[(*a)->posScore].a = a;
	reverse[a-body] = (*a)->pos + (*a)->posScore;
	initWeight (*w++);
      }
    else
      {
	(*a)->negScore--;
	(*a)->neg[(*a)->negScore].r = this;
	(*a)->neg[(*a)->negScore].a = a;
	reverse[a-body] = (*a)->neg + (*a)->negScore;
	upper_min += *w;
	upper_max += *w;
	initWeight (*w++);
      }
}

void
WeightRule::print ()
{
  cout << head->atom_name () << " :- { ";
  int comma = 0;
  for (Atom **a = body; a != bend; a++)
    if (positive[a-body])
      {
	if (comma)
	  cout << ", ";
	cout << (*a)->atom_name () << " = " << weight[a-body];
	comma = 1;
      }
    else
      {
	if (comma)
	  cout << ", ";
	cout << "not " << (*a)->atom_name () << " = "
	     << weight[a-body];
	comma = 1;
      }
  cout << "} >=" << atleast << " <=" << atmost << '.' << endl;
}


OptimizeRule::OptimizeRule (Program *p0)
  : Rule (p0)
{
  nbody = 0;
  pbody = 0;
  nend = 0;
  pend = 0;
  end = 0;
  weight = 0;
  minweight = 0;
  maxweight = 0;
  maxoptimum = 0;
  minoptimum = 0;
  visited = false;
  type = OPTIMIZERULE;
  maximize = false;
  next = 0;
}

OptimizeRule::~OptimizeRule ()
{
  delete[] nbody;
  delete[] weight;
}

void
OptimizeRule::setOptimum ()
{
  minoptimum = minweight;
  maxoptimum = maxweight;
}

bool
OptimizeRule::isInactive ()
{
  return false;  // Reduce(Strongly|Weakly) needs this
}

bool
OptimizeRule::isUpperActive ()
{
  return false;
}

void
OptimizeRule::inactivate (Atom **a)
{
  Weight w = weight[a-nbody];
  if (w >= 0)
    maxweight -= w;
  else
    minweight -= w;
}

void
OptimizeRule::inactivateInTrue (Atom **a)
{
  inactivate (a);
}

void
OptimizeRule::inactivateInFalse (Atom **a)
{
  inactivate (a);
}

void
OptimizeRule::fireInit ()
{
}

void
OptimizeRule::mightFireInitTrue (Atom **)
{
}

void
OptimizeRule::unInit ()
{
}

void
OptimizeRule::mightFire (Atom **a)
{
  Weight w = weight[a-nbody];
  if (w < 0)
    maxweight += w;
  else
    minweight += w;
}

void
OptimizeRule::mightFireTrue (Atom **a)
{
  mightFire (a);
}

void
OptimizeRule::mightFireFalse (Atom **a)
{
  mightFire (a);
}

void
OptimizeRule::backChainTrue ()
{
}

void
OptimizeRule::backChainFalse ()
{
}

void
OptimizeRule::backtrackFromTrueInPos (Atom **a)
{
  Weight w = weight[a-nbody];
  if (w < 0)
    maxweight -= w;
  else
    minweight -= w;
}

void
OptimizeRule::backtrackFromFalseInPos (Atom **a)
{
  Weight w = weight[a-nbody];
  if (w >= 0)
    maxweight += w;
  else
    minweight += w;
}

void
OptimizeRule::backtrackFromTrueInNeg (Atom **a)
{
  backtrackFromFalseInPos (a);
}

void
OptimizeRule::backtrackFromFalseInNeg (Atom **a)
{
  backtrackFromTrueInPos (a);
}

void
OptimizeRule::propagateFalse (Atom **)
{
}

void
OptimizeRule::propagateTrue (Atom **)
{
}

void
OptimizeRule::backtrackUpper (Atom **)
{
}

void
OptimizeRule::search (Atom *a)
{
  if (a->Bneg || a->Bpos)
    return;
  Atom **b;
  for (b = nbody; b != nend; b++)
    if (!(*b)->visited && (*b)->Bneg == false && (*b)->Bpos == false)
      {
	queue_push (*b);
	(*b)->visited = true;
      }
  for (b = pbody; b != pend; b++)
    if (!(*b)->visited && (*b)->Bneg == false && (*b)->Bpos == false)
      {
	queue_push (*b);
	(*b)->visited = true;
      }
  visited = true;
}

void
OptimizeRule::reduce (bool)
{
  Atom **a;
  Atom **b;
  Atom *t;
  b = nbody;
  for (a = nbody; a != nend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((*b)->Bpos == false && (*b)->Bneg == false)
	b++;
    }
  nend = b;
  b = pbody;
  for (a = pbody; a != pend; a++)
    {
      t = *b;
      *b = *a;
      *a = t;
      if ((*b)->Bpos == false && (*b)->Bneg == false)
	b++;
    }
  pend = b;
}

void
OptimizeRule::unreduce ()
{
  nend = pbody;
  pend = end;
}

void
OptimizeRule::initWeight (Weight w)
{
  if (w > 0)
    maxweight += w;
  else
    minweight += w;
}

void
OptimizeRule::setup ()
{
  Atom **a;
  Weight *w = weight;
  for (a = nbody; a != nend; a++)
    {
      (*a)->negScore--;
      (*a)->neg[(*a)->negScore].r = this;
      (*a)->neg[(*a)->negScore].a = a;
      initWeight (*w++);
    }
  for (a = pbody; a != pend; a++)
    {
      (*a)->posScore--;
      (*a)->pos[(*a)->posScore].r = this;
      (*a)->pos[(*a)->posScore].a = a;
      initWeight (*w++);
    }
}

void
OptimizeRule::print ()
{
  if (maximize)
    cout << "maximize { ";
  else
    cout << "minimize { ";
  Atom **a;
  int comma = 0;
  for (a = pbody; a != pend; a++)
    {
      if (comma)
	cout << ", ";
      cout << (*a)->atom_name () << " = " << weight[a-nbody];
      comma = 1;
    }
  for (a = nbody; a != nend; a++)
    {
      if (comma)
	cout << ", ";
      cout << "not " << (*a)->atom_name () << " = " << weight[a-nbody];
      comma = 1;
    }
  cout << " }" << endl;
}
