/*
 * Main routines:
 *
 * Copyright Keijo Heljanko <Keijo.Heljanko@hut.fi>
 *
 * Smodels:
 *
 * Copyright Patrik Simons <Patrik.Simons@hut.fi>
 *
 * PEP prefix reading code skeleton:
 *
 * Copyright Burkhard Graves, PEP project, University of Hildesheim
 * See: http://www.informatik.uni-hildesheim.de/~pep/HomePage.html 
 *
 */


#include <stdio.h>
#include "braproc.h"
#include "unf2smo.h"

// Include the smodels stuff.

#include <iostream.h>
#include "stable.h"
#include "api.h"
#include "timer.h"

static void usage(char *myname)
{
    fprintf(stderr,"============================================================================\n");
    fprintf(stderr,"Determination of deadlocks (Smodels reachability translation)\n");
    fprintf(stderr,"Usage: %s [OPTION]... file1 [file2]\n",myname);
    fprintf(stderr,"\n");
    fprintf(stderr,"Options:\n");
    fprintf(stderr,"        -O0 : Do not optimize the translation\n");
    fprintf(stderr,"        -O1 : Do only guaranteed linear time optimizations\n");
    fprintf(stderr,"        -O2 : Do cheap polynomial time optimizations (default)\n");
    fprintf(stderr,"        -O3 : Do expensive optimizations (much slower than O2)\n");
    fprintf(stderr,"        -d  : Produce smodels translation only (for debugging)\n");
    fprintf(stderr,"        -v  : Give verbose translation statistics\n");
    fprintf(stderr,"\n");
    fprintf(stderr,"        file1 : input file containing unfolding\n");
    fprintf(stderr,"                extension '.mci': binary coded format\n");
    fprintf(stderr,"        file2 : output result file\n");
    fprintf(stderr,"\n");
    fprintf(stderr,"Copyright (c) 1998 Keijo Heljanko <Keijo.Heljanko@hut.fi> (translation)\n");
    fprintf(stderr,"Copyright (c) 1998 Patrik Simons <Patrik.Simons@hut.fi> (smodels)\n");
    fprintf(stderr,"\n");
    fprintf(stderr,"Version 1.0, smodels version pre-2.0.30, (%s)\n", __DATE__);
    fprintf(stderr,"============================================================================\n");

}

int main (int argc, char *argv[])
{

  FILE * in;
  char * input_file;
  char * output_file;
  BraProcess * braproc;
  Unf2Smo * unf2smo;
  unsigned long optimize_level;
  BOOL debugging;
  BOOL verbose;
  BOOL error;
  Timer timer1;
  Timer timer2;

  timer1.start();
  
  optimize_level = 2;
  debugging = FALSE;
  verbose = FALSE;
  error = FALSE;
  
  in = (FILE *) NULL;
  input_file = (char *) NULL;
  output_file = (char *) NULL;
  unf2smo = (Unf2Smo *) NULL;
  
  for (int c = 1; c < argc && error == FALSE; c++) {
    if (argv[c][0] == '-') {
      if (strcmp (&argv[c][1], "O0") == 0) {
	optimize_level = 0;
      }
      else if (strcmp (&argv[c][1], "O1") == 0) {
	optimize_level = 1;
      }
      else if (strcmp (&argv[c][1], "O2") == 0) {
	optimize_level = 2;
      }
      else if (strcmp (&argv[c][1], "O3") == 0) {
	optimize_level = 3;
      }
      else if (strcmp (&argv[c][1], "d") == 0) {
	debugging = TRUE;
      }
      else if (strcmp (&argv[c][1], "v") == 0) {
	verbose = TRUE;
      } else {
	error = TRUE;
      }
    } else {
      if(input_file == NULL) {
	input_file = argv[c];
      } else if(output_file == NULL) {
	output_file = argv[c];
      } else {
	error = TRUE;
	break;
      }
    }
  }

  if(input_file == NULL) {
    error = TRUE;
  }
  
  if(error != FALSE) {
    usage(argv[0]);
    return 1;
  }
    
  if((in = fopen(input_file, "rb")) == NULL) {

    fprintf(stderr, "%s: No such prefix file: %s\n", argv[0], input_file);
    return 2;
   
  }

  if(output_file != NULL) {

    if(freopen(output_file, "wb", stdout) == NULL) {

      fprintf(stderr, "%s: Could not open output file: %s \n", argv[0], output_file);
      return 2;
   
    }
    
  }
  
  braproc = new BraProcess();

  braproc->ReadMciFile(in);

  if(braproc->num_cutoffs != 0) {

    unf2smo = new Unf2Smo(braproc);
    unf2smo->SetOptimizeLevel(optimize_level);
    unf2smo->CreateDeadlockProgram(debugging);

  }
    
  if(debugging != FALSE) {

    if(braproc->num_cutoffs != 0) {

      unf2smo->PrintProgram();

    } else {

      // Make a fake program with one models if no cutoffs exist.
      // (saves time)
      
      fprintf(stdout, "no_cutoffs.\n");
      
    }
      
    fflush(stdout);

  } else {

    timer2.start();

    if(braproc->num_cutoffs == 0) {

      fprintf(stdout, "FALSE - EVERY\n");

      timer2.stop();

    } else if(unf2smo->ComputeModel() != FALSE) {
      
      fprintf(stdout, "FALSE\n");

      timer2.stop();

      if(unf2smo->CreateDeadlockConfiguration() == FALSE) {

	// Internal error! (A bug somewhere caught by a sanity check.)

	return 2;

      } else {

	// Everything OK.
	;

      }

    } else {

      fprintf(stdout, "TRUE\n");
      timer2.stop();
    
    }

    fflush(stdout);
    
    timer1.stop();

    fprintf(stdout, "\nTime needed:      %s s\n", timer1.print());
    
    fflush(stdout);
    
    if(debugging == FALSE && verbose != FALSE) {

      fprintf(stdout, "\nSmodels statistics:\n");

      if(braproc->num_cutoffs == 0) {

	fprintf(stdout, "\nProgram not generated (no cutoff events)\n");
	
      } else {
	
	fprintf(stdout, "\nDuration: %s\n", timer2.print());

	unf2smo->PrintStatistics();

      }
	
      fflush(stdout);
      
    }


  }
    

  if(output_file != NULL) {
    fclose(stdout);
  }
  
  if(braproc->num_cutoffs != 0) {

    delete(unf2smo);

  }
    
  delete(braproc);
  
  fclose(in);
  
  return 0;

}


