// Copyright 1997, 1998 by Keijo Heljanko
// This software is provided as is, no warranty of any kind is given.

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "dassert.h"
#include "bitarr.h"
#include "alloc.h"
#include "combi.h"

/***************************************************************************
*
* Function:  CreateBitArray
* Purpose:   Creates an bit-array and allocates memory for it.
* Arguments: numbits - the number of bits in the array
* Returns:   A pointer to a bit-array and NULL on failure.
*            The bit-array is initialized to all zero bits.
* See also:
*
****************************************************************************
*/
BITARRAY * CreateBitArray(unsigned long numbits)
{
    unsigned long numitems;
    BITARRAY *array;
    ARRAY_ITEM *p;
    
    DEBUG_ASSERT(numbits > 0);

    numitems = ((numbits / (sizeof(ARRAY_ITEM) * BITSPERBYTE)) +
      (((numbits % (sizeof(ARRAY_ITEM) * BITSPERBYTE)) == 0) ? 0 : 1));

    DEBUG_ASSERT(numitems > 0);
    
    /* The array is a memory area with BITARRAY structure in the beginning
       followed by the real bit array storage. */

    array = (BITARRAY *) Allocate(sizeof(BITARRAY) +
				 ((numitems - 1) * sizeof(ARRAY_ITEM)));
    DEBUG_ASSERT(array != NULL);
    array->numbits = numbits;
    array->numitems = numitems;
    for (p = array->items; numitems--; *(p++) = (ARRAY_ITEM) 0L)
	;
    return(array);
}

/***************************************************************************
*
* Function:  DeleteBitArray
* Purpose:   Removes the contents of a bit-array from memory.
* Arguments: array - a pointer to a bit-array
* Returns:   
* See also:
*
****************************************************************************
*/
void DeleteBitArray(BITARRAY * array)
{
    DEBUG_ASSERT(array != NULL);

    if(array != NULL) {
	Deallocate((char *)array);
    } else {
	DEBUG_ASSERT(FALSE);
    }
}

#ifdef DEBUG


/***************************************************************************
*
* Function:  DumpBitArray
* Purpose:   Display the contents of a bit-array.
* Arguments: array - a pointer to a bit-array
*            depth - the output starts from the (depth + 1)th column
* Returns:   
* See also:
*
****************************************************************************
*/
void DumpBitArray(BITARRAY * array, unsigned depth)
{
    DEBUG_ASSERT(array != NULL);

    /* This is only a partial dump of the data in bit-array. */
    
    DumpIndent(depth);
    (void) fprintf(_err, "BITARRAY:\n");
    (void) fflush(_err);

    depth++;
    
    DumpIndent(depth);
    (void) fprintf(_err, "numbits %lu\n", array->numbits);
    DumpIndent(depth);
    (void) fprintf(_err, "numitems %lu\n", array->numitems);

    (void) fflush(_err);

}

#endif



/***************************************************************************
*
* Function:  BitArrayCopy
* Purpose:   Creates a copy of bit-array and stores the result
*            into the destination.
* Arguments: source1 - a pointer to a bit-array
*            dest - a pointer to a result bit-array
* Returns:
* See also:
*
****************************************************************************
*/
void BitArrayCopy(BITARRAY * source1,
		  BITARRAY * dest)
{
    unsigned long i,j;
    ARRAY_ITEM *s1, *d;
    
    DEBUG_ASSERT(source1 != NULL);
    DEBUG_ASSERT(dest != NULL);

    DEBUG_ASSERT(source1->numbits == dest->numbits);
    DEBUG_ASSERT(source1->numbits != 0);

    
    j = source1->numitems;
    s1 = &(source1->items[0]);
    d = &(dest->items[0]);

    for(i = j; i > 0 ; i--) {
	*d++ = *s1++;
    }

}


/***************************************************************************
*
* Function:  BitArrayAnd
* Purpose:   Creates an and of two bit-arrays and stores the result
*            into a third one. The destination can be the same as
*            one of the sources.
* Arguments: source1 - a pointer to a bit-array
*            source2 - a pointer to another bit-array
*            dest - a pointer to a result bit-array
* Returns:
* See also:
*
****************************************************************************
*/
void BitArrayAnd(BITARRAY * source1,
		 BITARRAY * source2,
		 BITARRAY * dest)
{
    unsigned long i,j;
    ARRAY_ITEM *s1, *s2, *d;
    
    DEBUG_ASSERT(source1 != NULL);
    DEBUG_ASSERT(source2 != NULL);
    DEBUG_ASSERT(dest != NULL);

    DEBUG_ASSERT(source1->numbits == source2->numbits);
    DEBUG_ASSERT(source1->numbits == dest->numbits);
    DEBUG_ASSERT(source1->numbits != 0);

    
    j = source1->numitems;
    s1 = &(source1->items[0]);
    s2 = &(source2->items[0]);
    d = &(dest->items[0]);

    for(i = j; i > 0 ; i--) {
	*d++ = *s1++ & *s2++;
    }
}


/***************************************************************************
*
* Function:  BitArrayOr
* Purpose:   Creates an or of two bit-arrays and stores the result
*            into a third one. The destination can be the same as
*            one of the sources.
* Arguments: source1 - a pointer to a bit-array
*            source2 - a pointer to another bit-array
*            dest - a pointer to a result bit-array
* Returns:
* See also:
*
****************************************************************************
*/
void BitArrayOr(BITARRAY * source1,
		BITARRAY * source2,
		BITARRAY * dest)
{
    unsigned long i,j;
    ARRAY_ITEM *s1, *s2, *d;
    
    DEBUG_ASSERT(source1 != NULL);
    DEBUG_ASSERT(source2 != NULL);
    DEBUG_ASSERT(dest != NULL);

    DEBUG_ASSERT(source1->numbits == source2->numbits);
    DEBUG_ASSERT(source1->numbits == dest->numbits);
    DEBUG_ASSERT(source1->numbits != 0);
   
    j = source1->numitems;
    s1 = &(source1->items[0]);
    s2 = &(source2->items[0]);
    d = &(dest->items[0]);

    for(i = j; i > 0 ; i--) {
	*d++ = *s1++ | *s2++;
    }
}


/***************************************************************************
*
* Function:  BitArrayNot
* Purpose:   Creates a not of a bit-array and stores the result
*            into another bit-array. The source and dest can be the
*            same bit-array.
*
* Arguments: source - a pointer to a bit-array
*            dest - a pointer to a result bit-array
* Returns:
* See also:
*
****************************************************************************
*/
void BitArrayNot(BITARRAY * source,
		 BITARRAY * dest)
{
    unsigned long i,j;
    ARRAY_ITEM *s, *d;
    
    DEBUG_ASSERT(source != NULL);
    DEBUG_ASSERT(dest != NULL);

    DEBUG_ASSERT(source->numbits == dest->numbits);
    DEBUG_ASSERT(source->numbits != 0);

    j = source->numitems;
    s = &(source->items[0]);
    d = &(dest->items[0]);

    for(i = j; i > 0 ; i--) {
	*d++ = ~(*s++);
    }
}



/***************************************************************************
*
* Function:  BitArrayNegateS1OrS2OverwriteS1
* Purpose:   Negates S1 and takes an or with S2 and stores the result
*            into S1. I need this in one inner loop, that's
*            why this one exists.
*
* Arguments: source1 - a pointer to a bit-array
*            source2 - a pointer to another bit-array
*            dest - a pointer to a result bit-array
* Returns:
* See also:
*
****************************************************************************
*/
void BitArrayNegateS1OrS2OverwriteS1(BITARRAY * source1,
				     BITARRAY * source2)
{
    unsigned long i,j, tmp;
    ARRAY_ITEM *s1, *s2;
    
    DEBUG_ASSERT(source1 != NULL);
    DEBUG_ASSERT(source2 != NULL);

    DEBUG_ASSERT(source1->numbits == source2->numbits);
    DEBUG_ASSERT(source1->numbits != 0);
   
    j = source1->numitems;
    s1 = &(source1->items[0]);
    s2 = &(source2->items[0]);

    for(i = j; i > 0 ; i--) {
	tmp = ~(*s1);
	tmp |= *s2++;
	*s1++ = tmp;
    }
}
