// Copyright 2001 by Keijo Heljanko
// This software is provided as is, no warranty of any kind is given.

#include "combi.h"
#include "dassert.h"
#include "tree2smo.h"
#include "y_tab.h"
#include "main.h"
#include "smodels.h"
#include "api.h"

TreeNode*
PushNegationsIn(TreeNode *node, BOOL negate)
{
  TreeNode *tmp;
  ASSERT(node != NULL);

  switch (node->info) {

  case RELEASE:
    if(negate) {
      node->info = UNTIL;
    }
    node->left = PushNegationsIn(node->left, negate);
    node->right = PushNegationsIn(node->right, negate);
    break;
  case UNTIL:
    if(negate) {
      node->info = RELEASE;
    }
    node->left = PushNegationsIn(node->left, negate);
    node->right = PushNegationsIn(node->right, negate);
    break;
  case BOX:
    if(negate) {
      node->info = DIAMOND;
    }
    ASSERT(node->left == NULL);
    node->right = PushNegationsIn(node->right, negate);
    break;
  case DIAMOND:
    if(negate) {
      node->info = BOX;
    }
    ASSERT(node->left == NULL);
    node->right = PushNegationsIn(node->right, negate);
    break;
  case OR:
    if(negate) {
      node->info = AND;
    }
    node->left = PushNegationsIn(node->left, negate);
    node->right = PushNegationsIn(node->right, negate);
    break;
  case AND:
    if(negate) {
      node->info = OR;
    }
    node->left = PushNegationsIn(node->left, negate);
    node->right = PushNegationsIn(node->right, negate);
    break;
  case NOT:
    if(negate) {
      negate = FALSE;
    } else {
      negate = TRUE;
    }
    ASSERT(node->left == NULL);
    tmp = node->right;
    node->right = NULL;
    TreeDelete(node);
    return (PushNegationsIn(tmp, negate));
    break;
  case PLACE:
    if(negate) {
      tmp = TreeCreateNode(PLACE, NULL, NULL, node->place_num);
      node->info = NOT;
      node->left = NULL;
      node->right = tmp;
      node->place_num = 0L;
    }
    break;
  default:
    ASSERT(FALSE);
    break;
  }

  return (node);

}

BOOL
FormulaHasBoxOrRelease(TreeNode *node)
{
  BOOL found;

  found = FALSE;

  switch (node->info) {

  case RELEASE:
  case BOX:
    found = TRUE;
    break;
  case UNTIL:
  case OR:
  case AND:
    if(FormulaHasBoxOrRelease(node->left) ||
       FormulaHasBoxOrRelease(node->right)) {
      found = TRUE;
    }
    break;
  case DIAMOND:
  case NOT:
    ASSERT(node->left == NULL);
    if(FormulaHasBoxOrRelease(node->right)) {
      found = TRUE;
    }
    break;
  case PLACE:
    break;
  default:
    ASSERT(FALSE);
    break;
  }

  return (found);

}


unsigned long
Tree2Smo(TreeNode *node, unsigned long *formula_num, Api *api, unsigned long bound)
{
  unsigned long i;
  unsigned long f_num;
  unsigned long f1_num;
  unsigned long f2_num;
  Atom *atom;
  char buf[256];

  ASSERT(node != NULL);


  // Allocate formula number for this formula

  f_num = *formula_num;
  *formula_num += 1;
  f1_num = 0;
  f2_num = 0;


  // Translate the subformulas, remember their formula numbers

  switch(node->info) {

  case RELEASE:
  case UNTIL:
  case OR:
  case AND:
    f1_num = Tree2Smo(node->left, formula_num, api, bound);
    f2_num = Tree2Smo(node->right, formula_num, api, bound);
    break;
  case BOX:
  case DIAMOND:
    f2_num = Tree2Smo(node->right, formula_num, api, bound);
    break;
  case NOT:
    ASSERT(node->right != NULL);
    ASSERT(node->right->info == PLACE);
    break;
  case PLACE:
    break;
  default:
    ASSERT(FALSE);
    break;

  }


  // Generate the needed atoms for all rounds

  for(i = 0; i <= (bound +1); i++) {

    switch(node->info) {
    case OR:
    case AND:
      if(i <= bound) {
	atom = api->new_atom();
	sprintf(buf, "f%ld(%ld)", f_num, i);
	api->set_name(atom, buf);
      }
      break;
    case RELEASE:
    case UNTIL:
    case BOX:
    case DIAMOND:
      atom = api->new_atom();
      sprintf(buf, "f%ld(%ld)", f_num, i);
      api->set_name(atom, buf);
      break;
    case NOT:
    case PLACE:
      if(i <= bound) {
	atom = api->new_atom();
	sprintf(buf, "f%ld(%ld)", f_num, i);
	api->set_name(atom, buf);

	if(node->info == NOT) {
	  sprintf(buf, "p%ld(%ld)", node->right->place_num, i);
	} else {
	  sprintf(buf, "p%ld(%ld)", node->place_num, i);
	}
	atom = api->get_atom(buf);
	if(atom == NULL) {
	  // The place has been optimized away, recreate it here!
	  atom = api->new_atom();
	  api->set_name(atom, buf);
	}
      }
      break;
  default:
    ASSERT(FALSE);
    break;

    }

  }


  // Do the translation

  for(i = 0; i <= bound; i++) {

    switch(node->info) {

    case PLACE:
      // f_num(i) :- p_place_num(i)
      api->begin_rule(BASICRULE);
      sprintf(buf, "f%ld(%ld)", f_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_head(atom);
      sprintf(buf, "p%ld(%ld)", node->place_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_body(atom, TRUE);
      api->end_rule();
      break;

    case NOT:
      // f_num(i) :- not p_place_num(i)
      api->begin_rule(BASICRULE);
      sprintf(buf, "f%ld(%ld)", f_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_head(atom);
      ASSERT(node->right != NULL);
      sprintf(buf, "p%ld(%ld)", node->right->place_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_body(atom, FALSE);
      api->end_rule();
      break;

    case AND:
      // f_num(i) :- f1_num(i), f2_num(i)
      api->begin_rule(BASICRULE);
      sprintf(buf, "f%ld(%ld)", f_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_head(atom);
      sprintf(buf, "f%ld(%ld)", f1_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_body(atom, TRUE);
      sprintf(buf, "f%ld(%ld)", f2_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_body(atom, TRUE);
      api->end_rule();
      break;

    case OR:
      // f_num(i) :- f1_num(i)
      // f_num(i) :- f2_num(i)
      api->begin_rule(BASICRULE);
      sprintf(buf, "f%ld(%ld)", f_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_head(atom);
      sprintf(buf, "f%ld(%ld)", f1_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_body(atom, TRUE);
      api->end_rule();
      api->begin_rule(BASICRULE);
      sprintf(buf, "f%ld(%ld)", f_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_head(atom);
      sprintf(buf, "f%ld(%ld)", f2_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_body(atom, TRUE);
      api->end_rule();
      break;

    case DIAMOND:

      // The common case:
      // f_num(i) :- f2_num(i)
      // f_num(i) :- f_num(i+1)

      api->begin_rule(BASICRULE);
      sprintf(buf, "f%ld(%ld)", f_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_head(atom);
      sprintf(buf, "f%ld(%ld)", f2_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_body(atom, TRUE);
      api->end_rule();
      api->begin_rule(BASICRULE);
      sprintf(buf, "f%ld(%ld)", f_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_head(atom);
      sprintf(buf, "f%ld(%ld)", f_num, (i+1));
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_body(atom, TRUE);
      api->end_rule();

      if(i != 0) {

	// The loop handling
	// f_num(bound +1) :- nl(i), f_num(i)

	api->begin_rule(BASICRULE);
	sprintf(buf, "f%ld(%ld)", f_num, (bound +1));
	atom = api->get_atom(buf);
	ASSERT(atom != NULL);
	api->add_head(atom);
	sprintf(buf, "nl(%ld)", i);
	atom = api->get_atom(buf);
	ASSERT(atom != NULL);
	api->add_body(atom, TRUE);
	sprintf(buf, "f%ld(%ld)", f_num, i);
	atom = api->get_atom(buf);
	ASSERT(atom != NULL);
	api->add_body(atom, TRUE);
	api->end_rule();

      }

      break;


    case UNTIL:

      // The easy case, first part:
      // f_num(i) :- f2_num(i)

      api->begin_rule(BASICRULE);
      sprintf(buf, "f%ld(%ld)", f_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_head(atom);
      sprintf(buf, "f%ld(%ld)", f2_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_body(atom, TRUE);
      api->end_rule();

      // The easy case, second part:
      // f_num(i) :- f1_num(i), f_num(i+1)

      api->begin_rule(BASICRULE);
      sprintf(buf, "f%ld(%ld)", f_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_head(atom);
      sprintf(buf, "f%ld(%ld)", f1_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_body(atom, TRUE);
      sprintf(buf, "f%ld(%ld)", f_num, (i+1));
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_body(atom, TRUE);
      api->end_rule();


      if(i != 0) {

	// The loop handling
	// f_num(bound +1) :- nl(i), f_num(i)

	api->begin_rule(BASICRULE);
	sprintf(buf, "f%ld(%ld)", f_num, (bound +1));
	atom = api->get_atom(buf);
	ASSERT(atom != NULL);
	api->add_head(atom);
	sprintf(buf, "nl(%ld)", i);
	atom = api->get_atom(buf);
	ASSERT(atom != NULL);
	api->add_body(atom, TRUE);
	sprintf(buf, "f%ld(%ld)", f_num, i);
	atom = api->get_atom(buf);
	ASSERT(atom != NULL);
	api->add_body(atom, TRUE);
	api->end_rule();

      }
	
      break;

    case BOX:

      // The easy case:
      // f_num(i) :- f2_num(i), f_num(i+1)

      api->begin_rule(BASICRULE);
      sprintf(buf, "f%ld(%ld)", f_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_head(atom);
      sprintf(buf, "f%ld(%ld)", f2_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_body(atom, TRUE);
      sprintf(buf, "f%ld(%ld)", f_num, (i+1));
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_body(atom, TRUE);
      api->end_rule();

      if(i != 0) {

	// Handling of the loop, part (i)
	// cstate(f_num) :- il(i), not f2_num(i)

	api->begin_rule(BASICRULE);
	sprintf(buf, "cstate(f%ld)", f_num);
	atom = api->get_atom(buf);
	if(atom == NULL) {
	  atom = api->new_atom();
	  api->set_name(atom, buf);
	}
	ASSERT(atom != NULL);
	api->add_head(atom);
	sprintf(buf, "il(%ld)", i);
	atom = api->get_atom(buf);
	ASSERT(atom != NULL);
	api->add_body(atom, TRUE);
	sprintf(buf, "f%ld(%ld)", f2_num, i);
	atom = api->get_atom(buf);
	ASSERT(atom != NULL);
	api->add_body(atom, FALSE);
	api->end_rule();
      
      }

      if(i == bound) {

	// The handling of the loop, part (ii)
	// f_num(bound +1) :- l, not cstate(f_num)

	api->begin_rule(BASICRULE);
	sprintf(buf, "f%ld(%ld)", f_num, (bound +1));
	atom = api->get_atom(buf);
	ASSERT(atom != NULL);
	api->add_head(atom);
	atom = api->get_atom("l");
	ASSERT(atom != NULL);
	api->add_body(atom, TRUE);
	sprintf(buf, "cstate(f%ld)", f_num);
	atom = api->get_atom(buf);
	ASSERT(atom != NULL);
	api->add_body(atom, FALSE);
	api->end_rule();

      }

      break;

    case RELEASE:

      // The easy case, first part:
      // f_num(i) :- f2_num(i), f1_num(i)

      api->begin_rule(BASICRULE);
      sprintf(buf, "f%ld(%ld)", f_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_head(atom);
      sprintf(buf, "f%ld(%ld)", f2_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_body(atom, TRUE);
      sprintf(buf, "f%ld(%ld)", f1_num, (i));
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_body(atom, TRUE);
      api->end_rule();

      // The easy case, second part:
      // f_num(i) :- f2_num(i), f_num(i+1)

      api->begin_rule(BASICRULE);
      sprintf(buf, "f%ld(%ld)", f_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_head(atom);
      sprintf(buf, "f%ld(%ld)", f2_num, i);
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_body(atom, TRUE);
      sprintf(buf, "f%ld(%ld)", f_num, (i+1));
      atom = api->get_atom(buf);
      ASSERT(atom != NULL);
      api->add_body(atom, TRUE);
      api->end_rule();

      if(i != 0) {

	// The loop handling, part (i)
	// f_num(bound +1) :- nl(i), f_num(i)

	api->begin_rule(BASICRULE);
	sprintf(buf, "f%ld(%ld)", f_num, (bound +1));
	atom = api->get_atom(buf);
	ASSERT(atom != NULL);
	api->add_head(atom);
	sprintf(buf, "nl(%ld)", i);
	atom = api->get_atom(buf);
	ASSERT(atom != NULL);
	api->add_body(atom, TRUE);
	sprintf(buf, "f%ld(%ld)", f_num, i);
	atom = api->get_atom(buf);
	ASSERT(atom != NULL);
	api->add_body(atom, TRUE);
	api->end_rule();

	// Handling of the loop, part (ii)
	// cstate(f_num) :- il(i), not f2_num(i)

	api->begin_rule(BASICRULE);
	sprintf(buf, "cstate(f%ld)", f_num);
	atom = api->get_atom(buf);
	if(atom == NULL) {
	  atom = api->new_atom();
	  api->set_name(atom, buf);
	}
	ASSERT(atom != NULL);
	api->add_head(atom);
	sprintf(buf, "il(%ld)", i);
	atom = api->get_atom(buf);
	ASSERT(atom != NULL);
	api->add_body(atom, TRUE);
	sprintf(buf, "f%ld(%ld)", f2_num, i);
	atom = api->get_atom(buf);
	ASSERT(atom != NULL);
	api->add_body(atom, FALSE);
	api->end_rule();

      }

      if(i == bound) {

	// The handling of the loop, part (iii)
	// f_num(bound +1) :- l, not cstate(f_num)

	api->begin_rule(BASICRULE);
	sprintf(buf, "f%ld(%ld)", f_num, (bound +1));
	atom = api->get_atom(buf);
	ASSERT(atom != NULL);
	api->add_head(atom);
	atom = api->get_atom("l");
	ASSERT(atom != NULL);
	api->add_body(atom, TRUE);
	sprintf(buf, "cstate(f%ld)", f_num);
	atom = api->get_atom(buf);
	ASSERT(atom != NULL);
	api->add_body(atom, FALSE);
	api->end_rule();

      }

      break;
      
    default:
      ASSERT(FALSE);
      break;

    }

  }

  return (f_num);

}


SET *
GetVisiblePlaces(TreeNode *node, PTNet *net)
{
  SET *vis_places;

  vis_places = CreateSet(net->getNumPlaces());
  AddVisiblePlaces(node, vis_places);
  return(vis_places);
}

void
AddVisiblePlaces(TreeNode *node, SET *set)
{
  ASSERT(node != NULL);

  switch (node->info) {
  case RELEASE:
  case UNTIL:
  case OR:
  case AND:
    AddVisiblePlaces(node->left, set);
    AddVisiblePlaces(node->right, set);
    break;
  case BOX:
  case DIAMOND:
  case NOT:
    ASSERT(node->left == NULL);
    AddVisiblePlaces(node->right, set);
    break;
  case PLACE:
    AddtoSet(set, (node->place_num -1));
    break;
  default:
    ASSERT(FALSE);
    break;
  }

}

SET *
GetVisibleTransitions(TreeNode *node, PTNet *net)
{
  unsigned long i;
  unsigned long j;
  Arc *ac;
  
  Place **places;
  Transition **transitions;
  unsigned long numplaces;
  unsigned long numtransitions;
  unsigned long p;
  unsigned long size;
  
  SET *vis_places;
  SET *vis_transitions;
  SET *changed;
  
  numplaces = net->getNumPlaces();
  numtransitions = net->getNumTransitions();
  places = net->getPlaces();
  transitions = net->getTransitions();

  vis_places = GetVisiblePlaces(node, net);
  vis_transitions = CreateSet(numtransitions);

  changed = CreateSet(numplaces);
  
  for(i = 0; i < numtransitions; i++) {


    ClearSet(changed);
    
    for(ac = transitions[i]->preset;
	ac != NULL;
	ac = ac->nextplace) {

      p = ac->place->id-1;
      AddtoSet(changed, p);

    }

    for(ac = transitions[i]->postset;
	ac != NULL;
	ac = ac->nextplace) {

      p = ac->place->id-1;
      if(InSet(changed, p)) {
	RemovefromSet(changed, p);
      }

    }

    // Changed contains places from which transition i effectively removes tokens
    
    for(j = 0, size = GetSetSize(changed); j < size ; j++) {

      p = GetSetItem(changed, j);
      if(InSet(vis_places, p)) {
	AddtoSet(vis_transitions, i);
      }
      
    }
    

    ClearSet(changed);
    
    for(ac = transitions[i]->postset;
	ac != NULL;
	ac = ac->nextplace) {

      p = ac->place->id-1;
      AddtoSet(changed, p);

    }

    for(ac = transitions[i]->preset;
	ac != NULL;
	ac = ac->nextplace) {

      p = ac->place->id-1;
      if(InSet(changed, p)) {
	RemovefromSet(changed, p);
      }

    }

    // Changed contains places from which transition i effectively adds tokens

    for(j = 0, size = GetSetSize(changed); j < size ; j++) {

      p = GetSetItem(changed, j);
      if(InSet(vis_places, p)) {
	AddtoSet(vis_transitions, i);
      }
      
    }

  }  

  
  DeleteSet(changed);
  DeleteSet(vis_places);

  return(vis_transitions);
}
