// Copyright 1998 by Keijo Heljanko
// This software is provided as is, no warranty of any kind is given.

#include <stdio.h>
#include <stdlib.h>
#include "dassert.h"
#include "shash.h"
#include "alloc.h"

SIMPLEHASH * CreateSimpleHash(unsigned long max_items)
{
  SIMPLEHASH * hash;
  unsigned long array_items;
  unsigned long i;
  
  // Round to the next power of two.

  for(array_items = 2; array_items < max_items; array_items = 2*array_items) {
    ;
  }

  // Add one more power of two to ensure < 50% hash table fill factor.

  array_items = 2*array_items;

  hash = (SIMPLEHASH *) xmalloc(sizeof(SIMPLEHASH) +
				((array_items -1) * sizeof(SIMPLE_HASH_ITEM)));

  hash->array_items = array_items;
  hash->mask = array_items-1;
  hash->max_items = max_items;
  hash->num_items = 0;

  for(i = 0; i < array_items; i++) {
    hash->items[i].data = (SIMPLE_HASH_DATA *) NULL;
    hash->items[i].signature = 0;
  }
  
  return hash;

}

void DeleteSimpleHash(SIMPLEHASH * hash)
{

  if(hash != NULL) {
    free(hash);
  }

}

void ClearSimpleHash(SIMPLEHASH * hash)
{
  unsigned long array_items;
  unsigned long i;

  array_items = hash->array_items;
  
  hash->num_items = 0;

  for(i = 0; i < array_items; i++) {
    hash->items[i].data = (SIMPLE_HASH_DATA *) NULL;
    hash->items[i].signature = 0;
  }

}

BOOL InsertDataTellIfNew(SIMPLEHASH * hash, SIMPLE_HASH_DATA * data)
{
  if(InsertDataIfNew(hash, data) == NULL) {
    return TRUE;
  } else {
    return FALSE;
  }
}


SIMPLE_HASH_DATA * InsertDataIfNew(SIMPLEHASH * hash, SIMPLE_HASH_DATA * data)
{
  BOOL data_mismatch;
  unsigned long mask;
  unsigned long signature;
  unsigned long i;
  unsigned long j;
  unsigned long data_len;
  
  mask = hash->mask;
  data_len = data[0];

  ASSERT(data_len != 0);
  
  signature = ulhash((unsigned char *) (&data[1]),
		     sizeof(unsigned long) * data_len,
		     1);
  i = signature;
  
  for(; ; i++) {

    // Do wraparound.
    
    i = i & mask;

    if(hash->items[i].data == NULL) {

      // Empty slot found, fill it

      DEBUG_ASSERT(hash->num_items != hash->max_items);
      
      hash->items[i].data = data;
      hash->items[i].signature = signature;
      hash->num_items++;
      
      return (SIMPLE_HASH_DATA *)NULL;

    } else {

      // Filled slot found, check if the same data

      if(signature != hash->items[i].signature) {

	// Signature mismatch, continue
	
	continue;

      } else {

	// Signatures match, do the datas match?

	if(data_len != hash->items[i].data[0]) {

	  // Sizes do not match, continue

	  continue;

	  
	} else {

	  data_mismatch = FALSE;
	  
	  for(j = 1; j <= data_len; j++) {

	    if(data[j] != hash->items[i].data[j]) {

	      // Datas do not match, continue

	      data_mismatch = TRUE;
	      break;

	    }
	    
	  }

	  if(data_mismatch == TRUE) {
	    // Datas do not match, continue
	    continue;

	  }

	  // Everything matches, this data existed before,
	  // return.

	  return (SIMPLE_HASH_DATA *)hash->items[i].data;

	}	

      }

	
    }
    
  }

  // NOTREACHED
  
  ASSERT(FALSE);
  
  return (SIMPLE_HASH_DATA *)NULL;
  
}

