// Copyright 1998,2000 by Keijo Heljanko
// This software is provided as is, no warranty of any kind is given.

#include <stdio.h>
#include <stdlib.h>
#include "dassert.h"
#include "alloc.h"
#include "dhash.h"
#include "bitarr.h"

DYNHASH *CreateDynHash(unsigned long initial_items)
{
  DYNHASH *hash;
  unsigned long array_items;
  unsigned long i;
  DYN_HASH_ITEM *items;

  if(initial_items < DYN_HASH_MINSIZE) {
    initial_items = DYN_HASH_MINSIZE;
  }
  
  hash = (DYNHASH *)xmalloc(sizeof(DYNHASH));

  // Round to the next power of two.

  for(array_items = 2; array_items < initial_items; array_items = 2*array_items) {
    ;
  }

  // OK, when max_items items have been inserted, allocate more space

  hash->max_items = array_items;


  // Add one more power of two to ensure < 50% hash table fill factor.

  array_items = 2*array_items;

  items = (DYN_HASH_ITEM *)xmalloc(array_items * sizeof(DYN_HASH_ITEM));

  hash->items = items;
  hash->array_items = array_items;
  hash->mask = array_items-1;
  hash->num_items = 0;

  for(i = 0; i < array_items; i++) {
    items[i].data = (DYN_HASH_DATA *)NULL;
    items[i].signature = 0;
  }
  
  return hash;

}

void DeleteDynHash(DYNHASH *hash)
{

  if(hash != NULL) {
    if(hash->items != NULL) {
      free(hash->items);
    }
    free(hash);
  }

}

void ClearDynHash(DYNHASH *hash)
{
  unsigned long array_items;
  unsigned long i;
  DYN_HASH_ITEM *items;

  array_items = hash->array_items;
  
  hash->num_items = 0;
  items = hash->items;

  for(i = 0; i < array_items; i++) {
    items[i].data = (DYN_HASH_DATA *)NULL;
    items[i].signature = 0;
  }

}

BOOL InsertDataTellIfNew(DYNHASH *hash, DYN_HASH_DATA *data)
{
  if(InsertDataIfNew(hash, data) == NULL) {
    return TRUE;
  } else {
    return FALSE;
  }
}


DYN_HASH_DATA *InsertDataIfNew(DYNHASH *hash, DYN_HASH_DATA *data)
{
  BOOL data_mismatch;
  unsigned long mask;
  unsigned long signature;
  unsigned long i;
  unsigned long j;
  unsigned long data_len;
  DYN_HASH_ITEM *items;
  
  mask = hash->mask;
  data_len = data[0];

  ASSERT(data_len != 0);
  
  signature = ulhash((unsigned char *) (&data[1]),
		     sizeof(unsigned long) * data_len,
		     1); // Arbitrary seed
  i = signature;

  items = hash->items;
  
  for(; ; i++) {

    // Do wraparound.
    
    i = i & mask;

    if(items[i].data == NULL) {

      // Empty slot found, fill it

      DEBUG_ASSERT(hash->num_items < hash->max_items);
      
      items[i].data = data;
      items[i].signature = signature;
      hash->num_items++;

      if(hash->num_items >= hash->max_items) {
	ExpandDynHash(hash); // Make the hash table twice as big
      }

      
      return (DYN_HASH_DATA *)NULL;

    } else {

      // Filled slot found, check if the same data

      if(signature != items[i].signature) {

	// Signature mismatch, continue
	
	continue;

      } else {

	// Signatures match, do the datas match?

	if(data_len != items[i].data[0]) {

	  // Sizes do not match, continue

	  continue;

	  
	} else {

	  data_mismatch = FALSE;
	  
	  for(j = 1; j <= data_len; j++) {

	    if(data[j] != items[i].data[j]) {

	      // Datas do not match, continue

	      data_mismatch = TRUE;
	      break;

	    }
	    
	  }

	  if(data_mismatch == TRUE) {
	    // Datas do not match, continue
	    continue;

	  }

	  // Everything matches, this data existed before,
	  // return.

	  return (DYN_HASH_DATA *)items[i].data;

	}	

      }

	
    }
    
  }

  // NOTREACHED
  
  ASSERT(FALSE);
  
  return (DYN_HASH_DATA *)NULL;
  
}



// Only for internal use!

void ExpandDynHash(DYNHASH *hash)
{
  unsigned long old_size;
  unsigned long new_size;
  unsigned long i;
  DYN_HASH_ITEM *items;

  // First double all the values

  old_size = hash->array_items;
  new_size =  2 * old_size;

  hash->max_items = old_size;
  hash->array_items = new_size;
  hash->mask = new_size -1;

  // Then reallocate the hash table

  items = (DYN_HASH_ITEM *)xrealloc(hash->items,
				     (new_size * sizeof(DYN_HASH_ITEM)));
  hash->items = items;


  // Clear all the new entrys

  for(i = old_size; i < new_size; i++) {
    items[i].data = (DYN_HASH_DATA *)NULL;
    items[i].signature = 0;
  }

  // Then rehash all the entrys

  ExpandRehashDynHash(hash);
 
}



// Only for internal use!

void
ExpandRehashDynHash(DYNHASH *hash)

{
  unsigned long mask;
  unsigned long i;
  unsigned long j;
  unsigned long old_size;
  DYN_HASH_ITEM *items;
  DYN_HASH_DATA *data;
  unsigned long signature;
  DYN_HASH_DATA *data2;
  unsigned long signature2;
  BOOL done;
  BITARRAY *fixed;

  old_size = hash->array_items/2; // The code depends on size of 2 increases
  mask = hash->mask;
  items = hash->items;


  // Then rehash all the entrys
  // 
  // A bitmap is used to check which entrys are already
  // fixed, and which ones can be moved during reahashing

  fixed = CreateBitArray(old_size);


  for(i = 0; i < old_size; i++) {

    if(items[i].data == NULL) {

      // Only rehash non-empty entrys

      continue;

    }

    if(GetBit(fixed, i) == FALSE) {

      // OK, This item needs yet to be fixed

      // Get the item, and clear the slot
      data = items[i].data;
      signature = items[i].signature;

      if((signature & mask) == i) {

	// Special case when the thing is already
	// at the right slot.

	SetBit(fixed, i);
	continue;

      }

      // OK, so the thing was not at the right slot, have to move
      //
      // Clear the slot first

      items[i].data = (DYN_HASH_DATA *)NULL;
      items[i].signature = 0;



      // Add the data to the hash table.
      // The bitarray fixed gives info on which slots have been
      // fixed. All other slots might be overwritten, as long as
      // the contents are also moved.

      j = signature;
      for(done = FALSE; done == FALSE; j++) {

	// Do wraparound.
    
	j = j & mask;

	if(j < old_size) {

	  // This is in the range which might contain fixed and
	  // non-fixed stuff. Check which case we are in.

	  if(GetBit(fixed, j) == FALSE) {

	    // Non-fixed slot found, fill it

	    if(items[j].data == NULL) {

	      // The slot is really empty, add the data
	      // and be done with it.

	      items[j].data = data;
	      items[j].signature = signature;

	      // This slot is now fixed, mark it

	      SetBit(fixed, j);

	      // Nothing more to do, return

	      done = TRUE;

	    } else {

	      // A filled non-fixed slot found, overwrite it
	      // with the data and find a new place for the
	      // overwritten data after that.

	      data2 = items[j].data;
	      signature2 = items[j].signature;

	      items[j].data = data;
	      items[j].signature = signature;

	      // This slot is now fixed, mark it

	      SetBit(fixed, j);

	      // Now we have to find a place for the data
	      // we threw away from the slot. Restart the
	      // search for a slot with this data.

	      data = data2;
	      signature = signature2;

	      // Search for a slot, start using the new signature

	      j = signature;

	    }


	  } else {

	    // A filled fixed slot found. It can't be overwritten,
	    // try the next slot

	    ;

	  }	


	} else {


	  // We are in the new part. This contains only fixed stuff,
	  // so not bitmap handling is needed.

	  if(items[j].data == NULL) {

	    // The slot is really empty, add the data
	    // and be done with it.

	    items[j].data = data;
	    items[j].signature = signature;

	    // Nothing more to do, return

	    done = TRUE;

	  } else {

	    // A filled slot found. It can't be overwritten,
	    // try the next slot

	    ;

	  }

	}

      }

    }

  }


  DeleteBitArray(fixed);

}

