#include <limits.h>
#include "alloc.h"
#include "dassert.h"
#include "darray.h"
#include "buchi.h"
#include "ptnet.h"



Buchi::Buchi(PTNet *in_net)
{
  unsigned long i;
  unsigned long j;
  unsigned long size;
  unsigned long place;
  unsigned long mapping;
  unsigned long from;
  unsigned long to;
  BOOL found;
  Place **places;
  Transition **transitions;

  net = in_net;

  places = net->getPlaces();
  transitions = net->getTransitions();

  num_buchi_states = 0;
  for(i = 0, size = net->getNumPlaces();
      i < size;
      i++) {

    if(net->isBuchiPlace(i+1)) {

      num_buchi_states++;

    }

  }


  // Allocate some memory for quick lookup
  // 0 = Uninteresting place
  // 1...num_atomic = Atomic propositions as positive
  // (num_atomic+1)...2*num_atomic = Atomic propositions as negative
  // (2*num_atomic+i)... = Buchi state (i-1)

  place_mapping = (unsigned long *)xmalloc(size *
					   sizeof(unsigned long));

  for(i = 0; i < size; i++) {
    place_mapping[i] = 0;
  }


  num_buchi_trans = 0;
  for(i = 0, size = net->getNumTransitions();
      i < size;
      i++) {

    if((net->isLTransition(i+1) == FALSE) &&
       net->isBuchiTransition(i+1)) {

      num_buchi_trans++;

    }

  }


  // Calculate how many observable places there are

  num_observables = 0;
  for(i = 0, size = net->getNumPlaces();
      i < size;
      i++) {

    if(net->isObservablePlace(i+1)) {

      num_observables++;

    }

  }


  // Allocate memory to store the observable places

  observables = (unsigned long *)xmalloc(num_observables *
					 sizeof(unsigned long));
  obs_atomic = (unsigned long *)xmalloc(num_observables *
					sizeof(unsigned long));
  obs_names = (char **)xmalloc(num_observables *
			       sizeof(char *));


  // Store the observable indexes and names

  num_observables = 0;
  for(i = 0, size = net->getNumPlaces();
      i < size;
      i++) {

    if(net->isObservablePlace(i+1)) {

      observables[num_observables] = i;
      obs_atomic[num_observables] = 0;
      obs_names[num_observables] = net->getObservableName(i + 1);

      num_observables++;

    }

  }


  // Allocate an atomic proposition name for each observable using the scheme

  num_atomic = 0;
  for(i = 0; i < num_observables; i++) {

    if(obs_atomic[i] == 0) {
      if(obs_names[i][0] == '!') {

	found = FALSE;
	for(j = 0; j < i; j++) {
	  if(strcmp(obs_names[j], &obs_names[i][1]) == 0) {
	    found = TRUE;
	    break;
	  }
	}
	if(found != FALSE) {
	  obs_atomic[i] = obs_atomic[j];
	  continue;
	}

      }

      num_atomic++;
      obs_atomic[i] = num_atomic;

    }

  }


  default_negative_ap = CreateBitArray(num_atomic);



  // Shift the negative atoms up by the num_atomic

  for(i = 0; i < num_observables; i++) {
      
    if(obs_names[i][0] == '!') {

      obs_atomic[i] += num_atomic;

    }

  }


  // Find the atomic propositions which only appear as negative
  
  for(i = 0; i < num_atomic; i++) {

    found = FALSE;

    for(j = 0; j < num_observables; j++) {
      
      if(obs_atomic[j] == (i+1)) {

	found = TRUE;
	break;
      }

    }

    if(found == FALSE) {
      SetBit(default_negative_ap, i);
    }

  }


  // Set up the reverse mapping from places

  for(i = 0; i < num_observables; i++) {

    place_mapping[observables[i]] = obs_atomic[i];

  }


  // Set up the reverse mapping for buch states

  num_buchi_states = 0;
  for(i = 0, size = net->getNumPlaces();
      i < size;
      i++) {

    if(net->isBuchiPlace(i+1)) {

      ASSERT(place_mapping[i] == 0);
      num_buchi_states++;
      place_mapping[i] = ((2 * num_atomic) +
			  num_buchi_states);

    }

  }


  // Allocate memory to store the arc counts & offsets & arcs

  num_arcs = (unsigned long *)xmalloc(num_buchi_states *
				      sizeof(unsigned long));
  arc_offsets = (unsigned long *)xmalloc(num_buchi_states *
					 sizeof(unsigned long));
  
  buchi_arcs = (BuchiArc *)xmalloc(num_buchi_trans *
				   sizeof(BuchiArc));
  
  // First count the arcs for each state
  
  for(i = 0, size = num_buchi_states; i < size; i++) {
    num_arcs[i] = 0;
  }

  for(i = 0, size = net->getNumTransitions();
      i < size;
      i++) {

    if((net->isLTransition(i+1) == FALSE) &&
       net->isBuchiTransition(i+1)) {

      place = net->getBuchiPrePlace(i+1);

      mapping = place_mapping[place -1];
      ASSERT(mapping > (2*num_atomic));
      ASSERT(mapping <= (2*num_atomic+num_buchi_states));
      from = mapping - ((2 * num_atomic) +1);

      // Count the number of arcs out of this state
      
      num_arcs[from]++;
      
    }

  }

  // Then count the offsets into the arcs array
  
  for(i = 0, j = 0, size = num_buchi_states; i < size; i++) {

    arc_offsets[i] = j;
    j += num_arcs[i];
    
  }
  ASSERT(j == num_buchi_trans);


  // Reuse the counts array for real arc insertion

  for(i = 0, size = num_buchi_states; i < size; i++) {
    num_arcs[i] = 0;
  }

  // Create the arcs

  for(i = 0, size = net->getNumTransitions();
      i < size;
      i++) {

    if((net->isLTransition(i+1) == FALSE) &&
       net->isBuchiTransition(i+1)) {

      
      place = net->getBuchiPrePlace(i+1);

      mapping = place_mapping[place -1];
      ASSERT(mapping > (2*num_atomic));
      ASSERT(mapping <= (2*num_atomic+num_buchi_states));
      from = mapping - ((2 * num_atomic) +1);

      place = net->getBuchiPostPlace(i+1);

      mapping = place_mapping[place -1];
      ASSERT(mapping > (2*num_atomic));
      ASSERT(mapping <= (2*num_atomic+num_buchi_states));
      to = mapping - ((2 * num_atomic) +1);

      // Add the arc in a subroutine
      
      AddArc((i+1), from, to, (&buchi_arcs[(arc_offsets[from] + num_arcs[from])]));
      
      // Count the number of arcs out of this state
      
      num_arcs[from]++;
      
    }

  }

  accepting_states = CreateBitArray(num_buchi_states);

  // Find the accepting Buchi states from the postsets of accepting
  // Buchi transitions
  
  for(i = 0, size = net->getNumTransitions();
      i < size;
      i++) {

    if((net->isLTransition(i+1) == FALSE) &&
       net->isBuchiTransition(i+1) &&
       net->isAccepting(i+1)) {

      place = net->getBuchiPostPlace(i+1);

      mapping = place_mapping[place -1];
      ASSERT(mapping > (2*num_atomic));
      ASSERT(mapping <= (2*num_atomic+num_buchi_states));
      to = mapping - ((2 * num_atomic) +1);

      SetBit(accepting_states, to);
      
    }

  }


  // Allocate stuff for the Tarjan's algorithm

  marked_bits = CreateBitArray(num_buchi_states);
  modified_bits = CreateBitArray(num_buchi_states);

  stack = CreateULStack();
  father_stack = CreateULStack();

  min_numbers = (unsigned long *)xmalloc(num_buchi_states *
					 sizeof(unsigned long));

  for(i = 0; i < num_buchi_states; i++) {
    min_numbers[i] = 0;
  }
  
}


Buchi::~Buchi()
{
  unsigned long i;

  free(min_numbers);
  DeleteULStack(father_stack);
  DeleteULStack(stack);
  DeleteBitArray(modified_bits);
  DeleteBitArray(marked_bits);
  DeleteBitArray(accepting_states);
  for(i = 0; i < num_buchi_trans; i++) {
    if(buchi_arcs[i].pos != NULL) {
      DeleteSArray(buchi_arcs[i].pos);
    }
    if(buchi_arcs[i].neg != NULL) {
      DeleteSArray(buchi_arcs[i].neg);
    }
  }
  free(buchi_arcs);
  free(arc_offsets);
  free(num_arcs);
  DeleteBitArray(default_negative_ap);
  for(i = 0; i < num_observables; i++) {
    free(obs_names[i]);
  }
  free(obs_names);
  free(obs_atomic);
  free(observables);
  free(place_mapping);
}


BOOL
Buchi::CanStutter(unsigned long transition_idx,
 		  unsigned long num_places,
		  unsigned long *places)
{
  unsigned long i;
  unsigned long size;
  unsigned long *t;
  unsigned long place;
  unsigned long mapping;
  unsigned long buchi_state;
#if 0
  BOOL first;
#endif
  BOOL result;
  BITARRAY *valuation;
#if 0
  Arc *ac;
  Transition **transitions;
#endif
  
  valuation = CreateBitArray(num_atomic);

  for(i = 0; i < num_atomic; i++) {

    if(GetBit(default_negative_ap, i)) {

      // This atomic propositions is TRUE unless shown otherwise

      SetBit(valuation, i);

    }

  }
  
  // Get the valuation, also check that there is at least
  // one buchi state

  size = num_places;
  t = places;
  buchi_state = ULONG_MAX;
  for(i = 0; i < size; i++) {

    place = *t++;
    mapping = place_mapping[place];

    if(mapping == 0) {
      continue;
    } else if(mapping <= num_atomic) {
      SetBit(valuation, (mapping -1));
    } else if(mapping <= (2*num_atomic)) {
      ClearBit(valuation, (mapping - (num_atomic +1)));
    } else if(mapping <= (2*num_atomic+num_buchi_states)) {
      ASSERT(buchi_state == ULONG_MAX);
      buchi_state = mapping - ((2 * num_atomic) +1);
    } else {
      // Mapping out of bounds!
      ASSERT(FALSE);
    }

  }
  ASSERT(buchi_state != ULONG_MAX);


  // The following code is removed to make the implementation consistent
  // with the paper. (This change might also make sense from
  // an optimization prespective.)

#if 0

  // Actually the real buchi state must be read from the
  // transition postset!

  // Note: This one relies on the non-L-version of the
  // same transition being in one smaller index in the transition table!
  // TODO: Remove this dependency!

  buchi_state = ULONG_MAX;
  transitions = net->getTransitions();
  for(ac = transitions[transition_idx -1]->postset;
      ac != NULL;
      ac = ac->nextplace) {

    place = ac->place->id -1;
    mapping = place_mapping[place];

    if(mapping >= ((2 * num_atomic) +1)) {

      if(mapping <= (2*num_atomic+num_buchi_states)) {

	ASSERT(buchi_state == ULONG_MAX);
	buchi_state = mapping - ((2 * num_atomic) +1);

      } else {

	// Mapping state out of range!

	ASSERT(FALSE);
      }

    }

  }
  ASSERT(buchi_state != ULONG_MAX);
#endif


#if 0
  fprintf(stdout, "Buchi state: %lu\n", buchi_state);

  fprintf(stdout, "Valuation: < ");
  first = TRUE;
  for(i = 0, size = num_atomic; i < size; i++) {

    if(first == FALSE) {
      fprintf(stdout, ", ");
    }

    if(GetBit(valuation, i) == FALSE) {
      fprintf(stdout, "!");
    }
    // TODO: Use real names!

    fprintf(stdout, "AP%lu", i);

    first = FALSE;

  }
  fprintf(stdout, " >\n");
#endif

  result = CheckStuttering(buchi_state, valuation);

  DeleteBitArray(valuation);

  return(result);

}

void
Buchi::AddArc(unsigned long transition_id,
	      unsigned long from,
	      unsigned long to,
	      BuchiArc *arc)
{
  Transition **tr;
  Place **pl;
  Arc *ac;
  DARRAY *pos;
  DARRAY *neg;
  unsigned long i;
  unsigned long size;
  unsigned long place_id;
  char *name;
  BOOL found;
  
  arc->from_state = from;
  arc->to_state = to;

  pos = CreateDArray();
  neg = CreateDArray();

  tr = net->getTransitions();
  pl = net->getPlaces();
  for(ac = tr[transition_id -1]->preset;
      ac != NULL;
      ac = ac->nextplace) {

    place_id = ac->place->id;

    if(net->isObservablePlace(place_id)) {

      name = net->getObservableName(place_id);

      found = FALSE;
      for(i = 0; i < num_observables; i++) {

	if(strcmp(obs_names[i], name) == 0) {

	  if(name[0] == '!') {
	    ASSERT(obs_atomic[i] >= (num_atomic +1));
	    ASSERT(obs_atomic[i] <= (2*num_atomic));
	    AppendDArray(neg, (obs_atomic[i]-(num_atomic +1)));
	  } else {
	    ASSERT(obs_atomic[i] >= 1);
	    ASSERT(obs_atomic[i] <= num_atomic);
	    AppendDArray(pos, (obs_atomic[i] -1));
	  }

	  found = TRUE;
	  break;
	  
	}
	
      }

      free(name);

      ASSERT(found != FALSE);
      
    }
    
  }

  size = GetDArraySize(pos);
  arc->pos = (SARRAY *)NULL;
  if(size > 0) {
    arc->pos = CreateSArray(size);
  }
  for(i = 0; i < size; i++) {
    AppendSArray(arc->pos, GetDArrayItem(pos, i));
  }

  size = GetDArraySize(neg);
  arc->neg = (SARRAY *)NULL;
  if(size > 0) {
    arc->neg = CreateSArray(size);
  }
  for(i = 0; i < size; i++) {
    AppendSArray(arc->neg, GetDArrayItem(neg, i));
  }

  DeleteDArray(pos);
  DeleteDArray(neg);
 
}


BOOL
Buchi::CheckStuttering(unsigned long buchi_state,
		       BITARRAY *valuation)
{

  // TODO: Add some caching!

  return(CheckTarjan(buchi_state, valuation));
  
}

unsigned long
Buchi::GetNextValuationRespectingSuccesor(unsigned long succ_num,
					  unsigned long s,
					  BITARRAY *valuation)
{
  BuchiArc *arc;
  SARRAY *sa;
  unsigned long i;
  unsigned long size;
  unsigned long t;
  BOOL bad_valuation;
  
  while(succ_num != 0) {
    succ_num--;
    arc = &buchi_arcs[arc_offsets[s] + succ_num];
    bad_valuation = FALSE;
    sa = arc->pos;
    if(sa != NULL) {
      for(i = 0, size = GetSArraySize(sa); i < size; i++) {
	t = GetSArrayItem(sa, i);
	if(GetBit(valuation, t) == FALSE) {
	  bad_valuation = TRUE;
	  break;
	}
      }
    }
    if(bad_valuation == FALSE) {
      sa = arc->neg;
      if(sa != NULL) {
	for(i = 0, size = GetSArraySize(sa); i < size; i++) {
	  t = GetSArrayItem(sa, i);
	  if(GetBit(valuation, t) != FALSE) {
	    bad_valuation = TRUE;
	    break;
	  }
	}
      }
    }
    if(bad_valuation == FALSE) {
      return(succ_num);
    }
  }

  return(ULONG_MAX);

}
  

BOOL
Buchi::SelfLoopsWithValuation(unsigned long s,
			      BITARRAY *valuation)
{
  BuchiArc *arc;
  SARRAY *sa;
  unsigned long i;
  unsigned long size;
  unsigned long t;
  BOOL bad_valuation;

  for(i = 0, size = num_arcs[s]; i < size; i++) {

    arc = &buchi_arcs[arc_offsets[s] + i];
    if(arc->to_state != s) {
      continue;
    }
    ASSERT(arc->from_state == s);

    bad_valuation = FALSE;
    sa = arc->pos;
    if(sa != NULL) {
      for(i = 0, size = GetSArraySize(sa); i < size; i++) {
	t = GetSArrayItem(sa, i);
	if(GetBit(valuation, t) == FALSE) {
	  bad_valuation = TRUE;
	  break;
	}
      }
    }
    if(bad_valuation == FALSE) {
      sa = arc->neg;
      if(sa != NULL) {
	for(i = 0, size = GetSArraySize(sa); i < size; i++) {
	  t = GetSArrayItem(sa, i);
	  if(GetBit(valuation, t) != FALSE) {
	    bad_valuation = TRUE;
	    break;
	  }
	}
      }
    }
    if(bad_valuation == FALSE) {
      return(TRUE);
    }

  }

  return(FALSE);
  
}


BOOL
Buchi::CheckTarjan(unsigned long buchi_state,
		   BITARRAY *valuation)
{
  BITARRAY *marked;
  BITARRAY *modified;
  UL_STACK *st;
  UL_STACK *fst;
  unsigned long *nums;
  unsigned long dfs;
  unsigned long s;
  unsigned long t;
  unsigned long succ_num;
  unsigned long succ_min;
  unsigned long component_size;
  BOOL accepting_state_found;
  
  marked = marked_bits;
  modified = modified_bits;
  st = stack;
  fst = father_stack;
  nums = min_numbers;
  dfs = 1;
  s = buchi_state;

  BitArrayClear(marked); // With some additional bookkeeping this would not be needed
  
  while(TRUE) {

    while(TRUE) {

      if(GetBit(marked, s) == FALSE) {

	ASSERT(GetBit(modified, s) == FALSE);
	ASSERT(nums[s] == 0);

	SetBit(marked, s);
	nums[s] = dfs;
	dfs++;

	succ_num = num_arcs[s];

	ASSERT(succ_num > 0);
	
	PushULStack(fst, s);
	PushULStack(st, s);
	succ_num = GetNextValuationRespectingSuccesor(succ_num, s, valuation);
	PushULStack(st, succ_num);
	if(succ_num == ULONG_MAX) {
	  break;
	}

	s = buchi_arcs[arc_offsets[s] + succ_num].to_state;
	
      } else {

	break;
	
      }
      
    }

    while(TRUE) {

      if(ULStackEmpty(st) != FALSE) {

	ASSERT(ULStackEmpty(fst) != FALSE);
	
	return(FALSE);

      }

      succ_min = nums[s];

      succ_num = RemoveULStack(st);
      s = RemoveULStack(st);

      if(succ_num != ULONG_MAX) {

	if((succ_min != 0) &&
	   (nums[s] > succ_min)) {

	  nums[s] = succ_min;
	  SetBit(modified, s);
	
	}

	succ_num = GetNextValuationRespectingSuccesor(succ_num, s, valuation);

      }

      if(succ_num != ULONG_MAX) {

	PushULStack(st, s);
	PushULStack(st, succ_num);
	s = buchi_arcs[arc_offsets[s] + succ_num].to_state;
	break;

      } else {

	if(GetBit(modified, s) == FALSE) {

	  component_size = 0;
	  accepting_state_found = FALSE;
	  do {
	    t = RemoveULStack(fst);
	    if(GetBit(accepting_states, t) != FALSE) {
	      accepting_state_found = TRUE;
	    }
	    nums[t] = 0;
	    ClearBit(modified, t);
	    component_size++;
	  } while (t != s);

	  if(accepting_state_found != FALSE) {

	    if((component_size > 1) ||
	       SelfLoopsWithValuation(s, valuation)) {

	      while(ULStackEmpty(fst) == FALSE) {
		t = RemoveULStack(fst);
		nums[t] = 0;
		ClearBit(modified, t);	    
	      }

	      ClearULStack(st);

	      return(TRUE);
	      
	    }
	      
	  }
  
	}
	
      }

    }

  }

  // NOTREACHED
  
  ASSERT(FALSE);
    
  return(FALSE);

}
