/*
 * Copyright Burkhard Graves, PEP project, University of Hildesheim
 * See: http://www.informatik.uni-hildesheim.de/~pep/HomePage.html 
 */

/*****************************/
/* set.c                     */
/* Burkhard Graves, 26.10.94 */
/*****************************/

#include "set.h"

#if 0           /* zwei Versionen zur Bestimmung der Position ... */
#define VERSION1
#undef  VERSION2
#else           /* ... eines Elementes in einer Menge.            */
#define VERSION2
#undef  VERSION1
#endif

#if 0           /* zum Austesten */
#define TEST
#endif

#define SECTOR_SIZE_BYTE  4
#define BYTE_SHIFT        2
#define SECTOR_SIZE_BIT  32
#define BIT_SHIFT         5
#define MASK             31

#ifdef TEST

/**************************************************************************/
/* Ein kleiner Test                                                       */
/**************************************************************************/

static void test(void)
{
    unsigned u;
    Set *set1,*set2;

    set1=SetCreate();
    for (u=1; u<=1030; ++u) {
        SetInsertElement(set1,u);
    }
    set2=SetCreate();
    for (u=1; u<=1130; ++u) {
        SetInsertElement(set2,u);
    }

    for (u=1; u<=1030; ++u) {
        if (u!=SetFirstOfInter(set1,set2))
            Fatal("Shit !");
        SetRemoveElement(set1,u);
    }

    SetDelete(set1);
    SetDelete(set2);
}

#endif

/***************************/
/* Ist dies schneller, ... */
/***************************/

#ifdef VERSION1

unsigned maske[SECTOR_SIZE_BIT];

#define TRANSL(element) (maske[element&MASK])

/**************************************************************************/
/* Test, ob Definitionen ok und Mengendarstellung wie geplant klappt.     */
/**************************************************************************/

void InitMaske(void)
{
    register unsigned mask1,mask2,*ptr;

    mask2=((mask1=1)<<MASK);
    if ((mask1>>1) || (mask2<<1) || (sizeof(unsigned)!=SECTOR_SIZE_BYTE))
        Fatal("Mengendarstellung klappt auf dieser Maschine nicht !");

    mask1=1;
    ptr=maske;
    while (mask1) {
        *ptr++=mask1;
        mask1<<1;
    }

#ifdef TEST
    test();
#endif

}

#endif

#ifdef VERSION2

/*********************/
/* ... oder dieses ? */
/*********************/

#define TRANSL(element) (1<<(element&MASK))

/**************************************************************************/
/* Test, ob Definitionen ok und Mengendarstellung wie geplant klappt.     */
/**************************************************************************/

void InitMaske(void)
{
    register unsigned mask1,mask2;

    mask2=((mask1=1)<<MASK);
    if ((mask1>>1) || (mask2<<1) || (sizeof(unsigned)!=SECTOR_SIZE_BYTE))
        Fatal("Mengendarstellung klappt auf dieser Maschine nicht !");

#ifdef TEST
    test();
#endif

}

#endif

/**************************************************************************/
/* Liefert Duplikat der Menge 'set'.                                      */
/**************************************************************************/

Set *SetDuplicate(Set *set)
{
    register unsigned i,*ptr1,*ptr2;
    Set *dupl;

    assert(set);
    (dupl=(Set *)MyMalloc(sizeof(Set)))->max_sector=(i=set->max_sector);
    ptr1=dupl->sector=(unsigned *)MyMalloc((++i)<<BYTE_SHIFT);
    ptr2=set->sector;
    do {
        *ptr1++=*ptr2++;
    } while (--i);
    return dupl;
}

/**************************************************************************/
/* Kopiert 'set2' nach 'set1' und liefert 'set1'.                         */
/**************************************************************************/

Set *SetCopy(Set *set1,Set *set2)
{
    register unsigned i,*ptr1,*ptr2;

    assert(set1 && set2);
    if (set1->max_sector!=(i=set2->max_sector))
        set1->sector=(unsigned *)MyRealloc(set1->sector,
                                           ((set1->max_sector=i)+1)
                                           <<BYTE_SHIFT);
    ptr1=set1->sector;
    ptr2=set2->sector;
    ++i;
    do {
        *ptr1++=*ptr2++;
    } while (--i);
    return set1;
}

/**************************************************************************/
/* Vertauscht die Mengen 'set1' und 'set2' und liefert 'set1'.            */
/**************************************************************************/

Set *SetSwap(Set *set1,Set *set2)
{
    register unsigned max_sector,*sector;

    assert(set1 && set2);
    max_sector=set1->max_sector;
    sector=set1->sector;
    set1->max_sector=set2->max_sector;
    set1->sector=set2->sector;
    set2->max_sector=max_sector;
    set2->sector=sector;
    return set1;
}

/**************************************************************************/
/* Liefert neue, leere Menge.                                             */
/**************************************************************************/

Set *SetCreate(void)
{
    register Set *set;

    (set=(Set *)MyMalloc(sizeof(Set)))->max_sector=0;
    *(set->sector=(unsigned *)MyMalloc(SECTOR_SIZE_BYTE))=0;
    return set;
}

/**************************************************************************/
/* Entfernt die Menge 'set'.                                              */
/**************************************************************************/

void SetDelete(Set *set)
{
    assert(set);
    MyFree(set->sector);
    MyFree(set);
}

/**************************************************************************/
/* Leert die Menge 'set' und liefert 'set'.                               */
/**************************************************************************/

Set *SetClear(Set *set)
{
    register unsigned *ptr,i;

    assert(set);
    ptr=set->sector;
    i=set->max_sector;
    ++i;
    do {
        *ptr++=0;
    } while (--i);
    return set;
}

/**************************************************************************/
/* Einfuegen des Elements 'element' in die Menge 'set'.                   */
/* Liefert '0' falls schon vorhanden, sonst '1'.                          */
/**************************************************************************/

int SetInsertElement(Set *set,unsigned element)
{
    register unsigned sector_number,i,*ptr;

    assert(set && element);
    sector_number=element>>BIT_SHIFT;
    if ((i=set->max_sector)<sector_number) {
        set->sector=(unsigned *)MyRealloc(set->sector,
                                          ((set->max_sector=sector_number)+1)
                                          <<BYTE_SHIFT);
        ptr=set->sector+i;
        i=sector_number-i+1;
        while (--i)
            *++ptr=0;
        i=TRANSL(element);
    } else if (*(ptr=set->sector+sector_number)&(i=TRANSL(element)))
        return 0;
    *ptr|=i;
    return 1;
}

/**************************************************************************/
/* Loeschen des Elements 'element' aus der Menge 'set'.                   */
/* Liefert '0' falls es nicht vorhanden war, sonst '1'.                   */
/**************************************************************************/

int SetRemoveElement(Set *set,unsigned element)
{
    register unsigned sector_number,i,*ptr;

    assert(set && element);
    sector_number=element>>BIT_SHIFT;
    if (sector_number<=set->max_sector
        && (*(ptr=set->sector+sector_number)&(i=TRANSL(element)))) {
        *ptr&=~i;
        return 1;
    }
    return 0;
}

/**************************************************************************/
/* Liefert - falls in der Menge 'set' vorhanden - das Element 'element',  */
/* oder das numerisch darauf folgende.                                    */
/**************************************************************************/

unsigned SetGetElementNext(Set *set,unsigned element)
{
    register unsigned sector_number,mask,*ptr;

    assert(set && element);
    if ((sector_number=element>>BIT_SHIFT)<=set->max_sector) {
        ptr=set->sector+sector_number;
        mask=TRANSL(element);
        do {
            while (mask) {
                if (*ptr&mask)
                    return element;
                mask<<=1;
                ++element;
            }
            mask=1;
            while (++sector_number<=set->max_sector && !*++ptr)
                element+=SECTOR_SIZE_BIT;
        } while (sector_number<=set->max_sector);
    }
    return 0;
}

/**************************************************************************/
/* Liefert - falls in der Menge 'set' vorhanden - das Element 'element',  */
/* oder das numerisch davorliegende.                                      */
/**************************************************************************/

unsigned SetGetElementPrev(Set *set,unsigned element)
{
    register unsigned sector_number,mask,*ptr;

    assert(set && element);
    if ((sector_number=element>>BIT_SHIFT)>set->max_sector)
        element=((sector_number=set->max_sector)<<BIT_SHIFT)+MASK;
    ptr=set->sector+sector_number;
    mask=TRANSL(element);
    ++sector_number;
    do {
        while (mask) {
            if (*ptr&mask)
                return element;
            mask>>=1;
            --element;
        }
        mask=(unsigned)1<<MASK;
        while (--sector_number && !*--ptr)
            element-=SECTOR_SIZE_BIT;
    } while (sector_number);
    return 0;
}

/**************************************************************************/
/* Test, ob das Element 'element' in der Menge 'set' enthalten ist.       */
/**************************************************************************/

int SetIsElement(Set *set,unsigned element)
{
    register unsigned sector_number;

    assert(set && element);
    sector_number=element>>BIT_SHIFT;
    return (sector_number<=set->max_sector)
           && *(set->sector+sector_number)&TRANSL(element);
}

/**************************************************************************/
/* Test, ob die Menge 'set' leer ist.                                     */
/**************************************************************************/

int SetIsEmpty(Set *set)
{
    register unsigned i,*ptr;

    assert(set);
    ptr=set->sector;
    i=set->max_sector;
    ++i;
    do {
        if (*ptr++)
            return 0;
    } while (--i);
    return 1;
}

/**************************************************************************/
/* Liefert die Maechtigkeit der Menge 'set'.                              */
/**************************************************************************/

unsigned SetCountElements(Set *set)
{
    register unsigned sector_number,count,mask,*ptr;

    assert(set);
    count=0;
    ptr=set->sector;
    sector_number=set->max_sector;
    ++sector_number;
    do {
        mask=1;
        if (*ptr)
            while (mask) {
                if (*ptr&mask)
                    ++count;
                mask<<=1;
            }
        ++ptr;
    } while (--sector_number);
    return count;
}

/**************************************************************************/
/* Test, ob die Menge 'set1' Teilmenge der Menge 'set2' ist.              */
/**************************************************************************/

int SetIsSubset(Set *set1, Set *set2)
{
    register unsigned i,*ptr1,*ptr2;

    assert(set1 && set2);
    if ((i=set1->max_sector)>set2->max_sector) {
        ptr1=set1->sector+i+1;
        while (i!=set2->max_sector) {
            if (*--ptr1)
                return 0;
            --i;
        }
    }
    ptr1=set1->sector;
    ptr2=set2->sector;
    ++i;
    do {
        if ((*ptr1&*ptr2++)!=*ptr1)
            return 0;
        ++ptr1;
    } while (--i);
    return 1;
}

/**************************************************************************/
/* Test, ob die Menge 'set1' gleich der Menge 'set2' ist.                 */
/**************************************************************************/

int SetIsEqual(Set *set1, Set *set2)
{
    register unsigned i,*ptr1,*ptr2;

    assert(set1 && set2);
    ptr1=set1->sector;
    ptr2=set2->sector;
    if (set1->max_sector<set2->max_sector) {
        i=set1->max_sector;
        ++i;
        do {
            if (*ptr1++!=*ptr2++)
                return 0;
        } while (--i);
        i=set2->max_sector-set1->max_sector;
        do {
            if (*ptr2++)
                return 0;
        } while (--i);
        return 1;
    } else {
        i=set2->max_sector;
        ++i;
        do {
            if (*ptr1++!=*ptr2++)
                return 0;
        } while (--i);
        i=set1->max_sector-set2->max_sector;
        ++i;
        while (--i) {
            if (*ptr1++)
                return 0;
        }
        return 1;
    }
}

/**************************************************************************/
/* Test, ob die Mengen 'set1' und 'set2' disjunkt sind.                   */
/**************************************************************************/

int SetDisjunct(Set *set1, Set *set2)
{
    register unsigned i,*ptr1,*ptr2;

    assert(set1 && set2);
    i=(set1->max_sector<set2->max_sector)?
      set1->max_sector:set2->max_sector;
    ++i;
    ptr1=set1->sector;
    ptr2=set2->sector;
    do {
        if (*ptr1&*ptr2)
            return 0;
        ++ptr1;
        ++ptr2;
    } while (--i);
    return 1;
}

/**************************************************************************/
/* Liefert 'set1' := 'set1' vereinigt 'set2'                              */
/**************************************************************************/

Set *SetUnion(Set *set1, Set *set2)
{
    register unsigned i,*ptr1,*ptr2;

    assert(set1 && set2);
    if ((i=set1->max_sector)<set2->max_sector) {
        set1->sector=(unsigned *)MyRealloc(set1->sector,
                                           ((set1->max_sector
                                             =set2->max_sector)+1)
                                           <<BYTE_SHIFT);
        ptr1=set1->sector+i;
        i=set2->max_sector-i+1;
        while (--i)
            *++ptr1=0;
    }
    ptr1=set1->sector;
    ptr2=set2->sector;
    i=set2->max_sector;
    ++i;
    do {
        *ptr1++|=*ptr2++;
    } while (--i);
    return set1;
}

/**************************************************************************/
/* Liefert "erstes" Element aus 'set1' geschnitten 'set2'                 */
/**************************************************************************/

unsigned SetFirstOfInter(Set *set1, Set *set2)
{
    register unsigned i,element,mask,*ptr1,*ptr2;

    assert(set1 && set2);
    i=(set1->max_sector<set2->max_sector)?
      set1->max_sector:set2->max_sector;
    ++i;
    ptr1=set1->sector;
    ptr2=set2->sector;
    element=0;
    do {
        if (*ptr1&*ptr2) {
            i=*ptr1&*ptr2;
            mask=1;
            while (mask) {
                if (i&mask)
                    return element;
                mask<<=1;
                ++element;
            }
            Fatal("Should not happen !");
        }
        ++ptr1;
        ++ptr2;
        element+=SECTOR_SIZE_BIT;
    } while (--i);
    return 0;
}

/**************************************************************************/
/* Liefert 'set1' := 'set1' geschnitten 'set2'                            */
/**************************************************************************/

Set *SetIntersection(Set *set1, Set *set2)
{
    register unsigned i,*ptr1,*ptr2;

    assert(set1 && set2);
    if (set1->max_sector>set2->max_sector)
        set1->sector=(unsigned *)MyRealloc(set1->sector,
                                           ((set1->max_sector
                                             =set2->max_sector)+1)
                                           <<BYTE_SHIFT);
    ptr1=set1->sector;
    ptr2=set2->sector;
    i=set1->max_sector;
    ++i;
    do {
        *ptr1++&=*ptr2++;
    } while (--i);
    return set1;
}

/**************************************************************************/
/* Liefert neue Menge 'set' := 'set1' geschnitten 'set2'                  */
/**************************************************************************/

Set *SetIntersectionCreate(Set *set1, Set *set2)
{
    register unsigned i,*ptr1,*ptr2,*ptr3;
             Set *set3;

    assert(set1 && set2);
    (set3=(Set *)MyMalloc(sizeof(Set)))
    ->max_sector=i=(set1->max_sector<set2->max_sector)?
                   set1->max_sector:set2->max_sector;
    ptr3=set3->sector=(unsigned *)MyMalloc((++i)<<BYTE_SHIFT);
    ptr2=set2->sector;
    ptr1=set1->sector;
    do {
        *ptr3++=*ptr1++&*ptr2++;
    } while (--i);
    return set3;
}

/**************************************************************************/
/* Liefert 'set1' := 'set1' ohne 'set2'                                   */
/**************************************************************************/

Set *SetDifference(Set *set1, Set *set2)
{
    register unsigned i,*ptr1,*ptr2;

    assert(set1 && set2);
    ptr1=set1->sector;
    ptr2=set2->sector;
    i=(set1->max_sector<set2->max_sector)?set1->max_sector:set2->max_sector;
    ++i;
    do {
        *ptr1++&=~*ptr2++;
    } while (--i);
    return set1;
}

/**************************************************************************/
/* Liefert 'set1' := ('set1' ohne 'set2') vereinigt ('set2' ohne 'set1')  */
/**************************************************************************/

Set *SetEOR(Set *set1, Set *set2)
{
    register unsigned i,*ptr1,*ptr2;

    assert(set1 && set2);
    if ((i=set1->max_sector)<set2->max_sector) {
        set1->sector=(unsigned *)MyRealloc(set1->sector,
                                           ((set1->max_sector
                                             =set2->max_sector)+1)
                                           <<BYTE_SHIFT);
        ptr1=set1->sector+i;
        i=set2->max_sector-i+1;
        while (--i)
            *++ptr1=0;
    }
    ptr1=set1->sector;
    ptr2=set2->sector;
    i=set2->max_sector;
    ++i;
    do {
        *ptr1++^=*ptr2++;
    } while (--i);
    return set1;
}

/**************************************************************************/
/* Ausgabe der Menge 'set'.                                               */
/**************************************************************************/

void SetDisplay(FILE *out,Set *set)
{
    register unsigned sector_number,element,mask,*ptr;

    assert(set);
    ptr=set->sector;
    sector_number=set->max_sector;
    ++sector_number;
    element=0;
    do {
        mask=1;
        if (*ptr)
            while (mask) {
                if (*ptr&mask)
                    fprintf(out,"%u;",element);
                mask<<=1;
                ++element;
            }
        else
            element+=SECTOR_SIZE_BIT;
        ++ptr;
    } while (--sector_number);
}

/**************************************************************************/
/* Input/Output einer Menge.                                              */
/**************************************************************************/
 
Set *SetRead(FILE *in)
{
    register Set *set;

    set=(Set *)MyMalloc(sizeof(Set));
    fread((void *)&(set->max_sector),1,sizeof(set->max_sector),in);
    set->sector=(unsigned *)MyMalloc(sizeof(*(set->sector))
                                     *(set->max_sector+1));
    fread((void *)(set->sector),set->max_sector+1,sizeof(*(set->sector)),in);
    return set;
}

void SetSave(FILE *out, Set *set)
{
    fwrite((void *)&(set->max_sector),1,sizeof(set->max_sector),out);
    fwrite((void *)(set->sector),set->max_sector+1,sizeof(*(set->sector)),out);
}

/*********************/
/* END OF FILE set.c */
/*********************/
