/*
 * Main routines:
 *
 * Copyright Keijo Heljanko <Keijo.Heljanko@hut.fi>
 *
 * Smodels:
 *
 * Copyright Patrik Simons <Patrik.Simons@hut.fi>
 *
 */


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "combi.h"
#include "main.h"
#include "dassert.h"
#include "ptnet.h"
#include "unroll.h"
#include "scanner.h"
#include "tree.h"
#include "tree2smo.h"

// Include the smodels stuff.

#include <iostream.h>
#include "stable.h"
#include "api.h"
#include "timer.h"
#if defined _BSD_SOURCE || defined _SVID_SOURCE || defined __SVR4
#include <sys/time.h>
#include <sys/resource.h>
#endif


// The following is not fully tested nor documented, use at your own risk!
// (It might work for reachability though...)
//#define USE_PROCESSES_FOR_REACHABILITY

static void usage(char *myname)
{
    fprintf(stderr,"============================================================================\n");
    fprintf(stderr,"Bounded LTL model checker for 1-safe nets (generates smodels programs)\n");
    fprintf(stderr,"Usage: %s [OPTION]... file1 [file2]\n",myname);
    fprintf(stderr,"\n");
    fprintf(stderr,"Options:\n");
    fprintf(stderr,"        -b=num   : The number of time steps\n");
    fprintf(stderr,"        -f=fname : Model check the LTL formula (in PEP format) in the file fname\n");
    fprintf(stderr,"        -D       : Deadlock check the net\n");
    fprintf(stderr,"        -s       : Use step semantics (default)\n");
    fprintf(stderr,"        -i       : Use interleaved semantics\n");
#ifdef USE_PROCESSES_FOR_REACHABILITY
    fprintf(stderr,"        -p       : Use process semantics\n");
#endif
    fprintf(stderr,"        -O0      : Do not optimize the translation\n");
    fprintf(stderr,"        -O1      : Do only simple optimizations\n");
    fprintf(stderr,"        -O2      : Do also duplicate rule removal (default)\n");
#if 0
    fprintf(stderr,"        -n       : Use smodels with no lookahead (fast for easy examples)\n");
    fprintf(stderr,"        -v       : Give verbose statistics\n");
#endif
    fprintf(stderr,"\n");
    fprintf(stderr,"        file1 : input file containing unfolding\n");
    fprintf(stderr,"                extension '.ll_net': PEP low level net\n");
    fprintf(stderr,"        file2 : output result file\n");
    fprintf(stderr,"\n");
    fprintf(stderr,"Copyright (c) 1998-2001 Keijo Heljanko <Keijo.Heljanko@hut.fi> (unfolder)\n");
    fprintf(stderr,"Copyright (c) 1998-2000 Patrik Simons <Patrik.Simons@hut.fi> (smodels)\n");
    fprintf(stderr,"\n");
    fprintf(stderr,"Version 0.9, smodels version 2.26, (%s)\n", __DATE__);
    fprintf(stderr,"============================================================================\n");

}

// Some global variables, used in scanner.cc only!

FILE * yyin_for_parser;
PTNet * net_for_parser;

int main (int argc, char *argv[])
{

  FILE * in;
  FILE * in_formula;
  char * input_file;
  char * formula_file;
  char * output_file;
  TreeNode * formula;
  BOOL verbose;
  BOOL error;
  BOOL debugging;
  BOOL lookahead;
  BOOL step;
  BOOL interleaved;
  BOOL process;
  BOOL do_deadlock;
  BOOL do_formula;
  BOOL arg_deadlock;
  BOOL arg_formula;
  unsigned long bound;
  unsigned long optimize_level;
  unsigned long semantics; // 0 = interleaved, 1 = step, 2 = process

  Timer timer1;
#if 0
#if defined _BSD_SOURCE || defined _SVID_SOURCE || defined __SVR4
  struct rusage rusage1;
#endif
#endif
  PTNet *net;

  timer1.start();
  

  // Give various defaults for parameters

  lookahead = TRUE;
  debugging = FALSE;
  verbose = FALSE;
  step = FALSE;
  interleaved = FALSE;
  process = FALSE;
  do_deadlock = FALSE;
  do_formula = FALSE;
  
  arg_deadlock = FALSE;
  arg_formula = FALSE;

  bound = 0;
  optimize_level = 2;
  semantics = 1;
  debugging = TRUE;

  error = FALSE;

  in = (FILE *)NULL;
  in_formula = (FILE *) NULL;
  input_file = (char *)NULL;
  formula_file = (char *) NULL;
  output_file = (char *)NULL;

  net = (PTNet *)NULL;

  for (int c = 1; c < argc && error == FALSE; c++) {
    if (argv[c][0] == '-') {
      if (strncmp (&argv[c][1], "b=", 2) == 0) {
	bound = atol(&argv[c][3]);
      }
      else if (strncmp (&argv[c][1], "f=", 2) == 0) {
        do_formula = TRUE;
        do_deadlock = FALSE;
        arg_formula = TRUE;
        formula_file = &argv[c][3];
      }
      else if (strcmp (&argv[c][1], "D") == 0) {
        do_deadlock = TRUE;
        do_formula = FALSE;
        arg_deadlock = TRUE;
      }
#if 0
      else if (strcmp (&argv[c][1], "d") == 0) {
	debugging = TRUE;
      }
#endif
      else if (strcmp (&argv[c][1], "O0") == 0) {
        optimize_level = 0;
      }
      else if (strcmp (&argv[c][1], "O1") == 0) {
        optimize_level = 1;
      }
      else if (strcmp (&argv[c][1], "O2") == 0) {
        optimize_level = 2;
      }
#if 0
      else if (strcmp (&argv[c][1], "n") == 0) {
	lookahead = FALSE;
      }
#endif
      else if (strcmp (&argv[c][1], "v") == 0) {
	verbose = TRUE;
      }
      else if (strcmp (&argv[c][1], "s") == 0) {
	step = TRUE;
      }
      else if (strcmp (&argv[c][1], "i") == 0) {
	interleaved = TRUE;
#ifdef USE_PROCESSES_FOR_REACHABILITY
      }
      else if (strcmp (&argv[c][1], "p") == 0) {
	process = TRUE;
#endif
      } else {
	error = TRUE;
      }
    } else {
      if(input_file == NULL) {
	input_file = argv[c];
      } else if(output_file == NULL) {
	output_file = argv[c];
      } else {
	error = TRUE;
	break;
      }
    }
  }

  if(input_file == NULL) {
    error = TRUE;
  }

  if(((interleaved != FALSE) && (step != FALSE)) ||
     ((interleaved != FALSE) && (process != FALSE)) ||
     ((step != FALSE) && (process != FALSE))) {
    fprintf(stderr, "%s: Use only one of the semantics\n", argv[0]);
    exit(1);
  }

  if(interleaved != FALSE) {
    semantics = 0; // Use interleaved semantics
  } else if (step != FALSE) {
    semantics = 1; // Use step semantics
  } else if (process != FALSE) {
    semantics = 2; // Use process semantics
  } else {
    semantics = 1; // Use step semantics by default
  }

  // Only do deadlock or formula, but not both
  
  if(arg_formula != FALSE && arg_deadlock != FALSE) {
    error = TRUE;
  }
  if(do_formula != FALSE && formula_file == NULL) {
    error = TRUE;
  }


  if(error != FALSE) {
    usage(argv[0]);
    exit(1);
  }
    
  if((in = fopen(input_file, "rb")) == NULL) {

    fprintf(stderr, "%s: No such PEP low level net file: %s\n", argv[0], input_file);
    exit(1);
   
  }

  if(output_file != NULL) {

    if(freopen(output_file, "wb", stdout) == NULL) {

      fprintf(stderr, "%s: Could not open output file: %s \n", argv[0], output_file);
      exit(1);
   
    }
    
  }

  if(formula_file != NULL) {

    if((in_formula = fopen(formula_file, "rb")) == NULL) {

      fprintf(stderr, "%s: No such formula file: %s\n", argv[0], formula_file);
      exit(1);
   
    }

  }



  net = new PTNet();

  if(net->ReadPEP(in) == FALSE) {

    fprintf(stderr, "%s: Error while parsing PEP low level net file: %s\n", argv[0], input_file);

    delete(net);
    exit(1);

  }


  formula = (TreeNode *)NULL;

  if(do_formula != FALSE) {

    // Pass two arguments through global variables for convenience.

    yyin_for_parser = in_formula;
    net_for_parser = net;

    formula = ParseFormula();

    fclose(in_formula);

    if(formula != NULL) {

     // Do formula processing
 
#define DEBUG_PRINT_FORMULA 1

#if 0
#if DEBUG_PRINT_FORMULA
     fprintf(stdout, "%% ");
     TreeDisplay(stdout, formula);
     fprintf(stdout, "\n");
#endif
#endif

     formula = PushNegationsIn(formula, TRUE); // Negate the formula while pushing negations in!

#if DEBUG_PRINT_FORMULA
     fprintf(stdout, "%% Formula after negation: ");
     TreeDisplay(stdout, formula);
     fprintf(stdout, "\n");
#endif



#undef DEBUG_PRINT_FORMULA


    } else {

      exit(1); // Parse error in formula

    }

  }



#if 0
  fprintf(stderr, "Bound: %ld, step %d\n", bound, step);
#endif

  unroll(net, bound, semantics, do_deadlock, do_formula, formula,
	 debugging, optimize_level);

  delete(net);

  fflush(stdout);
    
  timer1.stop();

  if(debugging == FALSE) {

    // Not implemented yet!

    ASSERT(FALSE);

#if 0
#if defined _BSD_SOURCE || defined _SVID_SOURCE || defined __SVR4
    getrusage (RUSAGE_SELF, &rusage1);
    fprintf(stdout,
	    "\nTime needed:      %ld.%.3ld\n",
	    rusage1.ru_utime.tv_sec +
	    rusage1.ru_stime.tv_sec +
	    ((rusage1.ru_utime.tv_usec +
	      rusage1.ru_stime.tv_usec)
	     / 1000000),
	    ((((rusage1.ru_utime.tv_usec +
		rusage1.ru_stime.tv_usec)
	       / 1000) < 1000) ?
	     ((rusage1.ru_utime.tv_usec +
	       rusage1.ru_stime.tv_usec)
	      / 1000) :
	     (((rusage1.ru_utime.tv_usec +
		rusage1.ru_stime.tv_usec)
	       / 1000) - 1000)));
#else
    fprintf(stdout, "\nTime needed:      %s s\n", timer1.print());
#endif
    
    fflush(stdout);
#endif

  }
    
  if(output_file != NULL) {
    fclose(stdout);
  }
  
  fclose(in);
  
  return 0;

}


