#ifndef TILE_H			// -*- c++ -*-
#define TILE_H
#pragma interface

// Copyright 2004,2006 Jouni K. Seppnen
// Distributed under the Boost Software License, Version 1.0.
// See accompanying file LICENSE.


#include <cassert>
#include <cstddef>
#include <vector>
#include <iostream>

namespace Tilings {
  // A geometric tile. (Combinatorial tiles do not suit this model
  // very well, so they are not anticipated as subclasses.)
  class Tile {
  protected:
    // The top, bottom, left and right of the data matrix
    // (static bounds will do for now, but not if we want to
    // recurse into tiles).
    static long int bound[4];
    static bool bounds_set;

  public:
    // The words TOP, BOTTOM, LEFT and RIGHT refer to how a
    // matrix is printed on screen (as in Matlab),
    // but the indexing is still done so that [0][0]
    // is the upper-left corner (as in Matlab, mutatis
    // mutandis (i.e., (1,1) instead of [0][0])).
    typedef enum { TOP, BOTTOM, LEFT, RIGHT } edge_idx;
    static const int DIRECTION[4];
    static edge_idx opposite(edge_idx dir) 
    { return (edge_idx)((int)dir^1); /* swap least significant bit */ }
    long int edge[4];

    static void set_bounds(long int t, long int b, 
			   long int l, long int r);
    void check_bounds() const
    {
#ifndef NDEBUG
      for (int i=0; i<4; i++)
	assert(DIRECTION[i] * edge[i] <= DIRECTION[i] * bound[i]);
#endif
    }

    bool operator==(const Tile& other) const
    { return edge[TOP] == other.edge[TOP]
	&& edge[BOTTOM] == other.edge[BOTTOM]
	&& edge[LEFT] == other.edge[LEFT]
	&& edge[RIGHT] == other.edge[RIGHT]; }

    bool operator!=(const Tile& other) const
    { return !(*this == other); }

    Tile(long int t, long int b, long int l, long int r)
    { edge[TOP] = t; edge[BOTTOM] = b; edge[LEFT] = l; edge[RIGHT] = r; }

    long int size() const 
    { 
      return (edge[BOTTOM]-edge[TOP]+1)*(edge[RIGHT]-edge[LEFT]+1); 
    }

    // Does this tile share any cells with the other one?
    bool overlaps(const Tile& other) const
    {
      return (edge[TOP] <= other.edge[BOTTOM] 
	      && edge[BOTTOM] >= other.edge[TOP]
	      && edge[LEFT] <= other.edge[RIGHT] 
	      && edge[RIGHT] >= other.edge[LEFT]);
    }

    // Does this tile contain every cell of the other one
    // without being the same?
    bool subsumes(const Tile& other) const
    {
      return (edge[TOP] <= other.edge[TOP] 
	      && edge[BOTTOM] >= other.edge[BOTTOM]
	      && edge[LEFT] <= other.edge[LEFT] 
	      && edge[RIGHT] >= other.edge[RIGHT]
	      && *this != other);
    }

    // Partially_overlaps means that this tile overlaps the other but
    // does not subsume it, and nor does the other tile subsume this
    // one.
    bool partially_overlaps(const Tile& other) const
    {
      return overlaps(other) && !subsumes(other) && !other.subsumes(*this);
    }


    // Enlarge this tile until it subsumes the other
    void expand_to_subsume(const Tile& other);

    // Shrink this tile until it no longer overlaps the other
    // by moving edge e
    void reduce_to_avoid(const Tile& other, edge_idx e);
    
    // Return the neighbors of this tile in tile-space.
    // Maxdelta is the maximum change in each coordinate:
    // if 1, there are at most 8 neighbors,
    // if larger, there might be more (the number is logarithmic
    //            in maxdelta),
    // if 0, only the data matrix limits the change.
    // The vector avoid contains tiles that may not be
    // overlapped partially (but may be subsumed).
    std::vector<Tile> neighbors(unsigned int maxdelta,
				const std::vector<Tile*>& avoid) const;

  };
  
  inline std::ostream& operator<<(std::ostream& s, const Tile& t)
  {
    return s << "[" << t.edge[Tile::TOP] 
	     << "-" << t.edge[Tile::BOTTOM]
	     << " X " << t.edge[Tile::LEFT] 
	     << "-" << t.edge[Tile::RIGHT] << "]";
  }

  //  std::ostream& 
  //  operator<<(std::ostream& s, const std::vector<Tile> vec);

}

#endif
