// Copyright 2004,2006 Jouni K. Seppnen          -*- coding: iso-8859-1 -*-
// Distributed under the Boost Software License, Version 1.0.
// See accompanying file LICENSE.

#pragma implementation "tile.h"
#include "tile.h"

#include <vector>
#include <iostream>

namespace Tilings {
  long int Tile::bound[4] = { 0, 0, 0, 0 };
  bool Tile::bounds_set = false;
  const int Tile::DIRECTION[4] = { -1, 1, -1, 1 };

  void 
  Tile::set_bounds(long int t, long int b, long int l, long int r)
  {
    assert(!bounds_set);
    bounds_set = true;

    bound[TOP] = t;
    bound[BOTTOM] = b;
    bound[LEFT] = l;
    bound[RIGHT] = r;
  }

  void 
  Tile::expand_to_subsume(const Tile& other)
  {
    check_bounds();
    other.check_bounds();
    for (int i = 0; i < 4; i++)
      if (DIRECTION[i] * edge[i] < DIRECTION[i] * other.edge[i])
	edge[i] = other.edge[i];
  }

  void 
  Tile::reduce_to_avoid(const Tile& other, edge_idx e)
  // It is not obvious which edge to move to avoid overlap
  // so we take it as input.
  // Perhaps the asserts should not cause an abort but
  // somehow make the tile get infinitely bad score?
  {
    check_bounds();
    other.check_bounds();

    switch(e) {
    case TOP:
      assert(other.edge[BOTTOM] < bound[BOTTOM]);
      edge[TOP] = other.edge[BOTTOM]+1;
      break;
    case BOTTOM:
      assert(other.edge[TOP] > bound[TOP]);
      edge[BOTTOM] = other.edge[TOP]-1;
      break;
    case LEFT:
      assert(other.edge[RIGHT] < bound[RIGHT]);
      edge[LEFT] = other.edge[RIGHT]+1;
      break;
    case RIGHT:
      assert(other.edge[LEFT] > bound[LEFT]);
      edge[RIGHT] = other.edge[LEFT]-1;
      break;
    default:
      assert(false);
    }
    // It is possible that we end up with a tile of area 0;
    // let the caller handle that
  }

  std::vector<Tile> 
  Tile::neighbors(unsigned int maxdelta, const std::vector<Tile*>& avoid) const
  {
    using namespace std;
    vector<Tile> result;

    for (int dir = -1; dir <= 1; dir += 2)
      for (edge_idx i = (edge_idx)0; 
	   i < (edge_idx)4; 
	   i = (edge_idx)((int)i+1)) 
	for (int delta = dir;
	     maxdelta ? (unsigned int)(dir*delta) < maxdelta : true;
	     delta *= 2)
	  {
	    Tile candidate(*this);
	    bool ok = false;
	    long int new_edge = candidate.edge[i] + delta*DIRECTION[i];
	    if(DIRECTION[i] * new_edge <= 
	       DIRECTION[i] * bound[i]
	       && DIRECTION[i] * new_edge > 
	       DIRECTION[i] * candidate.edge[opposite(i)])
	      /* OK */ ;
	    else
	      break; // no use going further in this direction

	    ok = true;
	    candidate.edge[i] = new_edge;
	    vector<Tile*>::const_iterator obstacle;
	    for (obstacle = avoid.begin(); 
		 obstacle != avoid.end(); 
		 obstacle++)
	      if (candidate.partially_overlaps(**obstacle)
		  || candidate == **obstacle) 
		{
		  ok = false;
		  break;
		}
	    if (ok)
	      result.push_back(candidate);
	  }
    return result;
  }


#if 0
  std::ostream& 
  operator<<(std::ostream& s, const std::vector<Tile> vec)
  {
    for (unsigned int i = 0; i < vec.size(); i++) {
      s << vec[i];
      if (i+1 < vec.size())
	s << ", ";
    }
    return s;
  }
#endif

}
