#ifndef SEARCH_H		// -*- c++ -*-
#define SEARCH_H
#pragma interface

// Copyright 2004,2006 Jouni K. Seppnen
// Distributed under the Boost Software License, Version 1.0.
// See accompanying file LICENSE.

#include "matrices.h"
#include "tile.h"
#include "model.h"

namespace Tilings {
  using namespace Matrices;
  class Search {
  protected:
    Matrix<double> data;
    Model model;
    virtual Tile random_tile() const;
  protected:
    void filter_neighbors(std::vector<Tile>&, const std::vector<Tile*>&) const;
    bool bad (const Tile&, const std::vector<Tile*>&) const;
    virtual bool bad(const Tile&, const Tile&) const = 0;
  public:
    Search(const Matrix<double>& data_) 
      : model(Model(Matrix<double>()))
    { 
      data = data_;
      //data.spectral_bisort();
      model = Model(data);
      Tile::set_bounds(0,data.dim1()-1,0,data.dim2()-1); 
    }
    virtual ~Search() { }

    virtual const Model& run() = 0;
    const Model& mod() { return model; }
  };

  class GreedySearch: public Search {
    unsigned int restarts;
    std::pair<Tile,double> restart();
    bool bad (const Tile&, const Tile&) const { return false; }
  public:
    GreedySearch(Matrix<double>& data_, unsigned int restarts_)
      : Search(data_), restarts(restarts_) {  }
    const Model& run();
  };

  class BottomUpGreedySearch: public GreedySearch {
    bool bad(const Tile& candidate, const Tile& existing) const
    { return existing.subsumes(candidate); }
  public:
    BottomUpGreedySearch(Matrix<double>& data_, unsigned int restarts_)
      : GreedySearch(data_, restarts_) { }
  };

  class TopDownGreedySearch: public GreedySearch {
    bool bad(const Tile& candidate, const Tile& existing) const
    { return candidate.subsumes(existing); }
  public:
    TopDownGreedySearch(Matrix<double>& data_, unsigned int restarts_)
      : GreedySearch(data_, restarts_) { }
  };

  class LevelGreedySearch: public GreedySearch {
    bool bad(const Tile& candidate, const Tile& existing) const
    { return candidate.subsumes(existing) || existing.subsumes(candidate); }
  public:
    LevelGreedySearch(Matrix<double>& data_, unsigned int restarts_)
      : GreedySearch(data_, restarts_) { }
  };
    
}

#endif
