function [e,perm] = reorder(e,similarity)
% [E,PERM] = REORDER(E,SIMILARITY)
%
% Spectral sort of the rows of a matrix E. The SIMILARITY argument may
% be either 'dot' or 'hamming', for dot-product or Hamming similarity. 
% The function returns both the reordered matrix and the row
% permutation.
%
% Copyright 2004,2006 Jouni K. Seppnen
% Distributed under the Boost Software License, Version 1.0.
% See accompanying file LICENSE.

% XXX magic constant for deciding when a split is too uneven:
cutoff = 0.5; 
if ~exist('similarity','var'),
  similarity = 'dot';
end
similarity = str2func(similarity);  
eigsopts = struct('issym',1,'disp',2);
perm = 1:size(e,1);

% The following loop emulates a recursive program. A recursive
% implementation needed to have Matlab's RecursionLimit parameter
% increased, and caused segmentation faults even then, presumably
% due to exhausting the stack space.

% A stack frame contains the local variables (from, to). No program
% counter is needed because all calls are tail-optimized. (This may
% change if we make the algorithm more clever.)
stack = [1 size(e,1)];
while length(stack) > 0,
  frame = stack(end,:); from = frame(1); to = frame(2);
  if to - from <= 1,
    % Too small: just return
    stack = stack(1:end-1,:);
    continue;
  end
  
  % Compute Laplacian
  s = feval(similarity, (e(from:to,:)));
  lap = diag(sum(s')) - s;
  clear s;
  % Compute Fiedler vector
  if issparse(lap),
    [vectors,values] = eigs(lap,2,'SA',eigsopts);
  else,
    [vectors,values] = eig(lap);
  end
  fiedler = vectors(:,2);
  clear lap vectors values;
  % Split data in two parts
  [group,score] = bestsplit(fiedler);

  if score < cutoff,
    % The split didn't reduce variance much: base case
    [junk,iii] = sort(fiedler);
    ee = e(from:to, :);
    pp = perm(from:to);
    e(from:to, :) = ee(iii,:);
    perm(from:to) = pp(iii);
    % Return from call
    stack = stack(1:end-1,:);
  else,
    % The split was uneven: recurse
    idx1 = from + find(group) - 1;
    idx2 = from + find(~group) - 1;
    e1 = e(idx1,:); e2 = e(idx2,:);
    p1 = perm(idx1); p2 = perm(idx2);
    siz = sum(group);
    e(from:from+siz-1,:) = e1;
    e(from+siz:to,:) = e2;
    perm(from:from+siz-1) = p1;
    perm(from+siz:to) = p2;
    % Set up for two recursive calls; we can optimize _both_ as tail
    % calls and delete our own stack frame!
    % Second call: recurse(from+siz,to) (replace our stack frame)
    stack(end,:) = [from+siz, to];
    % First call: recurse(from,from+siz-1) (new stack frame)
    stack(end+1,:) = [from, from+siz-1];
  end
end

function s = dot(matrix)
s = matrix*matrix';

function s = hamming(matrix)
s = full(matrix*matrix' + (1-matrix)*(1-matrix'));

function [group,value] = bestsplit(vector)

[vector,idx] = sort(vector);
bestj = 0;
value = -Inf;
wholedev = var(vector);
for j=2:length(vector),
  partdev = var(vector(1:j-1)) + var(vector(j:end));
  val = (wholedev-partdev)/wholedev;
  if val > value,
    bestj = j;
    value = val;
  end
end

group = logical(zeros(size(vector)));
group(idx(bestj:end)) = 1;
