#ifndef MODEL_H			// -*- c++ -*-
#define MODEL_H
#pragma interface

// Copyright 2004,2006 Jouni K. Seppnen
// Distributed under the Boost Software License, Version 1.0.
// See accompanying file LICENSE.

#include <iostream>
#include <vector>

#include "matrices.h"
#include "tile.h"

namespace Tilings {

  class Model {
    friend std::ostream& operator<<(std::ostream&, const Model&);
  public:
    class Node: public Tile {
      friend class Model;
      friend std::ostream& operator<< (std::ostream&, const Node&);
      friend std::ostream& operator<< (std::ostream& s, const Model& m);

      static long int next_id;
      long int id;
      long int ones, my_size, my_ones;
      std::vector<Node*> children;
      Node(Tile tile, long int ones_);
      Node(const Node& other);
      Node& operator=(const Node& other);
      ~Node();
      void into_tiles(std::vector<Tile*>&);
      void print(std::ostream& s, long int parent) const;
    };

  protected:
    // data holds the input data (we DON'T spectral-sort it here)
    // sums[i][j] is the sum of data[p][q] for all p>=i, q>=j
    Matrices::Matrix<double> data, sums;
    Node* root;
    // Pointers to all tiles
    std::vector<Tile*> tiles;

    long int ones_within(const Tile& t) const
    {
      // We want the sum of "1" cells, so we do an inclusion-exclusion:
      // 1111122222   
      // 1111122222
      // 3333344444
      // 3333344444
      return (long int)sums[t.edge[t.TOP]][(size_t) t.edge[t.LEFT]]       // 1234
	- (long int)sums[t.edge[t.BOTTOM]+1][t.edge[t.LEFT]]     // 34
	- (long int)sums[t.edge[t.TOP]][t.edge[t.RIGHT]+1]       // 24
	+ (long int)sums[t.edge[t.BOTTOM]+1][t.edge[t.RIGHT]+1]; // 4
    }
    bool place_tile(const Tile& tile, Node*& parent, 
		    std::vector<Node*>& children,
		    Node* const& under) const;

  public:
    Model(const Matrices::Matrix<double>& input);
    Model(const Model& other);
    Model& operator=(const Model& other);
    virtual ~Model();
    virtual double log_likelihood(const Node* node) const;
    virtual double log_likelihood() const;
    virtual void add_tile(const Tile& t);
    virtual double llh_increase(const Tile& t) const;
    long int dim1() const { return data.dim1(); }
    long int dim2() const { return data.dim2(); }
    const std::vector<Tile*> all_tiles() const { return tiles; }
    void gnuplot(std::ostream* output, const Model::Node& node) const;
    void gnuplot(std::ostream* output, const char* name) const;

  };

  inline double negent(double x)
  {
    if (x <= 0 || x >= 1)
      return 0;
    return x*log2(x) + (1-x)*log2(1-x);
  }

}


#endif
