#ifndef MATRICES_H		// -*- C++ -*-
#define MATRICES_H
#pragma interface

// Copyright 2004,2006 Jouni K. Seppnen
// Distributed under the Boost Software License, Version 1.0.
// See accompanying file LICENSE.


#include <assert.h>
#include <vector>
#include <cstddef>
#include <iostream>

extern "C" {
#include <cblas.h>
}

namespace Matrices
{
  template<class T>
  class Matrix;

  // Reference to a matrix row
  template<class T>
  class Row {
  private:
    const Matrix<T>& mat;
    size_t idx1;
  public:
    Row(const Matrix<T>& mat_, size_t idx_) : mat(mat_), idx1(idx_) { }
    //    { std::cerr << "Row(something," << idx1 << ")" << std::endl; }
    Row(const Row<T>& other) : mat(other.mat), idx1(other.idx1) { }
    //    { std::cerr << "Row(Row)" << std::endl; }
    void foobar() const;
    inline T& operator[](const size_t& idx2) const;
  };

  template<class T>
  class Matrix {
    friend class Row<T>;
    friend std::vector<double> newFiedler(Matrix<double>& A);
  private:
    size_t d1, d2;
    T* data;
    Matrix(size_t d1_, size_t d2_, T* data_): 
      d1(d1_), d2(d2_), data(data_) { }
    inline void multiply_into(const Matrix<T>& other, T* result) const;
  public:
    Matrix(): d1(0), d2(0), data(new T[0]) { }
    Matrix(size_t d1_, size_t d2_): d1(d1_), d2(d2_) { data = new T[d1*d2]; }
    Matrix(const Matrix& other): d1(other.d1), d2(other.d2)
    { data = new T[d1*d2]; std::memcpy(data, other.data, d1*d2*sizeof(T)); }
    ~Matrix() { delete[] data; }
    void randomize();		// replace values by random numbers
    inline Matrix& operator=(const T& value);
    inline Matrix& operator=(const Matrix& other);
    inline Matrix operator*(const Matrix& other) const;
    inline void replace_by_product(const Matrix& a, const Matrix& b);
    inline Matrix operator-() const;
    Matrix& operator-=(const Matrix& other);
    inline Row<T> operator[](const size_t& i) const 
    { return Row<T>(*this, i); }
    size_t dim1() const { return d1; }
    size_t dim2() const { return d2; }
    Matrix transpose() const;

    // Spectral sort of matrix rows
    void spectral_sort();
    // Spectral sort of matrix rows & columns
    void spectral_bisort();

    // The following three functions assume that the matrix is in fact
    // a column vector (i.e., dim2() == 1).

    // Enforce zero sum
    void zerosum();
    // Enforce unit length
    void unit();
    // Compute norm
    T norm();
  };

  // This also assumes a column vector
  template<class T>
  inline bool
  distance_at_least(const Matrix<T>& a, const Matrix<T>& b, const T& epsilon)
  {
    assert(a.dim1() == b.dim1());
    assert(a.dim2() == 1 && b.dim2() == 1);
    T result = 0;
    T epsilon_sq = epsilon*epsilon; // avoid sqrt()
    for (size_t i = 0; i < a.dim1(); i++) {
      result += (a[i][0] - b[i][0]) * (a[i][0] - b[i][0]);
      if (result >= epsilon_sq)
	return true;
    }
    return false;
  }

  // This also assumes a column vector
  template<class T>
  inline T
  distance(const Matrix<T>& a, const Matrix<T>& b)
  {
    assert(a.dim1() == b.dim1());
    assert(a.dim2() == 1 && b.dim2() == 1);
    T result = 0;
    for (size_t i = 0; i < a.dim1(); i++) {
      T diff = a[i][0] - b[i][0];
      result += diff*diff;
    }
    return sqrt(result);
  }

  template<class T>
  T& Row<T>::operator[](const size_t& idx2) const
  {
#if 0
    std::cerr << "Matrix[" << idx1 << "][" << idx2 << "]" << std::endl
	      << idx2 + idx1*mat.d2 << std::endl
	      << mat.data << mat.data[idx1 + idx2*mat.d1] << std::endl;
#endif
    return mat.data[idx1 + idx2*mat.d1];
  }

  template<class T>
  Matrix<T>& Matrix<T>::operator=(const T& value)
  { 
    for (size_t i = 0; i < d1*d2; i++) 
      data[i] = value;
    return *this;
  }

  template<class T>
  Matrix<T>& 
  Matrix<T>::operator=(const Matrix<T>& other)
  {
    if (this == &other)
      return *this;
    d1 = other.d1; 
    d2 = other.d2; 
    delete[] data; 
    data = new T[d1*d2]; 
    std::memcpy(data, other.data, d1*d2*sizeof(T)); 
    return *this; 
  }

  template<class T>
  Matrix<T> Matrix<T>::operator*(const Matrix<T>& other) const
  {
    assert(dim2() == other.dim1());
    T* result = new T[dim1()*other.dim2()];
    multiply_into(other, result);
    return Matrix<T>(dim1(), other.dim2(), result);
  }

  template<class T>
  inline void 
  Matrix<T>::replace_by_product(const Matrix& a, const Matrix& b)
  {
    assert(dim1() == a.dim1());
    assert(dim2() == b.dim2());
    assert(a.dim2() == b.dim1());
    a.multiply_into(b, data);
  }


  template<class T>
  inline void
  Matrix<T>::multiply_into(const Matrix<T>& other, T* result) const
  {
    for(size_t i = 0; i < dim1(); i++)
      for(size_t j = 0; j < other.dim2(); j++) {
	T r = 0;
	for(size_t k = 0; k < dim2(); k++)
	  r += (*this)[i][k] * other[k][j];
	result[i + d1*j] = r;
      }
  }

#if 0 // wrong!
  template<>
  inline void
  Matrix<float>::multiply_into(const Matrix<float>& other, float* result) const
  {
    // SGEMM: C = alpha*A*B + beta*C
    cblas_sgemm(CblasColMajor, 
		transposed ? CblasTrans : CblasNoTrans, 
		other.transposed ? CblasTrans : CblasNoTrans,
		d1, other.d2, d2,
		1,		      // alpha
		data, d2,	      // A
		other.data, other.d2, // B
		0,		      // beta
		result, d2);
  }
#endif

  template<>
  inline void
  Matrix<double>::multiply_into(const Matrix<double>& other, double* result) const
  {
    // DGEMM: C = alpha*A*B + beta*C
    cblas_dgemm(CblasColMajor, CblasNoTrans, CblasNoTrans,
		dim1(), other.dim2(), dim2(),
		1,			  // alpha
		data, dim1(),  	          // A and its leading dimension
		other.data, other.dim1(), // B and its leading dimension
		0,			  // beta
		result, dim1());
    // CblasColMajor means that we store matrices in the Fortran order:
    //   M(0,0), M(1,0), ..., M(n-1,0), [possibly other things,] M(0,1), ...
    // CblasNoTrans means that we really mean it about the Fortran order
    //   (cheap way to transpose a matrix: invert this flag!)
    // Leading dimension means how much to add to an address to get
    // to the next column. Usually this is dim1, but can be set to
    // something else; I think this is useful to allow multiplying
    // submatrices without copying them to a temp array.
  }

  template<class T>
  inline std::vector<T> column(const Matrix<T>& A, long int i)
  {
    std::vector<T> result;
    for (long int j = 0; j < (long int)A.dim1(); j++)
      result.push_back(A[j][i]);
    return result;
  }

  template<class T>
  Matrix<T> Matrix<T>::operator-() const
  {
    Matrix result(dim1(), dim2());
    for (size_t i = 0; i < dim1(); i++)
      for (size_t j = 0; j < dim2(); j++)
	result[i][j] = -(*this)[i][j];
    return result;
  }

  template<class T>
  std::ostream& operator<< (std::ostream& s, const Matrix<T>& m)
  {
    s << m.dim1() << ' ' << m.dim2() << std::endl;
    for(size_t i = 0; i < m.dim1(); i++) {
      for(size_t j = 0; j < m.dim2(); j++)
	s << m[i][j] << ' ';
      s << std::endl;
    }
    return s;
  }

  template<class T>
  std::istream& operator>> (std::istream& s, Matrix<T>& m)
  {
    size_t d1, d2;
    s >> d1 >> d2;
    Matrix<T> result(d1,d2);
    for(size_t i = 0; i < d1; i++)
      for(size_t j = 0; j < d2; j++)
	s >> result[i][j];
    m = result;
    return s;
  }

  // Utility
  // Like sort, but return a permutation vector that can be used
  // to sort the input vector.
  template<class T>
  std::vector<size_t> order(const std::vector<T>& vec);
  template<class T>
  std::vector<size_t> order(const Matrices::Matrix<T>& vec);
};

#endif
