// Copyright 2004,2006 Jouni K. Seppnen          -*- coding: iso-8859-1 -*-
// Distributed under the Boost Software License, Version 1.0.
// See accompanying file LICENSE.

#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <functional>
#include <iostream>
#include <utility>
#include <vector>
#include <cstring>

#pragma implementation "matrices.h"
#include "matrices.h"

//#include <clapack.h>

namespace Matrices {
  template<class T>
  void
  Matrix<T>::randomize()
  {
    for (size_t i = 0; i < d1*d2; i++) {
      T num = (T)random()/RAND_MAX;
      if (random() % 2)
	num = -num;
      data[i] = num;
    }
  }

  template<class T>
  Matrix<T>& Matrix<T>::operator-=(const Matrix& other)
  {
    assert(dim1() == other.dim1());
    assert(dim2() == other.dim2());
    for (size_t i = 0; i < dim1(); i++)
      for (size_t j = 0; j < dim2(); j++)
	(*this)[i][j] -= other[i][j];
    return *this;
  }


  // order(vec):
  // Like sort, but return a permutation vector that can be used
  // to sort the input vector.
  template<class T>
  std::vector<size_t> order(const std::vector<T>& vec)
  {
    using namespace std;

    vector<pair<T,size_t> > t;
    for (size_t i = 0; i < vec.size(); i++)
      t.push_back(make_pair(vec[i], i));
    sort(t.begin(), t.end());
    
    vector<size_t> result;
    typename vector<pair<T,size_t> >::iterator i;
    for (i = t.begin(); i != t.end(); i++)
      result.push_back(i->second);
    
    return result;
  }

  template<class T>
  std::vector<size_t> order(const Matrices::Matrix<T>& vec)
  {
    assert(vec.dim2() == 1);
    using namespace std;

    vector<pair<T,size_t> > t;
    for (size_t i = 0; i < vec.size(); i++)
      t.push_back(make_pair(vec[i][0], i));
    sort(t.begin(), t.end());
    
    vector<size_t> result;
    typename vector<pair<T,size_t> >::iterator i;
    for (i = t.begin(); i != t.end(); i++)
      result.push_back(i->second);
    
    return result;
  }

  // perm_matrix() takes the output of order() and produces
  // a permutation matrix
  template<class T>
  Matrix<T> perm_matrix(const std::vector<size_t>& vec)
  {
    Matrix<T> result(vec.size(), vec.size());
    result = 0.0;
    for (size_t i = 0; i < vec.size(); i++)
      result[i][vec[i]] = 1.0;
    return result;
  }

  template<class T>
  T max(const Matrix<T>& A)
  {
    T result = 0.0;
    for (size_t i = 0; i < (size_t)A.dim1(); i++)
      for (size_t j = 0; j < (size_t)A.dim2(); j++)
	result = (result>fabs(A[i][j]))?result:fabs(A[i][j]);
    return result;
  }

  template<class T>
  T 
  Matrix<T>::norm()
  {
    assert(dim2()==1);
    T result = 0.0;
    for (size_t i = 0; i < dim1(); i++)
      result += (*this)[i][0] * (*this)[i][0];
    return sqrt(result);
  }

  template<class T>
  void
  Matrix<T>::zerosum()
  {
    assert(dim2()==1);

    T sum = 0.0;
    for (size_t i = 0; i < dim1(); i++)
      sum += (*this)[i][0];
    sum /= dim1();
    for (size_t i = 0; i < dim1(); i++)
      (*this)[i][0] -= sum;
  }

  template<class T>
  void
  Matrix<T>::unit()
  {
    assert(dim2() == 1);
    T length = norm();
    for (size_t i = 0; i < dim1(); i++)
      (*this)[i][0] /= length;
  }

  template<class T>
  Matrix<T> identity(size_t dim, const T& diag)
  // sets a diagonal matrix of size dim x dim,
  // with all elements in the diagonal equal to diag
  {
    Matrix<T> result(dim, dim);
    result = 0.0;
    for (size_t i=0; i<dim; i++)
      result[i][i] = diag;
    return result;
  }

  const double EPS = 1.0e-6;

    extern "C" void 
      dsyevr_(char*, char*, char*, long int*, double*, long int*, double*, double*,
	      long int*, long int*, double*, long int*, double*, double*, long int*,
	      long int*, double*, long int*, long int*, long int*, long int*);

#if 0
  std::vector<double> newFiedler(Matrix<double>& A)
  // THIS DESTROYS THE DATA OF MATRIX A
  {
    char jobz = 'V', range = 'I', uplo = 'U';
    long int N = A.dim1(), il = 2, iu = 2;
    long int M, lwork, liwork, info, isuppz[2];
    double abstol = 0, vl=1.0, vu=-1.0;
    double *W, *Z;
    long int *iwork = new long int[1];
    double *work = new double[1];

    assert(A.dim2() == (size_t)N);
    // For large N, these might not fit on the stack:
    W = new double(N); // W must have dimension N (!)
    Z = new double(N);

    lwork = liwork = -1;
  again:
#if 0
    std::cerr << "DSYEVR("
	      << (void*)&jobz << '(' << jobz << ')' << ", "
	      << (void*)&range << '(' << range << ')' << ", "
	      << (void*)&uplo << '(' << uplo << ')' << ", "
	      << &N << '(' << N << ')' << ", "
	      << A.data << ", "
	      << &N << '(' << N << ')' << ", "
	      << &vl << ',' << &vu << ", "
	      << &il << '(' << il << ')' << ", "
	      << &iu << '(' << iu << ')' << ", "
	      << &abstol << '(' << abstol << ')' << ", "
	      << &M << ", "
	      << W << ", "
	      << Z << ", "
	      << &N << '(' << N << ')' << ", "
	      << isuppz << ", "
	      << work << ", " 
	      << &lwork << '(' << lwork << ')' << ", "
	      << iwork << '(' << *iwork << "), "
	      << &liwork << '(' << liwork << ')' << ", "
	      << &info << ")" << std::endl;
    std::cerr << "work==" << work << ", iwork==" << iwork << std::endl;
#endif
    dsyevr_(&jobz,		// compute eigenvalues and vectors
	    &range,		// compute values #il to #iu
	    &uplo,		// either U or L (matters if we store only half of the symmetric matrix)
	    &N,			// order of input matrix
	    A.data, &N,		// the actual matrix & leading dimension
	    &vl, &vu, 		// not used (would be if range == 'V')
	    &il, &iu,		// find eigenvalues #2 to #2
	    &abstol,		// error tolerance; 0 means "default" (?)
	    &M,			// number of eigenvalues found (should be 1)
	    W,			// eigenvalues
	    Z, &N,		// eigenvectors + leading dimension
	    isuppz,		// support indices for eigenvector
	    work, &lwork,	// workspace
	    iwork, &liwork,	// integer workspace
	    &info);		// success/failure indicator
#if 0
    std::cerr << "work==" << work << ", iwork==" << iwork << std::endl;
    std::cerr << "M: " << M << ", "
	      << "isuppz: " << isuppz[0] << ',' << isuppz[1] << ", "
	      << "info: " << info << ", "
	      << "W: " << W[0] << ", "
	      << "Z: " << Z[0] << "..., "
	      << "work[0]: " << work[0] << ", "
	      << "iwork[0]: " << iwork[0] << ", "
	      << std::endl;
#endif
    if(info != 0) {
      std::cerr << "DSYEVR: " << info << std::endl;
      abort();
    }
    if (lwork == -1) {
      lwork = (long int)work[0];
      //std::cerr << "Setting LWORK to " << lwork << std::endl;
      delete[] work;
      work = new double[lwork];

      liwork = iwork[0];
      //std::cerr << "Setting LIWORK to " << liwork << std::endl;
      delete[] iwork;
      iwork = new long int[liwork];
      goto again;
    }
    assert(M==1);
    delete[] work; delete[] iwork; delete[] W; 
    std::vector<double> result;
    for (long int i = 0; i < N; i++)
      result.push_back(Z[i]);
    delete[] Z;
    return result;

  }
#endif

  template<class T>
  std::vector<T> fiedler(const Matrix<T>& A)
  {
    std::vector<T> result;
    Matrix<T> one(A.dim1(),1), two(A.dim1(),1);
    Matrix<T> *curr = &one, *old = &two;
    T largest = max(A);
    Matrix<T> B = identity(A.dim1(), largest);
    B -= A;
    //    *curr = 1.0;
    curr->randomize();
    curr->unit();
    std::cout << "Finding Fiedler eigenvector" << std::endl;
    int iter = 1;
    do {
      //std::cout << *curr << std::endl;
#if 1
      std::swap(old,curr);
      curr->replace_by_product(B, *old);
#else
      *old = *curr;
      *curr = B * (*curr);
#endif
      curr->zerosum();
      curr->unit();
      std::cout << '.';
      std::cout.flush();
      if (!(iter++%10))
	std::cout << distance(*curr,*old);
    } while (distance_at_least(*curr,*old,EPS));
    std::cout << "result:" << std::endl << *curr << std::endl;
    for (size_t i = 0; i < (size_t)A.dim1(); i++)
      result.push_back((*curr)[i][0]);
    return result;
  }

  template<class T>
  Matrix<T> Matrix<T>::transpose() const
  {
    Matrix<T> result(dim2(), dim1());
    for (size_t i = 0; i < dim1(); i++)
      for (size_t j = 0; j < dim2(); j++)
	result[j][i] = (*this)[i][j];
    return result;
  }


  template<class T>
  void Matrix<T>::spectral_sort()
  {
    using namespace std;
    // construct laplacian matrix
    Matrix<T> sim = (*this) * this->transpose();
    Matrix<T> laplacian = -sim;
    cout << "Got similarity" << endl;
    Matrix<T> ones(sim.dim1(), 1); // ones = [1,1,...,1]'
    ones = 1.0;
    Matrix<T> sum = sim*ones;
    for (size_t i = 0; i < laplacian.dim1(); i++)
      laplacian[i][i] += sum[i][0]; // laplacian(i,i) = sum(i)-sim(i,i)
    cout << "Got Laplacian" << endl;

    // pick second smallest eigenvalue and sort the matrix
    // according to the corresponding eigenvector
    vector<T> ev = fiedler(laplacian);
    vector<size_t> ord = order(ev);
    cout << "Permutation:" << endl;
    for (size_t i = 0; i < ord.size(); i++)
      cout << ord[i] << ' ';
    cout << endl;
    Matrix<T> permutation = perm_matrix<T>(ord);
    //cout << "Permuting by" << endl << permutation;
    Matrix<T> result = permutation * (*this);
    //cout << "Reordered matrix" << endl << result;

    *this = result;
  }

  template<class T>
  void Matrix<T>::spectral_bisort()
  {
    spectral_sort();
    (*this) = transpose();
    spectral_sort();
    (*this) = transpose();
  }

  template void Matrix<double>::spectral_sort();
  template void Matrix<double>::spectral_bisort();

}
