// Copyright 2004,2006 Jouni K. Seppnen          -*- coding: iso-8859-1 -*-
// Distributed under the Boost Software License, Version 1.0.
// See accompanying file LICENSE.

#define VERSION "1.0"

#include <cstdlib>
#include <ctime>
#include <unistd.h>

#include <iosfwd>
#include <fstream>

#include "matrices.h"
#include "model.h"
#include "search.h"

static unsigned restarts = 5;
static unsigned ntiles = 2;
static std::istream* in = &std::cin;
static unsigned long int seed;
static const char* gnuplot = 0;
static enum { NORMAL, TOP_DOWN, BOTTOM_UP, LEVEL } strategy = NORMAL;

void
process_cmdline(int argc, char* const argv[])
{
  using namespace std;
  char ch;
  while ((ch = getopt(argc, argv, "g:s:r:n:tbhl")) != -1) {
    switch(ch) {
    case 't':
      strategy = TOP_DOWN;
      break;
    case 'b':
      strategy = BOTTOM_UP;
      break;
    case 'l':
      strategy = LEVEL;
      break;
    case 'g':
      gnuplot = optarg;
      break;
    case 's':
      if (sscanf(optarg, "%lu", &seed) != 1) {
	cerr << "Bad value for -s: " << optarg << endl;
	exit(1);
      }
      break;
    case 'r':
      if (sscanf(optarg, "%u", &restarts) != 1) {
	cerr << "Bad value for -r: " << optarg << endl;
	exit(1);
      }
      break;
    case 'n':
      if (sscanf(optarg, "%u", &ntiles) != 1) {
	cerr << "Bad value for -n: " << optarg << endl;
	exit(1);
      }
      break;
    case 'h':
    case '?':
      cout << "Usage: tilings [-t|-b|-l] [-r restarts] [-n ntiles] [-g gnuplotfile] [file]" << endl;
      exit(1);
    }
  }
  argc -= optind;
  argv += optind;
  if (argc) {
    ifstream* file = new ifstream(argv[0]);
    if (!file->is_open()) {
      cerr << "Couldn't open file " << argv[0] << endl;
      exit(1);
    }
    in = file;
  }
}

int 
main (int argc, char * const argv[]) 
{
  using namespace std;
  using namespace Matrices;
  using namespace Tilings;

  cout << "tilings version " << VERSION 
       << ", Copyright (C) 2004,2006 Jouni K. Seppnen\n"
       << "This program comes with ABSOLUTELY NO WARRANTY.\n"
       << "This is free software, and you are welcome to redistribute it\n"
       << "under certain conditions. See the file LICENSE for details\n"
       << "on the lack of warranty and your right to redistribute.\n"
       << "\n";
  
  seed = time(0);
  process_cmdline(argc, argv);
  srandom(seed);
  cout << "Rng seed " << seed << endl;

  Matrix<double> A;
  cout << "Input matrix: on first line, #rows and #columns, "
       << "then every row on a separate line." << endl;
  (*in) >> A;
  cout << "Got matrix" << endl;

  Tilings::GreedySearch* s = 0;
  switch(strategy) {
  case NORMAL:
    s = new GreedySearch(A,restarts);
    break;
  case TOP_DOWN:
    s = new TopDownGreedySearch(A,restarts);
    break;
  case BOTTOM_UP:
    s = new BottomUpGreedySearch(A,restarts);
    break;
  case LEVEL:
    s = new LevelGreedySearch(A,restarts);
    break;
  }
  vector<double> llhs;
  llhs.push_back(s->mod().log_likelihood());
  for (unsigned int i = 0; i < ntiles; i++) {
    const Model& m = s->run();
    llhs.push_back(m.log_likelihood());
    if (i+1 == ntiles) {
      cout << m;
      if (gnuplot) {
	ofstream* g = new ofstream(gnuplot);
	m.gnuplot(g, gnuplot);
	g->close();
      }
    }
  }
  delete s;

  cout << "% log-likelihoods" << endl;
  for (unsigned int i = 0; i < llhs.size(); i++) {
    cout << llhs[i] << ' ';
  }
  cout << endl;

  return 0;
}

  
