/* popsim: a simple population simulator
 * Copyright (C) 2004-2006  Jouni K. Seppänen
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
*/

#include <assert.h>
#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "params.h"
#include "randfun.h"
#include "util.h"

int yyparse(void);
FILE *yyin;

uint MARKERS=0;
uint INITIAL_POPULATION=0;
uint FINAL_POPULATION=0;
uint GENERATIONS=0;
uint *POPULATION_PLAN=NULL;

uint GENLEN=22;
population_rule_t POPRULE = UNSET;
double POPULATION_GROWTH = 0.0;

typedef struct output_rule_t {
  uint full;
  uint samplesize;
  char *filename;
  struct output_rule_t *next;
} output_rule_t;
output_rule_t *OUTPUT_RULES = NULL;

static char *escape(const char *string);

void 
add_output_rule(int full, uint samplesize, char *filename)
{
  output_rule_t *new = xmalloc(sizeof(output_rule_t));
  output_rule_t *p;

  new->full = full;
  new->samplesize = samplesize;
  new->filename = xmalloc(strlen(filename)+1);
  new->next = NULL;
  strcpy(new->filename, filename);

  if (OUTPUT_RULES == NULL)
    OUTPUT_RULES = new;
  else {
    for (p = OUTPUT_RULES; p->next != NULL; p = p->next)
      ;
    p->next = new;
  }
}

void
do_output()
{
  output_rule_t *p;
  for (p = OUTPUT_RULES; p != NULL; p = p->next)
    output_histogram(p->full, p->samplesize, p->filename);
}

int 
read_conf(const char *fname)
{
  yyin = fopen(fname, "r");
  if (!yyin) {
    perror(fname);
    return 1;
  }
  if (yyparse()) {
    return 1;
  }
  if (fclose(yyin))
    perror(fname);
  return 0;
}  

void
write_conf(void)
{
  output_rule_t *p;

  fprintf(stderr, "# Settings\n");
  fprintf(stderr, "%u markers\n", MARKERS);
  switch(POPRULE) {
  case UNSET:
    fprintf(stderr,
	    "error: population sizes not set\n");
    break;
  case FROM_TO_NGEN:
    fprintf(stderr, 
	    "population from %u to %u in %u generations\n",
	    INITIAL_POPULATION, FINAL_POPULATION, GENERATIONS);
    break;
  case FROM_BY_YEAR_NGEN:
    fprintf(stderr, 
	    "a generation has %u years\n"
	    "population from %u by %f%% per year for %u generations\n",
	    GENLEN,
	    INITIAL_POPULATION, POPULATION_GROWTH, GENERATIONS);
    break;
  case FROM_BY_GEN_NGEN:
    fprintf(stderr,
	    "population from %u by %f%% per generation for %u generations\n",
	    INITIAL_POPULATION, POPULATION_GROWTH, GENERATIONS);
    break;
  }
  for (p = OUTPUT_RULES; p != NULL; p = p->next)
    if (p->full)
      fprintf(stderr, "output full histogram to '%s'\n", escape(p->filename));
    else
      fprintf(stderr, "output sampled (%u) histogram to '%s'\n", 
	      p->samplesize, escape(p->filename));
  if (OUTPUT_RULES == NULL)
    fprintf(stderr, "# warning: no output requested\n");
  randfun_write_config();
}

void
plan_population(void)
{
  double growth=-1.0, current;
  uint i;
  
  switch(POPRULE) {
  case UNSET:
    abort();
    break;

  case FROM_TO_NGEN:
    /* Input: INITIAL_POPULATION, FINAL_POPULATION, GENERATIONS */
    growth = pow((double)FINAL_POPULATION/INITIAL_POPULATION,
		 1.0/GENERATIONS);
    break;

  case FROM_BY_YEAR_NGEN:
    /* Input: INITIAL_POPULATION, POPULATION_GROWTH, GENERATIONS */
    growth = pow(1.0 + POPULATION_GROWTH/100.0, GENLEN);
    break;

  case FROM_BY_GEN_NGEN:
    /* Input: INITIAL_POPULATION, POPULATION_GROWTH, GENERATIONS */
    growth = 1.0 + POPULATION_GROWTH/100.0;
    break;
  }

  POPULATION_PLAN = xmalloc(GENERATIONS*sizeof(uint));
  current = INITIAL_POPULATION;
  for(i=0; i<GENERATIONS; i++) {
    current *= growth;
    POPULATION_PLAN[i] = (uint)rint(current);
  }
  if (!FINAL_POPULATION)
    FINAL_POPULATION = POPULATION_PLAN[GENERATIONS-1];
  assert(FINAL_POPULATION == POPULATION_PLAN[GENERATIONS-1]);

}

static char *
escape(const char *string)
{
  static char *result = NULL;
  uint i, j, len=0;

  for (i=0; string[i] != '\0'; i++)
    switch(string[i]) {
    case '\\':
    case '\'':
      len += 2;
      break;
    default:
      len += 1;
      break;
    }

  result = xrealloc(result, len+1);
  for (j=i=0; string[i] != '\0'; i++)
    switch(string[i]) {
    case '\\':
    case '\'':
      result[j++] = '\\';
      /* fall through */
    default:
      result[j++] = string[i];
      break;
    }
  assert(j == len);
  result[j] = '\0';
  
  return result;
}
